package exports

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/zootopia/analytics/drive/models"
)

func castToInt(v interface{}) int {
	switch t := v.(type) {
	case nil:
		return 0
	case int64:
		return int(t)
	case uint64:
		return int(t)
	case string:
		if t == "" {
			return 0
		}
		x, err := strconv.Atoi(t)
		if err != nil {
			panic(err)
		}
		return x
	default:
		return 0
	}
}

func castToInt64(v interface{}) int64 {
	switch t := v.(type) {
	case nil:
		return 0
	case int64:
		return t
	case uint64:
		return int64(t)
	case string:
		if t == "" {
			return 0
		}
		x, err := strconv.ParseInt(t, 10, 64)
		if err != nil {
			panic(err)
		}
		return x
	default:
		return 0
	}
}

func castToNInt64(v interface{}) models.NInt64 {
	return models.NInt64(castToInt64(v))
}

func castToUInt64(v interface{}) uint64 {
	switch t := v.(type) {
	case nil:
		return 0
	case uint64:
		return t
	case int64:
		return uint64(t)
	case string:
		if t == "" {
			return 0
		}
		x, err := strconv.ParseUint(t, 10, 64)
		if err != nil {
			panic(err)
		}
		return x
	default:
		return 0
	}
}

func castToNUInt64(v interface{}) models.NUInt64 {
	return models.NUInt64(castToUInt64(v))
}

func castToUUID(v interface{}) uuid.UUID {
	switch t := v.(type) {
	case nil:
		return uuid.Nil
	case string:
		return uuid.FromStringOrNil(t)
	case []byte:
		return uuid.FromBytesOrNil(t)
	default:
		return uuid.Nil
	}
}

func castToString(v interface{}) string {
	switch t := v.(type) {
	case nil:
		return ""
	case string:
		return t
	case []byte:
		return string(t)
	case int64:
		return fmt.Sprintf("%d", t)
	default:
		return ""
	}
}

func castToNString(v interface{}) models.NString {
	return models.NString(castToString(v))
}

func castToBool(v interface{}) bool {
	switch t := v.(type) {
	case nil:
		return false
	case string:
		s := strings.ToLower(t)
		return s == "t" || s == "true"
	case int:
		return t > 0
	case int64:
		return t > 0
	case bool:
		return t
	default:
		return false
	}
}

func castToNBool(v interface{}) models.NBool {
	return models.NBool(castToBool(v))
}

const timeLayout = "2006-01-02 15:04:05.999999999-07"

func castToTime(v interface{}) time.Time {
	switch t := v.(type) {
	case nil:
		return time.Time{}
	case int64:
		return time.Unix(t, 0)
	case string:
		if t == "" {
			return time.Time{}
		}
		x, err := time.Parse(timeLayout, t)
		if err != nil {
			panic(err)
		}
		return x
	case []byte:
		if len(t) == 0 {
			return time.Time{}
		}
		x, err := time.Parse(timeLayout, string(t))
		if err != nil {
			panic(err)
		}
		return x
	default:
		return time.Time{}
	}
}

func castToNTime(v interface{}) models.NTime {
	return models.NTime{Time: castToTime(v)}
}
