package exports

import (
	"fmt"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
)

func init() {
	updateCompiledBillsCmd := cobra.Command{
		Use: "update-compiled-bills",
		Run: gotasks.WrapMain(updateCompiledBillsMain),
	}
	updateCompiledBillsCmd.Flags().String("yt-proxy", "hahn", "")
	ExportsCmd.AddCommand(&updateCompiledBillsCmd)
}

func updateCompiledBillsMain(ctx *gotasks.Context) error {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[ctx.Config.BackendDB]
	if !ok {
		return fmt.Errorf("database %q not found", ctx.Config.BackendDB)
	}
	exporter := NewExporter(
		models.CompiledBill{}, "history_event_id",
		yc, ctx.Config.YTPaths.CompiledBillsTable,
		db, "compiled_bills",
		ctx.Config.YTPaths.BackendCompiledBillsDir,
		compiledBillsMapper{},
	)
	return exporter.Export(ctx)
}

type rawCompiledBill struct {
	HistoryEventID      interface{} `yson:"history_event_id"`
	HistoryAction       string      `yson:"history_action"`
	HistoryTimestamp    interface{} `yson:"history_timestamp"`
	HistoryUserID       string      `yson:"history_user_id"`
	HistoryOriginatorID interface{} `yson:"history_originator_id"`
	HistoryComment      interface{} `yson:"history_comment"`
	SessionID           string      `yson:"session_id"`
	UserID              interface{} `yson:"user_id"`
	Bill                interface{} `yson:"bill"`
	BillingType         string      `yson:"billing_type"`
	Details             interface{} `yson:"details"`
}

func (e rawCompiledBill) Parse() (event models.CompiledBill, err error) {
	event.HistoryEventID = castToInt64(e.HistoryEventID)
	event.HistoryAction = e.HistoryAction
	event.HistoryTimestamp = castToInt64(e.HistoryTimestamp)
	event.HistoryUserID = e.HistoryUserID
	event.HistoryOriginatorID = castToNString(e.HistoryOriginatorID)
	event.HistoryComment = castToNString(e.HistoryComment)
	event.SessionID = e.SessionID
	event.UserID = castToUUID(e.UserID)
	event.Bill = castToInt(e.Bill)
	event.BillingType = e.BillingType
	event.Details = castToNString(e.Details)
	return
}

type compiledBillsMapper struct{}

func (compiledBillsMapper) InputTypes() []interface{} {
	return []interface{}{rawCompiledBill{}}
}

func (compiledBillsMapper) OutputTypes() []interface{} {
	return []interface{}{models.CompiledBill{}}
}

func (compiledBillsMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var row rawCompiledBill
		if err := in.Scan(&row); err != nil {
			return err
		}
		event, err := row.Parse()
		if err != nil {
			return err
		}
		if err := out[0].Write(event); err != nil {
			return err
		}
	}
	return nil
}
