package exports

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"strings"

	"github.com/gofrs/uuid"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/db"
	"a.yandex-team.ru/zootopia/library/go/db/events"

	"a.yandex-team.ru/library/go/core/log"
)

func init() {
	chCompiledRidesCmd := cobra.Command{
		Use: "ch-compiled-rides",
		Run: gotasks.WrapMain(chCompiledRidesMain),
	}
	chCompiledRidesCmd.Flags().String("backend-db", "backend", "Name of database connection")
	chCompiledRidesCmd.Flags().String("target-db", "clickhouse", "Name of database connection")
	chCompiledRidesCmd.Flags().String("target-table", "compiled_rides", "Name of target table")
	chCompiledRidesCmd.PersistentFlags().Int("batch-size", 1000, "Maximal size of batch")
	chCompiledRidesCmd.PersistentFlags().Int("min-batch-size", 50, "Minimal size of batch")
	chCompiledRidesCmd.PersistentFlags().Int64("history-id", 0, "HistoryId where to start export data")
	ExportsCmd.AddCommand(&chCompiledRidesCmd)
	updateCompiledRidesCmd := cobra.Command{
		Use: "update-compiled-rides",
		Run: gotasks.WrapMain(updateCompiledRidesMain),
	}
	updateCompiledRidesCmd.Flags().String("yt-proxy", "hahn", "")
	ExportsCmd.AddCommand(&updateCompiledRidesCmd)
}

type chUploader struct {
	db     *gosql.DB
	table  string
	mapper func(*gotasks.Context, events.Event) (interface{}, error)
}

func (u *chUploader) buildInsertQuery(names []string) string {
	var builder strings.Builder
	builder.WriteString(fmt.Sprintf("INSERT INTO %q (", u.table))
	for i, name := range names {
		if i > 0 {
			builder.WriteString(", ")
		}
		builder.WriteString(fmt.Sprintf("%q", name))
	}
	builder.WriteString(") VALUES (")
	for i := range names {
		if i > 0 {
			builder.WriteString(", ")
		}
		builder.WriteRune('?')
	}
	return builder.String()
}

func (u *chUploader) UploadEvents(ctx *gotasks.Context, events []events.Event) error {
	if len(events) == 0 {
		return nil
	}
	row, err := u.mapper(ctx, events[0])
	if err != nil {
		return err
	}
	names := gosql.StructNames(row)
	if len(names) == 0 {
		return fmt.Errorf("empty column set")
	}
	return gosql.WithTx(u.db, func(tx *sql.Tx) error {
		stmt, err := tx.Prepare(u.buildInsertQuery(names))
		if err != nil {
			return err
		}
		for _, event := range events {
			row, err := u.mapper(ctx, event)
			if err != nil {
				return err
			}
			rowNames, values := gosql.StructNameValues(row, false)
			if len(names) != len(rowNames) {
				return fmt.Errorf("invalid column set")
			}
			for i := range names {
				if names[i] != rowNames[i] {
					return fmt.Errorf("invalid %d column: %q != %q", i, names[i], rowNames[i])
				}
			}
			if _, err := stmt.Exec(values...); err != nil {
				return fmt.Errorf("unable to exec: %w", err)
			}
		}
		return nil
	})
}

type chCompiledRide struct {
	SessionID                string  `db:"session_id"`
	UserID                   string  `db:"user_id"`
	ObjectID                 string  `db:"object_id"`
	OfferType                string  `db:"offer_type"`
	OfferName                string  `db:"offer_name"`
	OfferGroupName           string  `db:"offer_group_name"`
	ChildSeat                int8    `db:"child_seat"`
	RoofRack                 int8    `db:"roof_rack"`
	GPS                      int8    `db:"gps"`
	SnowChains               int8    `db:"snow_chains"`
	EntryToEcoZonesInGermany int8    `db:"entry_to_eco_zones_in_germany"`
	InsuranceType            string  `db:"insurance_type"`
	OfferTimestamp           int64   `db:"offer_timestamp"`
	OfferDeadline            int64   `db:"offer_deadline"`
	StandardParkingPrice     uint32  `db:"standard_parking_price"`
	StandardRidingPrice      uint32  `db:"standard_riding_price"`
	PackPrice                uint32  `db:"pack_price"`
	TotalPrice               int64   `db:"total"`
	Currency                 string  `db:"currency"`
	TotalPayment             uint64  `db:"total_payment"`
	Deposit                  uint64  `db:"deposit"`
	StartTimestamp           int64   `db:"start_timestamp"`
	FinishTimestamp          int64   `db:"finish_timestamp"`
	StartLatitude            float64 `db:"start_latitude"`
	StartLongitude           float64 `db:"start_longitude"`
	FinishLatitude           float64 `db:"finish_latitude"`
	FinishLongitude          float64 `db:"finish_longitude"`
	Mileage                  float64 `db:"mileage"`
	Finished                 int8    `db:"finished"`
	UselessDuration          uint32  `db:"useless_duration"`
	RidingTimestamp          int64   `db:"riding_timestamp"`
	AcceptanceDuration       uint32  `db:"acceptance_duration"`
	RidingDuration           uint32  `db:"riding_duration"`
	ParkingDuration          uint32  `db:"parking_duration"`
}

func chCompiledRidesMapper(ctx *gotasks.Context, event events.Event) (interface{}, error) {
	ride := event.(models.CompiledRide)
	rideData, err := ride.ParseData()
	if err != nil {
		return nil, err
	}
	ctx.Logger.Debug("map compiled ride", log.Any("data", rideData))
	var row chCompiledRide
	row.SessionID = ride.SessionID
	row.UserID = ride.HistoryUserID
	if ride.ObjectID.UUID != uuid.Nil {
		row.ObjectID = ride.ObjectID.UUID.String()
	}
	row.TotalPrice = ride.Price
	row.StartTimestamp = ride.Start
	row.FinishTimestamp = ride.Finish
	if offer := rideData.Offer; offer != nil {
		row.OfferType = offer.GetInstanceType()
		row.OfferName = offer.GetName()
		row.OfferGroupName = offer.GetGroupName()
		row.OfferTimestamp = int64(offer.GetTimestamp())
		row.OfferDeadline = int64(offer.GetDeadline())
		if standard := offer.StandartOffer; standard != nil {
			row.StandardParkingPrice = standard.GetPriceParking()
			row.StandardRidingPrice = standard.GetPriceRiding()
		}
		if pack := offer.PackOffer; pack != nil {
			row.PackPrice = pack.GetPackPrice()
		}
		if rental := offer.RentalOffer; rental != nil {
			row.Currency = rental.GetCurrency()
			row.TotalPayment = rental.GetTotalPayment()
			row.Deposit = rental.GetDeposit()
			row.InsuranceType = rental.GetInsuranceType()
			if rental.GetChildSeat() {
				row.ChildSeat = 1
			}
			if rental.GetRoofRack() {
				row.RoofRack = 1
			}
			if rental.GetGPS() {
				row.GPS = 1
			}
			if rental.GetSnowChains() {
				row.SnowChains = 1
			}
			if rental.GetEntryToEcoZonesInGermany() {
				row.EntryToEcoZonesInGermany = 1
			}
		}
	}
	if diff := rideData.TSnapshotsDiff; diff != nil {
		if diff.GetFinished() {
			row.Finished = 1
		}
		row.Mileage = float64(diff.GetMileage())
		if diff.Start != nil {
			row.StartLatitude = diff.Start.GetLatitude()
			row.StartLongitude = diff.Start.GetLongitude()
		}
		if diff.Last != nil {
			row.FinishLatitude = diff.Last.GetLatitude()
			row.FinishLongitude = diff.Last.GetLongitude()
		}
	}
	if meta := rideData.Meta; meta != nil {
		row.UselessDuration = uint32(meta.GetUselessDuration())
		row.RidingTimestamp = int64(meta.GetAcceptanceFinished())
		row.AcceptanceDuration = meta.GetAcceptanceDuration()
		row.RidingDuration = meta.GetRidingDuration()
		row.ParkingDuration = meta.GetParkingDuration()
	}
	return row, nil
}

func chCompiledRidesMain(ctx *gotasks.Context) error {
	backendDBName, err := ctx.Cmd.Flags().GetString("backend-db")
	if err != nil {
		return err
	}
	targetDBName, err := ctx.Cmd.Flags().GetString("target-db")
	if err != nil {
		return err
	}
	targetTable, err := ctx.Cmd.Flags().GetString("target-table")
	if err != nil {
		return err
	}
	batchSize, err := ctx.Cmd.Flags().GetInt("batch-size")
	if err != nil {
		return err
	}
	minBatchSize, err := ctx.Cmd.Flags().GetInt("min-batch-size")
	if err != nil {
		return err
	}
	historyID, err := ctx.Cmd.Flags().GetInt64("history-id")
	if err != nil {
		return err
	}

	tableState, err := GetTableState(ctx.States, "exports/compiled_rides")
	if err != nil {
		return fmt.Errorf("unable to fetch state: %w", err)
	}

	var state State
	if historyID != 0 {
		state.BeginEventID = historyID
		state.MaxEventTime = 0
		if err := tableState.SaveState(state); err != nil {
			return err
		}
	}
	backendDB, ok := ctx.DBs[backendDBName]
	if !ok {
		return fmt.Errorf("db %q does not exists", backendDBName)
	}
	targetDB, ok := ctx.DBs[targetDBName]
	if !ok {
		return fmt.Errorf("db %q does not exists", targetDBName)
	}
	return ExportDBEvents(
		ctx, tableState, backendDB,
		events.NewStore(models.CompiledRide{}, "history_event_id", "compiled_rides", db.Postgres),
		&chUploader{
			db:     targetDB,
			table:  targetTable,
			mapper: chCompiledRidesMapper,
		},
		batchSize, minBatchSize,
	)
}

func updateCompiledRidesMain(ctx *gotasks.Context) error {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[ctx.Config.BackendDB]
	if !ok {
		return fmt.Errorf("database %q not found", ctx.Config.BackendDB)
	}
	exporter := NewExporter(
		models.CompiledRide{}, "history_event_id",
		yc, ctx.Config.YTPaths.CompiledRidesTable,
		db, "compiled_rides",
		ctx.Config.YTPaths.BackendCompiledRidesDir,
		compiledRidesMapper{},
	)
	return exporter.Export(ctx)
}

type rawCompiledRide struct {
	HistoryEventID      interface{} `yson:"history_event_id"`
	HistoryAction       string      `yson:"history_action"`
	HistoryTimestamp    interface{} `yson:"history_timestamp"`
	HistoryUserID       string      `yson:"history_user_id"`
	HistoryOriginatorID interface{} `yson:"history_originator_id"`
	HistoryComment      interface{} `yson:"history_comment"`
	SessionID           string      `yson:"session_id"`
	ObjectID            interface{} `yson:"object_id"`
	Price               interface{} `yson:"price"`
	Duration            interface{} `yson:"duration"`
	Start               interface{} `yson:"start"`
	Finish              interface{} `yson:"finish"`
	Meta                interface{} `yson:"meta"`
	MetaProto           interface{} `yson:"meta_proto"`
	HardProto           interface{} `yson:"hard_proto"`
}

func (e rawCompiledRide) Parse() (event models.CompiledRide, err error) {
	event.HistoryEventID = castToInt64(e.HistoryEventID)
	event.HistoryAction = e.HistoryAction
	event.HistoryTimestamp = castToInt64(e.HistoryTimestamp)
	event.HistoryUserID = e.HistoryUserID
	event.HistoryOriginatorID = castToNString(e.HistoryOriginatorID)
	event.HistoryComment = castToNString(e.HistoryComment)
	event.SessionID = e.SessionID
	event.ObjectID = models.NUUID{UUID: castToUUID(e.ObjectID)}
	event.Price = castToInt64(e.Price)
	event.Duration = castToInt(e.Duration)
	event.Start = castToInt64(e.Start)
	event.Finish = castToInt64(e.Finish)
	metaBytes, err := json.Marshal(e.Meta)
	if err != nil {
		return
	}
	event.Meta = models.NString(metaBytes)
	event.MetaProto = castToNString(e.MetaProto)
	event.HardProto = castToNString(e.HardProto)
	return
}

type compiledRidesMapper struct{}

func (compiledRidesMapper) InputTypes() []interface{} {
	return []interface{}{rawCompiledRide{}}
}

func (compiledRidesMapper) OutputTypes() []interface{} {
	return []interface{}{models.CompiledRide{}}
}

func (compiledRidesMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var row rawCompiledRide
		if err := in.Scan(&row); err != nil {
			return err
		}
		event, err := row.Parse()
		if err != nil {
			return err
		}
		if err := out[0].Write(event); err != nil {
			return err
		}
	}
	return nil
}
