package exports

import (
	"database/sql"
	"fmt"
	"time"

	"github.com/jmoiron/sqlx"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/datasync"
	zmodels "a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/db/events"

	"a.yandex-team.ru/library/go/core/log"
)

var ExportsCmd = cobra.Command{
	Use: "exports",
}

func init() {
	gotasks.RootCmd.AddCommand(&ExportsCmd)
	exportsCmd := cobra.Command{
		Use:   "run-exports",
		Short: "Run all export processes",
		Run:   gotasks.WrapMain(main),
	}
	exportsCmd.Flags().String("yt-proxy", "hahn", "")
	exportsCmd.Flags().String("datasync-source", "analytics", "")
	exportsCmd.Flags().String("datasync-target", "datasync", "")
	gotasks.RootCmd.AddCommand(&exportsCmd)
	passportExportsCmd := cobra.Command{
		Use: "run-passport-exports",
		Run: gotasks.WrapMain(passportMain),
	}
	passportExportsCmd.Flags().String("yt-proxy", "hahn", "")
	passportExportsCmd.Flags().String("datasync-source", "analytics", "")
	passportExportsCmd.Flags().String("datasync-target", "datasync", "")
	gotasks.RootCmd.AddCommand(&passportExportsCmd)
	licenseExportsCmd := cobra.Command{
		Use: "run-license-exports",
		Run: gotasks.WrapMain(licenseMain),
	}
	licenseExportsCmd.Flags().String("yt-proxy", "hahn", "")
	licenseExportsCmd.Flags().String("datasync-source", "analytics", "")
	licenseExportsCmd.Flags().String("datasync-target", "datasync", "")
	gotasks.RootCmd.AddCommand(&licenseExportsCmd)
}

type TableState interface {
	ScanState(interface{}) error
	SaveState(interface{}) error
}

func GetTableState(store *models.StateStore, name string) (TableState, error) {
	state, err := store.GetOrCreateByName(name)
	if err != nil {
		return nil, err
	}
	return &tableState{store: store, state: state}, nil
}

type tableState struct {
	store *models.StateStore
	state models.State
}

func (s *tableState) ScanState(state interface{}) error {
	return s.state.ScanState(state)
}

func (s *tableState) SaveState(state interface{}) error {
	if err := s.state.SetState(state); err != nil {
		return fmt.Errorf("unable to set state: %w", err)
	}
	return s.store.Update(s.state)
}

type EventUploader interface {
	UploadEvents(*gotasks.Context, []events.Event) error
}

func ExportDBEvents(ctx *gotasks.Context, tableState TableState, db *gosql.DB, store events.ROStore, uploader EventUploader, batchSize, minBatchSize int) error {
	var state State
	if err := tableState.ScanState(&state); err != nil {
		return err
	}
	ctx.Logger.Debug("Start export DB events", log.Any("state", state))
	consumer := events.NewOrderedConsumer(store, state.BeginEventID)
	var batch []events.Event
	if err := consumer.ConsumeEvents(db, func(event events.Event) error {
		select {
		case <-ctx.Context.Done():
			return ctx.Context.Err()
		default:
		}
		if len(batch) >= batchSize {
			if err := uploader.UploadEvents(ctx, batch); err != nil {
				return err
			}
			// batch is not empty, so we can get last event from it.
			lastEvent := batch[len(batch)-1]
			// Update state with new BeginEventID and MaxEventTime.
			state.BeginEventID = lastEvent.EventID() + 1
			state.MaxEventTime = lastEvent.EventTime().Unix()
			ctx.Logger.Debug("Save state", log.Any("state", state))
			if err := tableState.SaveState(state); err != nil {
				return err
			}
			// Empty batch.
			batch = nil
		}
		// We should append event only after uploading batch because
		// consumer dont apply event if error is returned.
		batch = append(batch, event)
		return nil
	}); err != nil && err != sql.ErrNoRows {
		return err
	}
	if len(batch) >= minBatchSize {
		if err := uploader.UploadEvents(ctx, batch); err != nil {
			return err
		}
		// batch is not empty, so we can get last event from it.
		lastEvent := batch[len(batch)-1]
		// Update state with new BeginEventID and MaxEventTime.
		state.BeginEventID = lastEvent.EventID() + 1
		state.MaxEventTime = lastEvent.EventTime().Unix()
		ctx.Logger.Debug("Save state", log.Any("state", state))
		if err := tableState.SaveState(state); err != nil {
			return err
		}
		// Empty batch.
		batch = nil
	} else {
		ctx.Logger.Debug(
			"Batch size is too small",
			log.Int("batch_len", len(batch)),
			log.Int("min_batch_size", minBatchSize),
		)
	}
	return nil
}

func main(ctx *gotasks.Context) error {
	service, err := NewService(ctx)
	if err != nil {
		panic(err)
	}
	if err := service.Start(); err != nil {
		panic(err)
	}
	service.RunProcess(usersExportProcess)
	service.RunProcess(carsExportProcess)
	// We should stop service when this is required
	defer service.Stop()
	// Now sleep almost forever
	select {
	case <-ctx.Context.Done():
	case <-time.After(time.Hour):
	}
	return nil
}

func licenseMain(ctx *gotasks.Context) error {
	source, err := ctx.Cmd.Flags().GetString("datasync-source")
	if err != nil {
		return err
	}
	target, err := ctx.Cmd.Flags().GetString("datasync-target")
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[ctx.Config.BackendDB]
	if !ok {
		return fmt.Errorf("database %q not found", ctx.Config.BackendDB)
	}
	ytProxy, err := ctx.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return err
	}
	yc, ok := ctx.YTs[ytProxy]
	if !ok {
		return fmt.Errorf("invalid YT proxy %q", ytProxy)
	}
	dbx := sqlx.NewDb(db.DB, "pgx")
	users := zmodels.NewUserDataStore()
	userMgr := zmodels.NewHistoryManager(users, dbx)
	ticket, err := ctx.GetServiceTicket(source, target)
	if err != nil {
		return err
	}
	datasyncClient := datasync.NewClient(ctx.Config.Exports.Datasync, ticket)
	licenseMgr := zmodels.NewUserLicenseManager(
		users, yc, datasyncClient,
		ctx.Config.Exports.LicenseCacheTable,
		string(ctx.Config.Exports.DocumentsHashKey),
	)
	if err := userMgr.Init(); err != nil {
		return err
	}
	ctx.Logger.Info("Users initialized")
	if err := licenseMgr.Init(); err != nil {
		return err
	}
	ctx.Logger.Info("Licenses initialized")
	if err := userMgr.Sync(); err != nil {
		return err
	}
	ctx.Logger.Info("Users synced")
	if err := licenseMgr.Sync(); err != nil {
		return err
	}
	ctx.Logger.Info("Licenses synced")
	return licenseMgr.SaveCache()
}

func passportMain(ctx *gotasks.Context) error {
	source, err := ctx.Cmd.Flags().GetString("datasync-source")
	if err != nil {
		return err
	}
	target, err := ctx.Cmd.Flags().GetString("datasync-target")
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[ctx.Config.BackendDB]
	if !ok {
		return fmt.Errorf("database %q not found", ctx.Config.BackendDB)
	}
	ytProxy, err := ctx.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return err
	}
	yc, ok := ctx.YTs[ytProxy]
	if !ok {
		return fmt.Errorf("invalid YT proxy %q", ytProxy)
	}
	dbx := sqlx.NewDb(db.DB, "pgx")
	users := zmodels.NewUserDataStore()
	userMgr := zmodels.NewHistoryManager(users, dbx)
	ticket, err := ctx.GetServiceTicket(source, target)
	if err != nil {
		return err
	}
	datasyncClient := datasync.NewClient(ctx.Config.Exports.Datasync, ticket)
	passportMgr := zmodels.NewUserPassportManager(
		users, yc, datasyncClient,
		ctx.Config.Exports.PassportCacheTable,
		string(ctx.Config.Exports.DocumentsHashKey),
	)
	if err := userMgr.Init(); err != nil {
		return err
	}
	ctx.Logger.Info("Users initialized")
	if err := passportMgr.Init(); err != nil {
		return err
	}
	ctx.Logger.Info("Passports initialized")
	if err := userMgr.Sync(); err != nil {
		return err
	}
	ctx.Logger.Info("Users synced")
	if err := passportMgr.Sync(); err != nil {
		return err
	}
	ctx.Logger.Info("Passports synced")
	return passportMgr.SaveCache()
}
