package exports

import (
	"database/sql"
	"fmt"
	"log"
	"sync"
	"time"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	zsql "a.yandex-team.ru/zootopia/library/go/db/sql"
)

type Service struct {
	DB                        *sql.DB
	Config                    *config.Config
	YT                        yt.Client
	Cars                      *models.CarStore
	Users                     *models.UserDataStore
	usersMgr                  *models.HistoryManager
	CarTags                   *models.CarTagStore
	carTagsMgr                *models.HistoryManager
	CarDocuments              *models.LegacyCarDocumentStore
	carDocumentsMgr           *models.HistoryManager
	CarDocumentAssignments    *models.LegacyCarDocumentAssignmentStore
	carDocumentAssignmentsMgr *models.HistoryManager
	YangAssignments           *models.YangAssignmentStore
	yangAssignmentsMgr        *models.UpdateManager
	BillingAccounts           *models.BillingAccountStore
	BillingDescriptions       *models.BillingDescriptionStore
	BillingUserAccounts       *models.BillingUserAccountStore
	closer                    chan struct{}
	processCloser             chan struct{}
	waiter                    sync.WaitGroup
	processWaiter             sync.WaitGroup
}

type ServiceProcessFunc func(service *Service, closer <-chan struct{})

func NewService(ctx *gotasks.Context) (*Service, error) {
	ytProxy, err := ctx.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return nil, err
	}
	yc, ok := ctx.YTs[ytProxy]
	if !ok {
		return nil, fmt.Errorf("invalid YT proxy %q", ytProxy)
	}
	cfg := ctx.Config
	if cfg.Exports == nil {
		panic("Exports section should exists")
	}
	db, ok := ctx.DBs[cfg.BackendDB]
	if !ok {
		return nil, fmt.Errorf("database %q not found", cfg.BackendDB)
	}
	manager := Service{
		DB:                     db.DB,
		Config:                 cfg,
		YT:                     yc,
		Cars:                   models.NewCarStore(db.DB),
		Users:                  models.NewUserDataStore(),
		CarTags:                models.NewCarTagStore(),
		CarDocuments:           models.NewLegacyCarDocumentStore(),
		CarDocumentAssignments: models.NewLegacyCarDocumentAssignmentStore(),
		YangAssignments:        models.NewYangAssignmentStore(),
		BillingAccounts:        models.NewBillingAccountStore(),
		BillingDescriptions:    models.NewBillingDescriptionStore(),
		BillingUserAccounts:    models.NewBillingUserAccountStore(),
	}
	dbx := sqlx.NewDb(db.DB, "pgx")
	manager.usersMgr = models.NewHistoryManager(manager.Users, dbx)
	manager.carTagsMgr = models.NewHistoryManager(manager.CarTags, dbx)
	manager.carDocumentsMgr = models.NewHistoryManager(
		manager.CarDocuments, dbx,
	)
	manager.carDocumentAssignmentsMgr = models.NewHistoryManager(
		manager.CarDocumentAssignments, dbx,
	)
	manager.yangAssignmentsMgr = models.NewUpdateManager(
		manager.YangAssignments, dbx,
	)
	return &manager, nil
}

func (s *Service) Start() error {
	s.closer = make(chan struct{})
	s.processCloser = make(chan struct{})
	return s.startManagers()
}

func (s *Service) RunProcess(process ServiceProcessFunc) {
	s.processWaiter.Add(1)
	go func() {
		defer s.processWaiter.Done()
		process(s, s.processCloser)
	}()
}

type registerManagerFunc func(
	name string, mgr interface{}, delay time.Duration,
)

type Manager interface {
	InitTx(tx *sql.Tx) error
	SyncTx(tx *sql.Tx) error
}

func (s *Service) registerManagers(register registerManagerFunc) {
	register("users", s.usersMgr, 5*time.Second)
	// register("user_photos", s.userPhotosMgr, 5*time.Minute)
	// register("user_licenses", s.UserLicenses, 30*time.Minute)
	// register("user_passports", s.UserPassports, 30*time.Minute)
	register("car_tags", s.carTagsMgr, 5*time.Second)
	register("car_documents", s.carDocumentsMgr, 5*time.Second)
	register(
		"car_document_assignments",
		s.carDocumentAssignmentsMgr, 5*time.Second,
	)
	register("yang_assignments", s.yangAssignmentsMgr, 5*time.Minute)
	register("billing_accounts", s.BillingAccounts, 5*time.Second)
	register("billing_descriptions", s.BillingDescriptions, 5*time.Second)
	register("billing_user_accounts", s.BillingUserAccounts, 5*time.Second)
}

func (s *Service) startManagers() error {
	errs := make(chan error)
	count := 0
	registerFunc := func(name string, mgr interface{}, delay time.Duration) {
		count++
		s.waiter.Add(1)
		go s.managerLoop(name, mgr, errs, delay)
	}
	s.registerManagers(registerFunc)
	var err error
	for i := 0; i < count; i++ {
		if mgrErr := <-errs; mgrErr != nil {
			err = mgrErr
			log.Println("error:", err)
		}
	}
	if err != nil {
		s.Stop()
	}
	return err
}

func (s *Service) Stop() {
	close(s.processCloser)
	s.processWaiter.Wait()
	close(s.closer)
	s.waiter.Wait()
}

func (s *Service) managerLoop(
	name string, mgr interface{}, errs chan<- error, delay time.Duration,
) {
	defer s.waiter.Done()
	log.Println(name, "manager initializing...")
	var err error
	switch mgr := mgr.(type) {
	case Manager:
		err = zsql.WithTx(s.DB, mgr.InitTx)
	case models.Manager:
		err = mgr.Init()
	default:
		panic(fmt.Errorf("unsupported type: %T", mgr))
	}
	log.Println(name, "manager initialized")
	errs <- err
	if err != nil {
		log.Println(name, "manager init error:", err)
		return
	}
	ticker := time.NewTicker(delay)
	defer ticker.Stop()
	for {
		select {
		case <-s.closer:
			log.Println(name, "manager exiting...")
			return
		case <-ticker.C:
			log.Println(name, "manager syncing...")
			var err error
			switch mgr := mgr.(type) {
			case Manager:
				err = zsql.WithTx(s.DB, mgr.SyncTx)
			case models.Manager:
				err = mgr.Sync()
			default:
				panic(fmt.Errorf("unsupported type: %T", mgr))
			}
			if err != nil {
				log.Println(name, "manager error:", err)
			} else {
				log.Println(name, "manager synced")
			}
		}
	}
}
