package exports

import (
	"context"
	"encoding/json"
	"log"
	"sort"
	"strconv"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/analytics/gotasks/models/users"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
)

func toUnix(t time.Time) *int64 {
	if t.IsZero() {
		return nil
	}
	ts := t.Unix()
	return &ts
}

func usersExportProcess(service *Service, closer <-chan struct{}) {
	e := usersExport{service, closer}
	e.Run()
}

type usersExport struct {
	service *Service
	closer  <-chan struct{}
}

func (e *usersExport) Run() {
	ticker := time.NewTicker(30 * time.Minute)
	defer ticker.Stop()
	log.Println("Users export running...")
	if err := e.run(); err != nil {
		log.Println("Error:", err)
	}
	log.Println("Users export finished...")
	for {
		select {
		case <-e.closer:
			log.Println("Users export exiting...")
			return
		case <-ticker.C:
			log.Println("Users export running...")
			if err := e.run(); err != nil {
				log.Println("Error:", err)
			}
			log.Println("Users export finished...")
		}
	}
}

// func (e *usersExport) findPassportDuplicates(
// 	p map[uuid.UUID]uuid.UUID, ids []uuid.UUID,
// ) {
// 	passports := make(map[string]uuid.UUID)
// 	for _, id := range ids {
// 		user, ok := e.service.Users.Get(id)
// 		if !ok {
// 			continue
// 		}
// 		passport, ok := e.service.UserPassports.Get(id)
// 		if !ok || passport.NumberHash == "" {
// 			continue
// 		}
// 		if did, ok := passports[passport.NumberHash]; ok {
// 			if user.Status == "active" || user.Status == "passive" {
// 				p[getParentUUID(p, did)] = getParentUUID(p, id)
// 			} else {
// 				p[getParentUUID(p, id)] = getParentUUID(p, did)
// 			}
// 		} else {
// 			passports[passport.NumberHash] = id
// 		}
// 	}
// }

// func (e *usersExport) findLicenseDuplicates(
// 	p map[uuid.UUID]uuid.UUID, ids []uuid.UUID,
// ) {
// 	licenses := make(map[string]uuid.UUID)
// 	for _, id := range ids {
// 		user, ok := e.service.Users.Get(id)
// 		if !ok {
// 			continue
// 		}
// 		license, ok := e.service.UserLicenses.Get(id)
// 		if !ok || license.Number == "" {
// 			continue
// 		}
// 		if did, ok := licenses[license.Number]; ok {
// 			if user.Status == "active" || user.Status == "passive" {
// 				p[getParentUUID(p, did)] = getParentUUID(p, id)
// 			} else {
// 				p[getParentUUID(p, id)] = getParentUUID(p, did)
// 			}
// 		} else {
// 			licenses[license.Number] = id
// 		}
// 	}
// }

type uuidSorter []uuid.UUID

func (s uuidSorter) Len() int {
	return len(s)
}

func (s uuidSorter) Less(i, j int) bool {
	return s[i].String() < s[j].String()
}

func (s uuidSorter) Swap(i, j int) {
	s[i], s[j] = s[j], s[i]
}

func (e *usersExport) run() error {
	// Setup Yt client
	yc := e.service.YT
	// Create table if does not exists
	tableSchema, err := schema.Infer(users.User{})
	if err != nil {
		return err
	}
	tablePath := e.service.Config.YTPaths.SimpleUsersTable.Rich().
		SetSchema(tableSchema)
	if _, err := yc.CreateNode(
		context.TODO(), tablePath, yt.NodeTable,
		&yt.CreateNodeOptions{IgnoreExisting: true},
	); err != nil {
		return err
	}
	// Prefetch user IDs
	// This is slow and blocking operation, so we do it very accurately
	// consuming more memory than we can
	ids := e.service.Users.GetIDs()
	// Sort IDs to make our table more static
	sort.Sort(uuidSorter(ids))
	// Calculate parent ID
	parents := make(map[uuid.UUID]uuid.UUID)
	// e.findPassportDuplicates(parents, ids)
	// e.findLicenseDuplicates(parents, ids)
	// Create writer to YT
	writer, err := yc.WriteTable(context.TODO(), tablePath, nil)
	if err != nil {
		return err
	}
	// Iterate over IDs
	for _, id := range ids {
		// Prepare full information about user by his ID
		user, ok := e.getExportedUser(id)
		if p := getParentUUID(parents, user.UserID); p != user.UserID {
			pid := p.String()
			user.ParentID = &pid
		}
		if ok {
			if err := writer.Write(user); err != nil {
				return err
			}
		}
	}
	// Apply updates
	return writer.Commit()
}

func (e *usersExport) getExportedUser(id uuid.UUID) (users.User, bool) {
	baseUser, ok := e.service.Users.Get(id)
	if !ok {
		return users.User{}, ok
	}
	docsKey := string(e.service.Config.Exports.DocumentsHashKey)
	uid, err := strconv.ParseInt(string(baseUser.UID), 10, 64)
	if err != nil {
		uid = 0
	}
	user := users.User{
		UserID:        baseUser.ID,
		Status:        baseUser.Status,
		UID:           uid,
		Username:      baseUser.Username,
		Email:         baseUser.Email,
		Phone:         baseUser.Phone,
		PhoneVerified: baseUser.PhoneVerified,
		RegisterDBGeo: baseUser.RegisterGeo,
		DBFirstName:   models.HashWithHMAC(baseUser.FirstName, docsKey),
		DBLastName:    models.HashWithHMAC(baseUser.LastName, docsKey),
		DBMiddleName:  models.HashWithHMAC(baseUser.MiddleName, docsKey),
	}
	if t := baseUser.RegisterTime.Unix(); t > 0 {
		user.RegisterTime = &t
	}
	e.buildYangFields(&user)
	e.buildWalletFields(&user)
	return user, true
}

func (e *usersExport) buildWalletFields(user *users.User) {
	accounts, err := e.service.BillingUserAccounts.FindByUser(user.UserID)
	if err != nil {
		log.Println("Error:", err)
	}
	for _, joint := range accounts {
		account, err := e.service.BillingAccounts.Get(joint.AccountID)
		if err != nil {
			log.Println("Error:", err)
			continue
		}
		description, err := e.service.BillingDescriptions.Get(account.TypeID)
		if err != nil {
			log.Println("Error:", err)
			continue
		}
		user.Wallets = append(user.Wallets, users.Wallet{
			ID:   account.ID,
			Name: description.Name,
		})
		if description.Type == "bonus" {
			var meta struct {
				Balance int64 `json:"balance"`
			}
			if err := json.Unmarshal([]byte(account.Meta), &meta); err != nil {
				log.Println("Error:", err)
				continue
			}
			user.BonusBalance += float64(meta.Balance) / 100
			user.BonusTotal += float64(account.Spent) / 100
		}
	}
}

func fixGender(gender string) string {
	switch gender {
	case "FEMALE", "F", "ЖЕН", "Ж":
		return "f"
	case "MALE", "M", "МУЖ", "М":
		return "m"
	}
	return gender
}

func (e *usersExport) buildYangFields(user *users.User) {
	assignments := e.service.YangAssignments.GetByUser(user.UserID)
	var lastTime time.Time
	for _, assignment := range assignments {
		if assignment.ID == uuid.Nil {
			continue
		}
		if assignment.CreateTime.After(lastTime) {
			sid := assignment.ID.String()
			user.SecretID = &sid
			lastTime = assignment.CreateTime
		}
	}
}

func getParentUUID(p map[uuid.UUID]uuid.UUID, id uuid.UUID) uuid.UUID {
	pid, ok := p[id]
	if !ok {
		return id
	}
	if id != pid {
		p[id] = getParentUUID(p, pid)
	}
	return p[id]
}
