package fuelings

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/opti24"
)

func init() {
	RegisterProvider(&gazpromProvider{})
}

type GazpromConfig struct {
	Endpoint string        `json:"endpoint"`
	Login    string        `json:"login"`
	Password secret.Secret `json:"password"`
	APIKey   secret.Secret `json:"api_key"`
	Contract secret.Secret `json:"contract"`
}

type gazpromProvider struct {
	config GazpromConfig
}

func (gazpromProvider) New(cfg models.JSON) (Provider, error) {
	var config GazpromConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &gazpromProvider{config: config}, nil
}

func (gazpromProvider) Name() string {
	return "gazprom"
}

func (gazpromProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	ops, err := opti24.ParseRawReport(data)
	if err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, op := range ops {
		result = append(result, OperationsTableRow{
			Provider:    "gazprom",
			OperationID: op.ID,
			Card:        op.Card,
			Time:        op.Time,
			Total:       op.Total,
			Amount:      op.Amount,
			Price:       op.Price,
			Fuel:        getFixedFuel(op.FuelName),
			Address:     op.Address,
			Region:      getFixedRegion(op.Region),
		})
	}
	return result, nil
}

func (p *gazpromProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client := opti24.NewClient(opti24.Config{
		Login:    p.config.Login,
		Password: p.config.Password.Secret(),
		APIKey:   p.config.APIKey.Secret(),
		Endpoint: p.config.Endpoint,
	})
	if err := client.Login(); err != nil {
		return nil, err
	}
	defer func() {
		if err := client.Logout(); err != nil {
			ctx.Logger.Error("Unable to logout", log.Error(err))
		}
	}()
	id, err := getOrCreateGazpromReport(client, p.config.Contract.Secret(), from, to)
	if err != nil {
		return nil, err
	}
	select {
	case <-ctx.Context.Done():
		return nil, ctx.Context.Err()
	case <-time.After(15 * time.Second):
	}
	return client.GetRawReport(id)
}

func getOrCreateGazpromReport(
	client *opti24.Client, contract string, begin, end time.Time,
) (string, error) {
	id, err := client.RequestReport(contract, begin, end)
	if err != nil {
		return "", err
	}
	ticker := time.NewTicker(time.Minute)
	timeout := time.After(30 * time.Minute)
	for {
		select {
		case <-timeout:
			return "", fmt.Errorf("failed to wait for report")
		case <-ticker.C:
			jobs, err := client.GetReportJobs()
			if err != nil {
				continue
			}
			for _, job := range jobs {
				if job.ID == id {
					return id, nil
				}
			}
		}
	}
}
