package fuelings

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/clients/lkmtzs"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/library/go/core/log"
)

func init() {
	RegisterProvider(&lkmtzsProvider{})
}

type LkmtzsConfig struct {
	Endpoint string        `json:"endpoint"`
	Login    secret.Secret `json:"login"`
	Password secret.Secret `json:"password"`
}

type lkmtzsProvider struct {
	config LkmtzsConfig
}

func (lkmtzsProvider) New(cfg models.JSON) (Provider, error) {
	var config LkmtzsConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &lkmtzsProvider{config: config}, nil
}

func (lkmtzsProvider) Name() string {
	return "lkmtzs"
}

func (lkmtzsProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	var txs []lkmtzs.Tx
	if err := json.Unmarshal(data, &txs); err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, tx := range txs {
		result = append(result, OperationsTableRow{
			Provider:    LkmtzsReport,
			OperationID: fmt.Sprintf("%d-%s", tx.Time.Unix(), tx.Card),
			Card:        tx.Card,
			Time:        tx.Time.Unix(),
			Amount:      -tx.Amount,
			Fuel:        getFixedFuel(tx.FuelCode),
		})
	}
	return result, nil
}

func (p *lkmtzsProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client := lkmtzs.NewClient(p.config.Endpoint)
	session, err := client.Login(p.config.Login.Secret(), p.config.Password.Secret())
	if err != nil {
		return nil, fmt.Errorf("unable to login: %w", err)
	}
	defer func() {
		if err := client.Logout(session); err != nil {
			ctx.Logger.Error("Unable to logout", log.Error(err))
		}
	}()
	txs, err := client.RequestTxs2(session, from, to)
	if err != nil {
		return nil, fmt.Errorf("unable to fetch transactions: %w", err)
	}
	return json.Marshal(txs)
}
