package fuelings

import (
	"encoding/json"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/licard"
)

func init() {
	RegisterProvider(&lukoilProvider{})
}

type LukoilConfig struct {
	Endpoint string        `json:"endpoint"`
	CrtData  secret.Secret `json:"crt_data"`
	KeyData  secret.Secret `json:"key_data"`
	Contract secret.Secret `json:"contract"`
}

type lukoilProvider struct {
	config LukoilConfig
}

func (lukoilProvider) New(cfg models.JSON) (Provider, error) {
	var config LukoilConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &lukoilProvider{config: config}, nil
}

func (lukoilProvider) Name() string {
	return "lukoil"
}

func (lukoilProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	ops, err := licard.ParseRawContractOperations(data)
	if err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, op := range ops {
		result = append(result, OperationsTableRow{
			Provider:    "lukoil",
			OperationID: fmt.Sprintf("%d", op.ID),
			Card:        op.Card,
			Time:        op.Time,
			Total:       op.Total,
			Amount:      op.Amount,
			Price:       op.Price,
			Fuel:        getFixedFuel(op.FuelCode),
			Address:     op.Address,
			Region:      getFixedRegion(op.Region),
		})
	}
	return result, nil
}

func (p *lukoilProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client, err := licard.NewClient(licard.Config{
		Endpoint: p.config.Endpoint,
		CrtData:  []byte(p.config.CrtData.Secret()),
		KeyData:  []byte(p.config.KeyData.Secret()),
	})
	if err != nil {
		return nil, err
	}
	contractID, err := strconv.ParseInt(p.config.Contract.Secret(), 10, 64)
	if err != nil {
		return nil, err
	}
	return client.GetRawContractOperations(contractID, from, to)
}
