package fuelings

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/clients/lkmtzs"
	"a.yandex-team.ru/drive/library/go/clients/tatneft"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/licard"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/opti24"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/rncard"
)

func init() {
	buildOperationsCmd := cobra.Command{
		Use: "build-operations",
		Run: gotasks.WrapMain(buildOperationsMain),
	}
	buildOperationsCmd.PersistentFlags().String(
		"yt-proxy", "hahn", "YT proxy",
	)
	FuelingsCmd.AddCommand(&buildOperationsCmd)
	mapreduce.Register(ReportsMapper{})
	mapreduce.Register(ReportsReducer{})
}

const (
	Fuel92  = "g92"
	Fuel95  = "g95"
	FuelDT  = "gDT"
	Fuel100 = "g100"
)

const (
	GazpromReport  = "gazprom"
	RosneftReport  = "rosneft"
	LukoilReport   = "lukoil"
	TatneftReport  = "tatneft"
	LkmtzsReport   = "lkmtzs"
	ZapravkiReport = "zapravki"
	ProliveReport  = "prolive"
)

const (
	RegionMSK = "MSK"
	RegionSPB = "SPB"
	RegionKZN = "KZN"
	RegionSCH = "SCH"
)

type OperationsTableRow struct {
	// Time contains amount of seconds from unix timestamp
	Time int64 `yson:"time"`
	// Provider contains operation provider
	Provider string `yson:"provider"`
	// ProviderID contains unique identifier
	OperationID string `yson:"operation_id"`
	// Card contains card number
	Card string `yson:"card"`
	// Total contains total cost (with all discounts and etc.)
	Total float64 `yson:"total"`
	// Amount contains amount of fuel
	Amount float64 `yson:"amount"`
	// Price contains price of fuel
	Price float64 `yson:"price"`
	// Fuel contains type of fuel
	Fuel string `yson:"fuel"`
	// Address contains address of fueling station
	Address string `yson:"address"`
	// Region contains region location of fueling station
	Region string `yson:"region"`
}

type ReportsMapper struct{}

func (ReportsMapper) InputTypes() []interface{} {
	return []interface{}{&ReportsTableRow{}}
}

func (ReportsMapper) OutputTypes() []interface{} {
	return []interface{}{&OperationsTableRow{}}
}

type tatneftTx struct {
	Tx   tatneft.Tx
	Item tatneft.TxItem
}

func (ReportsMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		if in.TableIndex() == 1 {
			var row zapravkiTx
			if err := in.Scan(&row); err != nil {
				return err
			}
			out[0].MustWrite(OperationsTableRow{
				Provider:    ZapravkiReport,
				OperationID: row.ID,
				Card:        row.CarID,
				Time:        row.Time,
				Total:       row.Total,
				Amount:      row.Amount,
				Price:       row.Price,
				Fuel:        getFixedFuel(row.Fuel),
			})
			continue
		}
		var row ReportsTableRow
		if err := in.Scan(&row); err != nil {
			return err
		}
		switch row.Provider {
		case GazpromReport:
			operations, err := opti24.ParseRawReport(row.Data)
			if err != nil {
				return err
			}
			for _, op := range operations {
				out[0].MustWrite(OperationsTableRow{
					Provider:    GazpromReport,
					OperationID: op.ID,
					Card:        op.Card,
					Time:        op.Time,
					Total:       op.Total,
					Amount:      op.Amount,
					Price:       op.Price,
					Fuel:        getFixedFuel(op.FuelName),
					Address:     op.Address,
					Region:      getFixedRegion(op.Region),
				})
			}
		case RosneftReport:
			operations, err := rncard.ParseRawOperationsByContract(row.Data)
			if err != nil {
				return err
			}
			for _, op := range operations {
				out[0].MustWrite(OperationsTableRow{
					Provider:    RosneftReport,
					OperationID: op.ID,
					Card:        op.Card,
					Time:        op.Time,
					Total:       op.Total,
					Amount:      op.Amount,
					Price:       op.Price,
					Fuel:        getFixedFuel(op.FuelCode),
					Address:     op.Address,
					Region:      getFixedRegion(op.Region),
				})
			}
		case LukoilReport:
			operations, err := licard.ParseRawContractOperations(row.Data)
			if err != nil {
				return err
			}
			for _, op := range operations {
				out[0].MustWrite(OperationsTableRow{
					Provider:    LukoilReport,
					OperationID: fmt.Sprintf("%d", op.ID),
					Card:        op.Card,
					Time:        op.Time,
					Total:       op.Total,
					Amount:      op.Amount,
					Price:       op.Price,
					Fuel:        getFixedFuel(op.FuelCode),
					Address:     op.Address,
					Region:      getFixedRegion(op.Region),
				})
			}
		case TatneftReport:
			var txs []tatneftTx
			if err := json.Unmarshal(row.Data, &txs); err != nil {
				return err
			}
			for _, tx := range txs {
				// Return should have negative total and amount.
				if tx.Tx.Category == "J" || tx.Tx.Category == "R" {
					tx.Item.Total = -tx.Item.Total
					tx.Item.Amount = -tx.Item.Amount
				}
				out[0].MustWrite(OperationsTableRow{
					Provider:    TatneftReport,
					OperationID: fmt.Sprintf("%d", tx.Item.ID),
					Card:        tx.Tx.Card,
					Time:        tx.Tx.Time.Unix(),
					Total:       tx.Item.Total,
					Amount:      tx.Item.Amount,
					Price:       tx.Item.Price,
					Fuel:        getFixedFuel(tx.Item.FuelName),
				})
			}
		case LkmtzsReport:
			txs, err := lkmtzs.ParseRawTxs(row.Data)
			if err != nil {
				return err
			}
			for _, tx := range txs {
				out[0].MustWrite(OperationsTableRow{
					Provider:    LkmtzsReport,
					OperationID: fmt.Sprintf("%d-%s", tx.Time.Unix(), tx.Card),
					Card:        tx.Card,
					Time:        tx.Time.Unix(),
					Amount:      -tx.Amount,
					Fuel:        getFixedFuel(tx.FuelCode),
				})
			}
		case ProliveReport:
			var txs []lkmtzs.Tx
			if err := json.Unmarshal(row.Data, &txs); err != nil {
				return err
			}
			for _, tx := range txs {
				out[0].MustWrite(OperationsTableRow{
					Provider:    ProliveReport,
					OperationID: fmt.Sprintf("%d-%s", tx.Time.Unix(), tx.Card),
					Card:        tx.Card,
					Time:        tx.Time.Unix(),
					Amount:      -tx.Amount,
					Fuel:        getFixedFuel(tx.FuelCode),
				})
			}
		default:
			if !strings.HasPrefix(row.Provider, "v2/") {
				return fmt.Errorf("unknown provider %q", row.Provider)
			}
			impl, ok := providers[row.Provider[3:]]
			if !ok {
				return fmt.Errorf("unknown provider %q", row.Provider)
			}
			txs, err := impl.ParseReport(row.Data)
			if err != nil {
				return err
			}
			for _, tx := range txs {
				out[0].MustWrite(tx)
			}
		}
	}
	return nil
}

func getFixedFuel(fuel string) string {
	switch fuel {
	case "g92", "Аи-92", "АИ-92", "G-92", "FUEL92", "АИ92":
		return Fuel92
	case "g95", "Аи-95", "АИ-95", "G-95", "FUEL95", "АИ95":
		return Fuel95
	case "gDT", "ДТ", "ДТ ОПТИ", "G-ДТ", "FUELDEISEL":
		return FuelDT
	case "G-Drive 100":
		return Fuel100
	default:
		return fuel
	}
}

func getFixedRegion(region string) string {
	switch region {
	case "Москва", "г.Москва", "Московская область":
		return RegionMSK
	case "Санкт-Петербург", "г.Санкт-Петербург", "г. Санкт-Петербург", "Ленинградская область":
		return RegionSPB
	case "Республика Татарстан":
		return RegionKZN
	case "Краснодарский край":
		return RegionSCH
	default:
		return region
	}
}

type ReportsReducer struct{}

func (ReportsReducer) InputTypes() []interface{} {
	return []interface{}{&OperationsTableRow{}}
}

func (ReportsReducer) OutputTypes() []interface{} {
	return []interface{}{&OperationsTableRow{}}
}

func (r ReportsReducer) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out[0])
	})
}

func (ReportsReducer) reduceGroup(
	in mapreduce.Reader, out mapreduce.Writer,
) error {
	var row OperationsTableRow
	if in.Next() {
		if err := in.Scan(&row); err != nil {
			return err
		}
		out.MustWrite(row)
	}
	for in.Next() {
	}
	return nil
}

func buildOperationsMain(ctx *gotasks.Context) error {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	if ctx.Config.Fuelings == nil {
		return fmt.Errorf("fueligns is not configured")
	}
	tablePath := ctx.Config.Fuelings.OperationsTable.Rich()
	tableSchema := schema.MustInfer(OperationsTableRow{})
	opSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{
			ctx.Config.Fuelings.ReportsTable,
			ypath.Path("//home/carsharing/production/data/fuelings/zapravki"),
		},
		OutputTablePaths: []ypath.YPath{
			tablePath.SetSchema(tableSchema),
		},
		SortBy:   []string{"provider", "operation_id"},
		ReduceBy: []string{"provider", "operation_id"},
		Pool:     "carsharing",
	}
	ctx.Signal("fuelings.build_operations.start_sum", nil).Add(1)
	startTime := time.Now()
	defer func() {
		ctx.Signal("fuelings.build_operations.duration_last", nil).
			Set(time.Since(startTime).Seconds())
	}()
	tx, err := yc.BeginTx(context.TODO(), nil)
	if err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	mr := mapreduce.New(yc).WithTx(tx)
	op, err := mr.MapReduce(
		&ReportsMapper{},
		&ReportsReducer{},
		opSpec.MapReduce(),
	)
	if err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	if err := op.Wait(); err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	sortSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{tablePath},
		OutputTablePath: tablePath,
		SortBy:          []string{"time"},
		Pool:            "carsharing",
	}
	op, err = mr.Sort(sortSpec.Sort())
	if err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	if err := op.Wait(); err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	if err := tx.Commit(); err != nil {
		ctx.Signal("fuelings.build_operations.error_sum", nil).Add(1)
		return err
	}
	ctx.Signal("fuelings.build_operations.success_sum", nil).Add(1)
	return nil
}
