package fuelings

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/clients/lkmtzs"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/library/go/core/log"
)

func init() {
	RegisterProvider(&proliveProvider{})
}

type ProliveConfig struct {
	Endpoint string        `json:"endpoint"`
	Login    secret.Secret `json:"login"`
	Password secret.Secret `json:"password"`
}

type proliveProvider struct {
	config ProliveConfig
}

func (proliveProvider) New(cfg models.JSON) (Provider, error) {
	var config ProliveConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &proliveProvider{config: config}, nil
}

func (proliveProvider) Name() string {
	return "prolive"
}

func (proliveProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	var txs []lkmtzs.Tx
	if err := json.Unmarshal(data, &txs); err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, tx := range txs {
		result = append(result, OperationsTableRow{
			Provider:    ProliveReport,
			OperationID: fmt.Sprintf("%d-%s", tx.Time.Unix(), tx.Card),
			Card:        tx.Card,
			Time:        tx.Time.Unix(),
			Amount:      -tx.Amount,
			Fuel:        getFixedFuel(tx.FuelCode),
		})
	}
	return result, nil
}

func (p *proliveProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client := lkmtzs.NewClient(p.config.Endpoint)
	session, err := client.Login(p.config.Login.Secret(), p.config.Password.Secret())
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := client.Logout(session); err != nil {
			ctx.Logger.Error("Unable to logout", log.Error(err))
		}
	}()
	txs, err := client.RequestTxs2(session, from, to)
	if err != nil {
		return nil, err
	}
	return json.Marshal(txs)
}
