package fuelings

import (
	"context"
	"fmt"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/yt"
)

func init() {
	fetchReportCmd := cobra.Command{
		Use: "fetch-report",
		Run: gotasks.WrapMain(fetchReportMain),
	}
	fetchReportCmd.PersistentFlags().String("yt-proxy", "hahn", "YT proxy")
	fetchReportCmd.PersistentFlags().String(
		"provider", "", "Name of fuelings tx provider",
	)
	fetchReportCmd.PersistentFlags().String("from-date", "", "")
	fetchReportCmd.PersistentFlags().String("to-date", "", "")
	FuelingsCmd.AddCommand(&fetchReportCmd)
}

type ReportsTableRow struct {
	Timestamp int64
	Provider  string
	Data      []byte
}

func fetchReportMain(ctx *gotasks.Context) error {
	provider, err := ctx.Cmd.Flags().GetString("provider")
	if err != nil {
		return err
	}
	fromDate, err := ctx.Cmd.Flags().GetString("from-date")
	if err != nil {
		return err
	}
	toDate, err := ctx.Cmd.Flags().GetString("to-date")
	if err != nil {
		return err
	}
	toTime := time.Now().UTC()
	fromTime := toTime.AddDate(0, 0, -3)
	if fromDate != "" {
		fromTime, err = time.Parse("2006-01-02", fromDate)
		if err != nil {
			return err
		}
	}
	if toDate != "" {
		toTime, err = time.Parse("2006-01-02", toDate)
		if err != nil {
			return err
		}
	}
	if toTime.Before(fromTime) {
		return fmt.Errorf("'from-date' can not be less that 'to-date'")
	}
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	if ctx.Config.Fuelings == nil {
		return fmt.Errorf("fueligns is not configured")
	}
	cfg, ok := ctx.Config.Fuelings.Providers[provider]
	if !ok {
		return fmt.Errorf(
			"provider config %q does not specified",
			provider,
		)
	}
	tags := map[string]string{"provider": provider}
	ctx.Signal("fuelings.fetch_report.start_sum", tags).Add(1)
	startTime := time.Now()
	defer func() {
		ctx.Signal("fuelings.fetch_report.duration_last", tags).
			Set(time.Since(startTime).Seconds())
	}()
	builder, ok := providers[cfg.Provider]
	if !ok {
		ctx.Signal("fuelings.fetch_report.error", tags).Add(1)
		return fmt.Errorf("provider %q does not exists", cfg.Provider)
	}
	impl, err := builder.New(cfg.Config)
	if err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to prepare provider: %w", err)
	}
	tableSchema, err := schema.Infer(ReportsTableRow{})
	if err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to infer schema: %w", err)
	}
	table := ctx.Config.Fuelings.ReportsTable.Rich()
	if _, err := yc.CreateNode(
		context.TODO(), table, yt.NodeTable,
		&yt.CreateNodeOptions{
			IgnoreExisting: true,
			Attributes: map[string]interface{}{
				"schema": tableSchema,
			},
		},
	); err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to create reports table: %w", err)
	}
	writer, err := yc.WriteTable(
		context.TODO(), table.SetAppend(), nil,
	)
	if err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to create writer: %w", err)
	}
	report, err := impl.FetchReport(ctx, fromTime, toTime)
	if err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to fetch report: %w", err)
	}
	if err := writer.Write(ReportsTableRow{
		Timestamp: time.Now().Unix(),
		Provider:  "v2/" + cfg.Provider,
		Data:      report,
	}); err != nil {
		if err := writer.Rollback(); err != nil {
			ctx.Logger.Warn("Unable to rollback", log.Error(err))
		}
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("unable to write report: %w", err)
	}
	if err := writer.Commit(); err != nil {
		ctx.Signal("fuelings.fetch_report.error_sum", tags).Add(1)
		return fmt.Errorf("failed to commit report: %w", err)
	}
	ctx.Signal("fuelings.fetch_report.success_sum", tags).Add(1)
	return nil
}

type Provider interface {
	New(cfg models.JSON) (Provider, error)
	Name() string
	ParseReport(data []byte) ([]OperationsTableRow, error)
	FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error)
}

var providers = map[string]Provider{}

func RegisterProvider(provider Provider) {
	name := provider.Name()
	if _, ok := providers[name]; ok {
		panic(fmt.Errorf("provider %q already registered", name))
	}
	providers[name] = provider
}
