package fuelings

import (
	"encoding/json"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/zootopia/analytics/drive/clients/rncard"
)

func init() {
	RegisterProvider(&rosneftProvider{})
}

type RosneftConfig struct {
	Endpoint string        `json:"endpoint"`
	Login    string        `json:"login"`
	Password secret.Secret `json:"password"`
	Contract secret.Secret `json:"contract"`
}

type rosneftProvider struct {
	config RosneftConfig
}

func (rosneftProvider) New(cfg models.JSON) (Provider, error) {
	var config RosneftConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &rosneftProvider{config: config}, nil
}

func (rosneftProvider) Name() string {
	return "rosneft"
}

func (rosneftProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	ops, err := rncard.ParseRawOperationsByContract(data)
	if err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, op := range ops {
		result = append(result, OperationsTableRow{
			Provider:    "rosneft",
			OperationID: op.ID,
			Card:        op.Card,
			Time:        op.Time,
			Total:       op.Total,
			Amount:      op.Amount,
			Price:       op.Price,
			Fuel:        getFixedFuel(op.FuelCode),
			Address:     op.Address,
			Region:      getFixedRegion(op.Region),
		})
	}
	return result, nil
}

func (p *rosneftProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client := rncard.NewClient(rncard.Config{
		Endpoint: p.config.Endpoint,
		Login:    p.config.Login,
		Password: p.config.Password.Secret(),
	})
	return client.GetRawOperationsByContract(p.config.Contract.Secret(), from, to)
}
