package fuelings

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/clients/tatneft"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/library/go/core/log"
)

func init() {
	RegisterProvider(&tatneftProvider{})
}

type TatneftConfig struct {
	Endpoint string        `json:"endpoint"`
	Login    secret.Secret `json:"login"`
	Password secret.Secret `json:"password"`
}

type tatneftProvider struct {
	config TatneftConfig
}

func (tatneftProvider) New(cfg models.JSON) (Provider, error) {
	var config TatneftConfig
	if err := json.Unmarshal(cfg, &config); err != nil {
		return nil, err
	}
	return &tatneftProvider{config: config}, nil
}

func (tatneftProvider) Name() string {
	return "tatneft"
}

func (tatneftProvider) ParseReport(data []byte) ([]OperationsTableRow, error) {
	var txs []tatneft.Tx
	if err := json.Unmarshal(data, &txs); err != nil {
		return nil, err
	}
	var result []OperationsTableRow
	for _, tx := range txs {
		result = append(result, OperationsTableRow{
			Provider:    "tatneft",
			OperationID: fmt.Sprintf("%d-%s", tx.Time.Unix(), tx.Card),
			Card:        tx.Card,
			Time:        tx.Time.Unix(),
			Total:       tx.Total,
			Amount:      tx.Amount,
			Price:       tx.Price,
			Fuel:        getFixedFuel(tx.FuelName),
		})
	}
	return result, nil
}

func (p *tatneftProvider) FetchReport(ctx *gotasks.Context, from, to time.Time) ([]byte, error) {
	client := tatneft.NewClient(p.config.Endpoint)
	session, err := client.Login(p.config.Login.Secret(), p.config.Password.Secret())
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := client.Logout(session); err != nil {
			ctx.Logger.Error("Unable to logout", log.Error(err))
		}
	}()
	if err := client.BuildReport(session, from, to); err != nil {
		ctx.Logger.Error("Unable to request build report", log.Error(err))
		return nil, err
	}
	reports, err := client.ListReports(session, from, to)
	if err != nil {
		ctx.Logger.Error("Unable to list build reports", log.Error(err))
		return nil, err
	}
	startDate := from.Truncate(24 * time.Hour).Format("2006-01-02T15:04:05")
	reportPos := -1
	for i := 1; i < 15 && reportPos == -1; i++ {
		for j, report := range reports {
			if !report.Finished() {
				continue
			}
			if report.Template != tatneft.ReportTemplate {
				continue
			}
			if report.StartDate != startDate {
				continue
			}
			reportPos = j
			break
		}
		if reportPos == -1 {
			select {
			case <-ctx.Context.Done():
				return nil, ctx.Context.Err()
			case <-time.After(time.Duration(i) * time.Second):
			}
			reports, err = client.ListReports(session, from, to)
			if err != nil {
				ctx.Logger.Error("Unable to list build reports", log.Error(err))
				return nil, err
			}
		}
	}
	if reportPos == -1 {
		ctx.Logger.Error("Unable to find report template")
		return nil, fmt.Errorf("unable to find report template")
	}
	txs, err := client.LoadReport(session, reports[0])
	if err != nil {
		ctx.Logger.Error("Unable to load report", log.Error(err))
		return nil, err
	}
	return json.Marshal(txs)
}
