package fuelings

import (
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/goback/models"
	"a.yandex-team.ru/drive/analytics/goback/models/tags"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

func init() {
	buildZapravkiTxsCmd := cobra.Command{
		Use: "build-zapravki-txs",
		Run: gotasks.WrapMain(buildZapravkiTxsMain),
	}
	buildZapravkiTxsCmd.PersistentFlags().String(
		"yt-proxy", "hahn", "YT proxy",
	)
	FuelingsCmd.AddCommand(&buildZapravkiTxsCmd)
	// Register MapReduce.
	mapreduce.Register(fuelingTagsMapper{})
	mapreduce.Register(zapravkiTxMapper{})
	mapreduce.Register(zapravkiTxReducer{})
}

type fuelingTagsMapper struct {
	mapreduce.Untyped
}

type fuelingTagsMapperRow struct {
	OrderID string `yson:"order_id"`
	CarID   string `yson:"car_id"`
	UserID  string `yson:"user_id"`
}

func (fuelingTagsMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var row models.Tag
		if err := in.Scan(&row); err != nil {
			return err
		}
		if row.Tag != "user_fueling_tag" {
			continue
		}
		var tagData tags.UserFuelingTagData
		if err := row.ScanData(&tagData); err != nil {
			return err
		}
		if tagData.OrderId == nil || tagData.ObjectId == nil {
			continue
		}
		if *tagData.OrderId == "" {
			continue
		}
		if err := out[0].Write(fuelingTagsMapperRow{
			OrderID: *tagData.OrderId,
			CarID:   *tagData.ObjectId,
			UserID:  row.ObjectID,
		}); err != nil {
			return err
		}
	}
	return nil
}

type zapravkiTxMapper struct {
	mapreduce.Untyped
}

type zapravkiTxMapperRow struct {
	ID   string `yson:"id"`
	Kind int    `yson:"kind"`
	// Tag.
	Tag *fuelingTagsMapperRow `yson:"tag"`
	// Order.
	Order *zapravkiOrder `yson:"order"`
}

type zapravkiOrder struct {
	ID     string  `yson:"Id"`
	Fuel   string  `yson:"FuelMarka"`
	Total  float64 `yson:"SumPaidCompleted"`
	Price  float64 `yson:"PriceFuelStationWithDiscountTotal"`
	Amount float64 `yson:"LitreCompleted"`
	Time   uint64  `yson:"DateUpdateUnix"`
}

func (zapravkiTxMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		switch in.TableIndex() {
		case 0:
			var row fuelingTagsMapperRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			if err := out[0].Write(zapravkiTxMapperRow{
				ID: row.OrderID, Kind: 0, Tag: &row,
			}); err != nil {
				return err
			}
		case 1:
			var row zapravkiOrder
			if err := in.Scan(&row); err != nil {
				return err
			}
			if err := out[0].Write(zapravkiTxMapperRow{
				ID: row.ID, Kind: 1, Order: &row,
			}); err != nil {
				return err
			}
		}
	}
	return nil
}

type zapravkiTxReducer struct {
	mapreduce.Untyped
}

func (r zapravkiTxReducer) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out)
	})
}

type zapravkiTx struct {
	ID     string  `yson:"id"`
	CarID  string  `yson:"car_id"`
	UserID string  `yson:"user_id"`
	Fuel   string  `yson:"fuel"`
	Total  float64 `yson:"total"`
	Price  float64 `yson:"price"`
	Amount float64 `yson:"amount"`
	Time   int64   `yson:"time"`
}

func (r zapravkiTxReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	var tag *fuelingTagsMapperRow
	var order *zapravkiOrder
	for in.Next() {
		var row zapravkiTxMapperRow
		if err := in.Scan(&row); err != nil {
			return err
		}
		switch row.Kind {
		case 0:
			tag = row.Tag
		case 1:
			order = row.Order
		}
	}
	if tag != nil && order != nil {
		return out[0].Write(zapravkiTx{
			ID:     order.ID,
			CarID:  tag.CarID,
			UserID: tag.UserID,
			Fuel:   order.Fuel,
			Total:  order.Total,
			Price:  order.Price,
			Amount: order.Amount,
			Time:   int64(order.Time),
		})
	}
	return nil
}

func buildZapravkiTxsMain(ctx *gotasks.Context) error {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	ctx.Signal("fuelings.fetch_zapravki_txs.start_sum", nil).Add(1)
	startTime := time.Now()
	defer func() {
		ctx.Signal("fuelings.fetch_zapravki_txs.duration_last", nil).
			Set(time.Since(startTime).Seconds())
	}()
	if err := goyt.New(yc).WithTx(func(tx *goyt.Tx) error {
		temp, err := tx.TempTable("")
		if err != nil {
			return err
		}
		mapSpec := spec.Spec{
			InputTablePaths: []ypath.YPath{
				ctx.Config.YTPaths.UserTagsHistoryTable,
			},
			OutputTablePaths: []ypath.YPath{temp},
			Pool:             "carsharing",
		}
		mapOp, err := tx.RawMR().Map(fuelingTagsMapper{}, mapSpec.Map())
		if err != nil {
			return err
		}
		if err := mapOp.Wait(); err != nil {
			return err
		}
		rowSchema, err := schema.Infer(zapravkiTx{})
		if err != nil {
			return err
		}
		mrSpec := spec.Spec{
			InputTablePaths: []ypath.YPath{
				temp,
				ypath.Path("//home/zapravki/production/replica/mongo/struct/orders_full"),
			},
			OutputTablePaths: []ypath.YPath{
				ypath.Path("//home/carsharing/production/data/fuelings/zapravki").
					Rich().SetSchema(rowSchema),
			},
			Pool:     "carsharing",
			SortBy:   []string{"id"},
			ReduceBy: []string{"id"},
		}
		mrOp, err := tx.RawMR().MapReduce(
			zapravkiTxMapper{}, zapravkiTxReducer{}, mrSpec.MapReduce(),
		)
		if err != nil {
			return err
		}
		if err := mrOp.Wait(); err != nil {
			return err
		}
		sortSpec := spec.Spec{
			InputTablePaths: []ypath.YPath{
				ypath.Path("//home/carsharing/production/data/fuelings/zapravki"),
			},
			OutputTablePath: ypath.Path("//home/carsharing/production/data/fuelings/zapravki"),
			SortBy:          []string{"time", "id"},
			Pool:            "carsharing",
		}
		sortOp, err := tx.RawMR().Sort(sortSpec.Sort())
		if err != nil {
			return err
		}
		return sortOp.Wait()
	}); err != nil {
		ctx.Signal("fuelings.fetch_zapravki_txs.error_sum", nil).Add(1)
		return err
	} else {
		ctx.Signal("fuelings.fetch_zapravki_txs.success_sum", nil).Add(1)
		return nil
	}
}
