package users

import (
	"database/sql"
	"encoding/json"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/library/go/gosql"
)

type RobotState struct {
	ID     int64       `db:"id" yson:"id"`
	Robot  string      `db:"robot" yson:"robot"`
	UserID string      `db:"user_id" yson:"user_id"`
	State  models.JSON `db:"state" yson:"state"`
}

func (s *RobotState) ScanState(v interface{}) error {
	return json.Unmarshal(s.State, &v)
}

func (s *RobotState) SetState(v interface{}) error {
	var err error
	s.State, err = json.Marshal(v)
	return err
}

type RobotStateStore struct {
	db    *gosql.DB
	table string
}

// DB returns database that used by store.
func (s *RobotStateStore) DB() *gosql.DB {
	return s.db
}

func (s *RobotStateStore) GetByRobotUserTx(
	tx gosql.Runner, robot, userID string,
) (RobotState, error) {
	var state RobotState
	names, scans := gosql.StructNameValues(&state, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(
		gosql.Column("robot").Equal(robot).
			And(gosql.Column("user_id").Equal(userID)),
	).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return RobotState{}, err
	}
	return state, nil
}

func (s *RobotStateStore) GetForUpdateByRobotUserTx(
	tx gosql.Runner, robot, userID string,
) (RobotState, error) {
	var state RobotState
	names, scans := gosql.StructNameValues(&state, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(
		gosql.Column("robot").Equal(robot).
			And(gosql.Column("user_id").Equal(userID)),
	).ForUpdate(true).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return RobotState{}, err
	}
	return state, nil
}

func (s *RobotStateStore) CreateTx(
	tx gosql.Runner, state *RobotState,
) error {
	names, values := gosql.StructNameValues(state, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if len(state.State) == 0 {
		state.State = models.JSON("null")
	}
	return row.Scan(&state.ID)
}

func (s *RobotStateStore) UpdateTx(
	tx gosql.Runner, state RobotState,
) error {
	names, values := gosql.StructNameValues(state, false, "id")
	query, values := s.db.Update(s.table).
		Names(names...).Values(values...).
		Where(gosql.Column("id").Equal(state.ID)).Build()
	if res, err := tx.Exec(query, values...); err != nil {
		return err
	} else if rows, err := res.RowsAffected(); err != nil {
		return err
	} else if rows != 1 {
		return sql.ErrNoRows
	}
	return nil
}

func NewRobotStateStore(db *gosql.DB, table string) *RobotStateStore {
	return &RobotStateStore{db: db, table: table}
}
