package orders

import (
	"container/list"
	"fmt"
	"math"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/proto"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/goback/models/tags"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/gotasks/models/cars"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
	"a.yandex-team.ru/zootopia/analytics/drive/helpers"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/geom"

	bp "a.yandex-team.ru/drive/backend/proto"
	tp "a.yandex-team.ru/drive/telematics/protocol/proto"
	gp "a.yandex-team.ru/rtline/library/geometry/protos"
)

func init() {
	updateLatestCmd := cobra.Command{
		Use: "update-latest",
		Run: gotasks.WrapMain(updateLatestMain),
	}
	updateLatestCmd.Flags().String("yt-proxy", "hahn", "YT proxy")
	updateLatestCmd.Flags().String("drive", "prestable", "Name of Drive client")
	OrdersCmd.AddCommand(&updateLatestCmd)
	// Register mapreduce operations.
	mapreduce.Register(&ordersMapper{})
	mapreduce.Register(&ordersReducer{})
	mapreduce.Register(&enrichMapper{})
	mapreduce.Register(&enrichReducer{})
}

func updateLatestMain(ctx *gotasks.Context) (errMain error) {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	driveName, err := ctx.Cmd.Flags().GetString("drive")
	if err != nil {
		return err
	}
	drive, ok := ctx.Drives[driveName]
	if !ok {
		return fmt.Errorf("invalid drive name %q", driveName)
	}
	rowSchema, err := schema.Infer(Order{})
	if err != nil {
		return err
	}
	areas, err := drive.GetAreas()
	if err != nil {
		return err
	}
	walletInfos, err := getWalletInfos(ctx, yc)
	if err != nil {
		return err
	}
	tx, err := yc.BeginTx(ctx.Context, nil)
	if err != nil {
		return err
	}
	defer func() {
		if errMain == nil {
			errMain = tx.Commit()
			return
		}
		_ = tx.Abort()
	}()
	mr := mapreduce.New(yc).WithTx(tx)
	latestTable := ctx.Config.YTPaths.LatestOrdersTable
	currentTable := ctx.Config.YTPaths.CurrentOrdersTable
	// Build orders joining compiled_rides and car_tags_history.
	ordersSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{
			ctx.Config.YTPaths.CompiledRidesTable,
			ctx.Config.YTPaths.CarTagsHistoryTable,
			ctx.Config.YTPaths.ExtendedCarsTable,
		},
		OutputTablePaths: []ypath.YPath{
			latestTable.Rich().SetSchema(rowSchema),
			currentTable.Rich().SetSchema(rowSchema),
		},
		ReduceBy: []string{"car_id"},
		SortBy:   []string{"car_id", "time", "event_id"},
		Pool:     "carsharing",
		Weight:   10,
	}
	ordersOp, err := mr.MapReduce(
		&ordersMapper{},
		&ordersReducer{
			Time: time.Now().Unix(),
		},
		ordersSpec.MapReduce(),
	)
	if err != nil {
		return err
	}
	if err := ordersOp.Wait(); err != nil {
		return err
	}
	tables := []ypath.YPath{latestTable}
	tableParts := []int{latestPart}
	addPart := func(path ypath.Path, kind int) {
		if len(path.String()) == 0 {
			ctx.Logger.Warn(
				"Table part disabled",
				log.String("path", path.String()),
				log.Int("kind", kind),
			)
			return
		}
		tables = append(tables, path)
		tableParts = append(tableParts, kind)
	}
	addPart(ctx.Config.YTPaths.CompiledBillsTable, compiledBillsPart)
	addPart(ctx.Config.YTPaths.OrdersTelematicsStatsTable, ordersTmStatsPart)
	addPart(ctx.Config.YTPaths.OrdersCostProfitTable, ordersCostProfitPart)
	addPart(ctx.Config.YTPaths.OrdersCarDistributionTable, ordersCarDistributionPart)
	addPart(ctx.Config.YTPaths.SessionsUserAgentTable, sessionsUserAgentPart)
	enrichSpec := spec.Spec{
		InputTablePaths: tables,
		OutputTablePaths: []ypath.YPath{
			latestTable.Rich().SetSchema(rowSchema),
		},
		ReduceBy: []string{"session_id"},
		SortBy:   []string{"session_id"},
		Pool:     "carsharing",
		Weight:   10,
	}
	enrichOp, err := mr.MapReduce(
		&enrichMapper{
			Parts:       tableParts,
			Areas:       areas,
			WalletInfos: walletInfos,
		},
		&enrichReducer{},
		enrichSpec.MapReduce(),
	)
	if err != nil {
		return err
	}
	if err := enrichOp.Wait(); err != nil {
		return err
	}
	sortSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{latestTable},
		OutputTablePath: latestTable,
		SortBy:          []string{"finish_timestamp"},
		Pool:            "carsharing",
		Weight:          10,
	}
	sortOp, err := mr.Sort(sortSpec.Sort())
	if err != nil {
		return err
	}
	return sortOp.Wait()
}

const (
	latestPart = iota
	compiledBillsPart
	ordersTmStatsPart
	ordersCostProfitPart
	ordersCarDistributionPart
	sessionsUserAgentPart
)

type Discount struct {
	ID       string  `yson:"id"`
	Discount float64 `yson:"discount"`
	Title    string  `yson:"title"`
}

type Step struct {
	StartTime    int64  `yson:"start_timestamp"`
	FinishTime   int64  `yson:"finish_timestamp,omitempty"`
	FreeDuration int64  `yson:"free_duration,omitempty"`
	Type         string `yson:"type"`
}

type PriceModel struct {
	Name   string  `yson:"name"`
	Before float64 `yson:"before"`
	After  float64 `yson:"after"`
}

type Area struct {
	ID string `yson:"id"`
}

type Order struct {
	FinishTime             int64     `yson:"finish_timestamp"`
	StartTime              int64     `yson:"start_timestamp"`
	OfferType              string    `yson:"offer_type"`
	PriceConstructorID     string    `yson:"price_constructor_id"`
	BehaviourConstructorID string    `yson:"behaviour_constructor_id"`
	ID                     string    `yson:"session_id"`
	CarID                  uuid.UUID `yson:"object_id"`
	UserID                 string    `yson:"user_id"`
	Currency               string    `yson:"currency"`
	Total                  float64   `yson:"total"`
	TariffName             string    `yson:"tariff_name"`
	GroupName              string    `yson:"group_name"`
	FromScanner            bool      `yson:"from_scanner"`
	ParkingPrice           float64   `yson:"parking_price"`
	RidingPrice            float64   `yson:"riding_price"`
	FreeDuration           int64     `yson:"free_duration"`
	PricedDuration         int64     `yson:"priced_duration"`
	StartPointLatitude     float64   `yson:"start_point_latitude"`
	StartPointLongitude    float64   `yson:"start_point_longitude"`
	FinishPointLatitude    float64   `yson:"finish_point_latitude"`
	FinishPointLongitude   float64   `yson:"finish_point_longitude"`
	// Parking fields.
	ParkingTotal    float64 `yson:"parking_total"`
	ParkingDuration int64   `yson:"parking_duration"`
	ParkingCount    int64   `yson:"parking_count"`
	// Riding fields.
	RidingTotal      float64 `yson:"riding_total"`
	RidingDuration   int64   `yson:"riding_duration"`
	TmRidingDuration int64   `yson:"tm_riding_duration"`
	RidingCount      int64   `yson:"riding_count"`
	// Reservation fields.
	ReservationTotal    float64 `yson:"reservation_total"`
	ReservationDuration int64   `yson:"reservation_duration"`
	// Acceptance fields.
	AcceptanceTotal    float64 `yson:"acceptance_total"`
	AcceptanceDuration int64   `yson:"acceptance_duration"`
	// Discount fields.
	DiscountTotal float64            `yson:"discount_total"`
	Discounts     []Discount         `yson:"discounts"`
	DiscountCosts map[string]float64 `yson:"discount_costs"`
	// Bonus fields.
	BonusTotal float64 `yson:"bonus_total"`
	// Total fields.
	OverrunTotal  float64 `yson:"overrun_total"`
	OvertimeTotal float64 `yson:"overtime_total"`
	// Wallet fields.
	Wallet       string             `yson:"wallet"`
	ParentWallet string             `yson:"parent_wallet"`
	WalletTotals map[string]float64 `yson:"wallet_totals"`
	// AccountIDs contains billing account ID.
	AccountIDs []uint32 `yson:"account_ids"`
	// Other fields.
	Mileage      float64 `yson:"mileage"`
	MileageLimit float64 `yson:"mileage_limit"`
	// MileageLimitModels contains price models applied to this order.
	MileageLimitModels []PriceModel `yson:"mileage_limit_models"`
	// DurationLimit.
	DurationLimit int64 `yson:"duration_limit"`
	// DurationLimitModels contains price models applied to this order.
	DurationLimitModels []PriceModel `yson:"duration_limit_models"`
	// PriceModel.
	PriceModel string `yson:"price_model"`
	// Object fields.
	ObjectIMEI   uint64 `yson:"object_imei"`
	ObjectHeadID string `yson:"object_head_id"`
	ObjectModel  string `yson:"object_model"`
	// Steps contains information about all steps.
	Steps []Step `yson:"steps"`
	// Ptags fields.
	StartPtags  []string `yson:"start_ptags"`
	FinishPtags []string `yson:"finish_ptags"`
	// Deprecated.
	City string `yson:"city"`
	// City fields.
	StartCity  string `yson:"start_city"`
	FinishCity string `yson:"finish_city"`
	// Pack fields.
	PackPrice float64 `yson:"pack_price"`
	// PackPriceModels.
	PackPriceModels []PriceModel `yson:"pack_price_models"`
	// Fix point fields.
	FixPointFinishArea geom.Polygon `yson:"fix_point_finish_area"`
	// Fix point finish position fields.
	FixPointFinishLatitude  float64 `yson:"fix_point_finish_latitude"`
	FixPointFinishLongitude float64 `yson:"fix_point_finish_longitude"`
	// InsuranceType fields.
	InsuranceType string `yson:"insurance_type"`
	// PriceModels contains price models applied to this order.
	PriceModels []PriceModel `yson:"price_models"`
	// ParkingPriceModels contains price models applied to this order.
	ParkingPriceModels []PriceModel `yson:"parking_price_models"`
	// CashbackPercentModels contains optional cashback models applied to this order.
	CashbackPercentModels []PriceModel `yson:"cashback_percent_models"`
	// FinishMinPrice contains minimal price for order.
	FinishMinPrice float64 `yson:"finish_min_price"`
	// FinishAddPrice contains additional price for order.
	FinishAddPrice float64 `yson:"finish_add_price"`
	// OverrunPrice contains overrun price for km.
	OverrunPrice float64 `yson:"overrun_price"`
	// OverrunPriceModels.
	OverrunPriceModels []PriceModel `yson:"overrun_price_models"`
	// DropZoneTotal contains total for drop zone.
	DropZoneTotal float64 `yson:"drop_zone_total"`
	// ExternalUserID contains external user ID.
	ExternalUserID string `yson:"external_user_id"`
	// Origin contains order origin.
	Origin string `yson:"origin"`
	// FixPointRouteDuration contains duration of fix point route.
	FixPointRouteDuration int64 `yson:"fix_point_route_duration"`
	// FixPointRouteLength contains length of fix point route.
	FixPointRouteLength float64 `yson:"fix_point_route_length"`
	// FixPointAddRouteDuration contains additional route duration.
	FixPointAddRouteDuration int64 `yson:"fix_point_add_route_duration"`
	// FixPointAddRouteLength contains additional route length.
	FixPointAddRouteLength float64 `yson:"fix_point_add_route_length"`
	// StartAreas contains IDs of start areas.
	StartAreas []Area `yson:"start_areas"`
	// FinishAreas contains IDs of finish areas.
	FinishAreas []Area `yson:"finish_areas"`
	// UseMapsRouter contains flag that maps router is used.
	UseMapsRouter bool `yson:"use_maps_router"`
	// Cashback contains percentage of cashback.
	Cashback float64 `yson:"cashback"`
	// CashbackTotal contains size of cashback.
	CashbackTotal float64 `yson:"cashback_total"`
	// ParentID contains id of parent session.
	ParentID string `yson:"parent_id"`
	// RidingCost contains riding cost per minute.
	RidingCost float64 `yson:"riding_cost"`
	// RidingCost contains parking cost per minute.
	ParkingCost float64 `yson:"parking_cost"`
	// RidingCost contains leasing cost per minute.
	LeasingCost float64 `yson:"leasing_cost"`
	// CostTotal contains total cost of order.
	CostTotal float64 `yson:"cost_total"`
	// ProfitTotal contains total profit of order.
	ProfitTotal float64 `yson:"profit_total"`
	// DynamicCostTotal contains total dynamic cost of order.
	DynamicCostTotal float64 `yson:"dynamic_cost_total"`
	// DynamicCostTotal contains total dynamic cost of order.
	DynamicCostNewTotal float64 `yson:"dynamic_cost_new_total"`
	// LeasingDynamicCostTotal contains total leasing dynamic cost of order.
	LeasingDynamicCostTotal float64 `yson:"leasing_dynamic_cost_total"`
	// DynamicProfitTotal contains total dynamic profit of order.
	DynamicProfitTotal float64 `yson:"dynamic_profit_total"`
	// DynamicProfitTotal contains total dynamic profit of order.
	DynamicProfitNewTotal float64 `yson:"dynamic_profit_new_total"`
	// ReturnDurationLimit contains duration limit for alowed recalc.
	ReturnDurationLimit int64 `yson:"return_duration_limit"`
	// AcceptancePrice contains price of acceptance.
	AcceptancePrice float64 `yson:"acceptance_price"`
	// CarDistributionIsGoodOrder.
	CarDistributionIsGoodOrder *int32 `yson:"car_distribution_is_good_order"`
	// CarDistributionMseBefore.
	CarDistributionMseBefore *float64 `yson:"car_distribution_mse_before"`
	// CarDistributionMseAfter.
	CarDistributionMseAfter *float64 `yson:"car_distribution_mse_after"`
	// CarDistributionMseDiff.
	CarDistributionMseDiff *float64 `yson:"car_distribution_mse_diff"`
	// StartOdoMileage.
	StartOdoMileage float64 `yson:"start_odo_mileage"`
	// RouteDurationModels contains price models applied to this order.
	RouteDurationModels []PriceModel `yson:"route_duration_models"`
	// Float2Features.
	Float2Features []float64 `yson:"float2_features"`
	// Category2Features.
	Category2Features []string `yson:"category2_features"`
	// IsPlusUser.
	IsPlusUser bool `yson:"is_plus_user"`
	// ServicingDuration.
	ServicingDuration int64 `yson:"servicing_duration"`
	// User agent info.
	OSFamily       string `yson:"os_family"`
	DeviceIDAccept string `yson:"device_id_accept"`
	// DepositTotal.
	DepositTotal float64 `yson:"deposit_total"`
	// PriceModels.
	AcceptancePriceModels         []PriceModel `yson:"acceptance_price_models"`
	FixPointAcceptancePriceModels []PriceModel `yson:"fix_point_acceptance_price_models"`
	// StandardWithDiscountArea offer fields.
	SWDAFinishArea   geom.Polygon `yson:"swda_finish_area"`
	SWDADiscount     *float64     `yson:"swda_discount"`
	SWDAIsDiscounted *bool        `yson:"swda_is_discounted"`
	// DebugInfo
	DebugInfo DebugInfo `yson:"debug_info"`
}

type DebugInfo struct {
	HasHistory         bool `yson:"has_history"`
	HasUserAgent       bool `yson:"has_user_agent"`
	HasCompiledBill    bool `yson:"has_compiled_bill"`
	HasTmStats         bool `yson:"has_tm_stats"`
	HasCarDistribution bool `yson:"has_car_distribution"`
	HasCostProfit      bool `yson:"has_cost_profit"`
}

type ordersMapper struct{}

func (m ordersMapper) InputTypes() []interface{} {
	return []interface{}{models.CompiledRide{}, models.CarTagEvent{}}
}

func (m ordersMapper) OutputTypes() []interface{} {
	return []interface{}{latestMapperRow{}}
}

type latestMapperRow struct {
	Type    int           `yson:"type"`
	CarID   string        `yson:"car_id"`
	Time    int64         `yson:"time"`
	EventID int64         `yson:"event_id"`
	Row     yson.RawValue `yson:"row"`
}

func isCarTagTracked(tag string) bool {
	return tag == "old_state_reservation" ||
		tag == "old_state_acceptance" ||
		tag == "old_state_riding" ||
		tag == "old_state_parking" ||
		tag == "servicing"
}

func (m *ordersMapper) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		rawRow := latestMapperRow{Type: in.TableIndex()}
		switch rawRow.Type {
		case 0: // CompiledRide.
			event := models.CompiledRide{}
			if err := in.Scan(&event); err != nil {
				return err
			}
			rawRow.CarID = event.ObjectID.String()
			rawRow.Time = event.HistoryTimestamp
			rawRow.EventID = event.HistoryEventID
			var err error
			if rawRow.Row, err = yson.Marshal(event); err != nil {
				return err
			}
		case 1: // CarTagEvent.
			event := models.CarTagEvent{}
			if err := in.Scan(&event); err != nil {
				return err
			}
			rawRow.CarID = event.ObjectID.String()
			rawRow.Time = event.HistoryTimestamp
			rawRow.EventID = event.HistoryEventID
			// Skip tags that not used by orders latest.
			if !isCarTagTracked(string(event.Tag)) {
				continue
			}
			var err error
			if rawRow.Row, err = yson.Marshal(event); err != nil {
				return err
			}
		case 2: // ExtendedCar.
			car := cars.ExtendedCar{}
			if err := in.Scan(&car); err != nil {
				return err
			}
			rawRow.CarID = car.CarID.String()
			rawRow.Time = math.MinInt64
			rawRow.EventID = math.MinInt64
			var err error
			if rawRow.Row, err = yson.Marshal(car); err != nil {
				return err
			}
		}
		if err := out[0].Write(rawRow); err != nil {
			return err
		}
	}
	return nil
}

type session struct {
	Order
	CurrentStep Step `yson:"current_step"`
}

type ordersReducer struct {
	// Time contains last time for reducer.
	Time int64
	// rides contains compiled rides.
	rides *list.List
	// sessions represents simple sessions.
	sessions map[uuid.UUID]*session
	// orders represents simple orders.
	orders map[string]Order
	// car contains information about car.
	car cars.ExtendedCar
}

func (r ordersReducer) InputTypes() []interface{} {
	return []interface{}{latestMapperRow{}}
}

func (r ordersReducer) OutputTypes() []interface{} {
	return []interface{}{Order{}}
}

func (r *ordersReducer) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out)
	})
}

func (r *ordersReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	r.rides = list.New()
	r.sessions = map[uuid.UUID]*session{}
	r.orders = map[string]Order{}
	r.car = cars.ExtendedCar{}
	for in.Next() {
		var rawRow latestMapperRow
		if err := in.Scan(&rawRow); err != nil {
			return err
		}
		switch rawRow.Type {
		case 0: // CompiledRide.
			var row models.CompiledRide
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			r.rides.PushBack(row)
		case 1: // CarTagEvent.
			var row models.CarTagEvent
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			switch row.Tag {
			case "old_state_reservation":
				switch row.HistoryAction {
				case models.SetPerformerAction:
					// User started his session.
					r.startSession(row)
				case models.DropPerformerAction:
					// User finished his session.
					r.finishSession(row)
				case models.RemoveAction:
					// Implicitly finish current session.
					// This is bad behaviour of backend.
					r.finishSession(row)
				}
			case "old_state_acceptance", "old_state_parking", "old_state_riding", "servicing":
				switch row.HistoryAction {
				case models.EvolveAction:
					r.updateSessionStep(row)
				}
			}
		case 2:
			if err := yson.Unmarshal(rawRow.Row, &r.car); err != nil {
				return err
			}
		}
		if err := r.writeRows(out, rawRow.Time); err != nil {
			return err
		}
	}
	if err := r.writeRows(out, r.Time); err != nil {
		return err
	}
	// Save tail orders to the current table.
	for _, row := range r.orders {
		if r.car.CarID != uuid.Nil {
			updateOrderWithCar(&row, r.car)
		}
		if err := out[1].Write(row); err != nil {
			return err
		}
	}
	// Save running sessions to the current table.
	for _, row := range r.sessions {
		order := row.Order
		order.Steps = append(order.Steps, Step{
			Type:      row.CurrentStep.Type,
			StartTime: row.CurrentStep.StartTime,
		})
		if r.car.CarID != uuid.Nil {
			updateOrderWithCar(&order, r.car)
		}
		if err := out[1].Write(order); err != nil {
			return err
		}
	}
	return nil
}

func (r *ordersReducer) writeRows(out []mapreduce.Writer, ts int64) error {
	for r.rides.Front() != nil {
		ride := r.rides.Front().Value.(models.CompiledRide)
		if ride.HistoryTimestamp+60*5 > ts {
			break
		}
		r.rides.Remove(r.rides.Front())
		rideData, err := ride.ParseData()
		if err != nil {
			return err
		}
		row := Order{
			FinishTime:    ride.Finish,
			StartTime:     ride.Start,
			OfferType:     StandartOfferType,
			ID:            ride.SessionID,
			CarID:         ride.ObjectID.UUID,
			UserID:        ride.HistoryUserID,
			Currency:      "RUB",
			Total:         float64(ride.Price) / 100,
			City:          "msc_area",
			StartCity:     helpers.Moscow,
			FinishCity:    helpers.Moscow,
			DiscountCosts: map[string]float64{},
		}
		updateOrderWithDiff(&row, rideData.TSnapshotsDiff)
		updateOrderWithBill(&row, rideData.TCompiledRidingHard.Bill)
		updateOrderWithOffer(&row, rideData.TCompiledRidingHard.Offer)
		if r.car.CarID != uuid.Nil {
			updateOrderWithCar(&row, r.car)
		}
		if order, ok := r.orders[row.ID]; ok {
			// Remove order from memory.
			delete(r.orders, row.ID)
			// Update order with tag order.
			updateOrderWithOrder(&row, order)
			// Mark that row updated with car_tags_history.
			row.DebugInfo.HasHistory = true
		}
		if err := out[0].Write(row); err != nil {
			return err
		}
	}
	return nil
}

func updateOrderWithCar(row *Order, car cars.ExtendedCar) {
	row.ObjectModel = car.Model
	for _, imei := range car.IMEILog {
		if imei.BeginTime <= row.StartTime {
			row.ObjectIMEI = imei.IMEI
		}
	}
	if row.ObjectIMEI == 0 {
		row.ObjectIMEI = car.IMEI
	}
	for _, head := range car.HeadLog {
		if head.BeginTime <= row.StartTime {
			row.ObjectHeadID = head.HeadID
		}
	}
	if len(row.ObjectHeadID) == 0 {
		row.ObjectHeadID = car.HeadID
	}
}

func updateOrderWithDiff(row *Order, diff *bp.TSnapshotsDiff) {
	if diff == nil {
		return
	}
	if diff.Mileage != nil {
		row.Mileage = float64(*diff.Mileage)
	}
	if diff.Start != nil {
		if diff.Start.Latitude != nil {
			row.StartPointLatitude = *diff.Start.Latitude
		}
		if diff.Start.Longitude != nil {
			row.StartPointLongitude = *diff.Start.Longitude
		}
	}
	if diff.Last != nil {
		if diff.Last.Latitude != nil {
			row.FinishPointLatitude = *diff.Last.Latitude
		}
		if diff.Last.Longitude != nil {
			row.FinishPointLongitude = *diff.Last.Longitude
		}
	}
}

func updateOrderWithBill(row *Order, bill *bp.TBill) {
	if bill == nil {
		return
	}
	if bill.FreeDuration != nil {
		row.FreeDuration = int64(*bill.FreeDuration)
	}
	if bill.PricedDuration != nil {
		row.PricedDuration = int64(*bill.PricedDuration)
	}
	for _, record := range bill.Record {
		switch *record.Type {
		case "old_state_reservation":
			if *record.Cost < 0 {
				panic("old_state_reservation: cost is less than zero")
			}
			row.ReservationTotal += float64(*record.Cost) / 100
			row.ReservationDuration += int64(*record.Duration)
		case "old_state_acceptance":
			if *record.Cost < 0 {
				panic("old_state_acceptance: cost is less than zero")
			}
			row.AcceptanceTotal += float64(*record.Cost) / 100
			row.AcceptanceDuration += int64(*record.Duration)
		case "old_state_riding":
			if *record.Cost < 0 {
				panic("old_state_riding: cost is less than zero")
			}
			row.RidingTotal += float64(*record.Cost) / 100
			row.RidingDuration += int64(*record.Duration)
		case "old_state_parking":
			if *record.Cost < 0 {
				panic("old_state_parking: cost is less than zero")
			}
			row.ParkingTotal += float64(*record.Cost) / 100
			row.ParkingDuration += int64(*record.Duration)
		case "discount":
			if *record.Cost > 0 {
				panic("discount: cost is greater than zero")
			}
			var discount float64
			if _, err := fmt.Sscanf(
				*record.Details, "%v%%", &discount,
			); err != nil {
				panic(err)
			}
			rowDiscount := Discount{
				ID:       *record.Id,
				Discount: discount / 100,
			}
			if record.Title != nil {
				rowDiscount.Title = *record.Title
			}
			row.Discounts = append(row.Discounts, rowDiscount)
			row.DiscountCosts[*record.Id] -= float64(*record.Cost) / 100
			row.DiscountTotal -= float64(*record.Cost) / 100
		case "billing_bonus":
			cost := *record.Cost
			if cost < 0 {
				cost = -cost
			}
			row.BonusTotal += float64(cost) / 100
		case "pack":
			if *record.Cost < 0 {
				panic("pack: cost is less than zero")
			}
			row.PackPrice += float64(*record.Cost) / 100
		case "overrun":
			if *record.Cost < 0 {
				panic("overrun: cost is less than zero")
			}
			row.OverrunTotal += float64(*record.Cost) / 100
		case "overtime":
			if *record.Cost < 0 {
				panic("overtime: cost is less than zero")
			}
			row.OvertimeTotal += float64(*record.Cost) / 100
		case "fee_drop_zone_max", "fee_drop_zone_fix":
			if *record.Cost < 0 {
				panic("fee_drop_zone: cost is less than zero")
			}
			row.DropZoneTotal += float64(*record.Cost) / 100
		case "cashback":
			if *record.Cost < 0 {
				panic("cashback: cost is less than zero")
			}
			row.CashbackTotal += float64(*record.Cost) / 100
		case "total", "billing_wallet", "mileage", "acceptance_cost":
			// Ignore records with this type.
		default:
			fmt.Fprintln(os.Stderr, "unsupported record type:", *record.Type)
		}
	}
}

func updateOrderWithOffer(row *Order, offer *bp.TOffer) {
	if offer == nil {
		return
	}
	if offer.InstanceType != nil {
		row.OfferType = *offer.InstanceType
	}
	if offer.Name != nil {
		row.TariffName = strings.TrimSpace(*offer.Name)
	}
	if offer.GroupName != nil {
		row.GroupName = *offer.GroupName
	}
	if offer.FromScanner != nil {
		row.FromScanner = *offer.FromScanner
	}
	// Extract data for standart offer.
	if standard := offer.StandartOffer; standard != nil {
		if standard.PriceRiding != nil {
			row.RidingPrice = float64(*standard.PriceRiding) / 100
		}
		if standard.PriceParking != nil {
			row.ParkingPrice = float64(*standard.PriceParking) / 100
		}
		if standard.PriceModel != nil {
			row.PriceModel = *standard.PriceModel
		}
		if standard.InsuranceType != nil {
			row.InsuranceType = *standard.InsuranceType
		}
		if len(standard.PriceModelInfo) > 0 {
			row.PriceModels = nil
			for _, info := range standard.PriceModelInfo {
				row.PriceModels = append(row.PriceModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before) / 100,
					After:  float64(*info.After) / 100,
				})
			}
		}
		if len(standard.ParkingPriceModelInfo) > 0 {
			row.ParkingPriceModels = nil
			for _, info := range standard.ParkingPriceModelInfo {
				row.ParkingPriceModels = append(row.ParkingPriceModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before) / 100,
					After:  float64(*info.After) / 100,
				})
			}
		}
		if len(standard.CashbackPercentModelInfo) > 0 {
			row.CashbackPercentModels = nil
			for _, info := range standard.CashbackPercentModelInfo {
				row.CashbackPercentModels = append(row.CashbackPercentModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before),
					After:  float64(*info.After),
				})
			}
		}
		if context := standard.FullPricesContext; context != nil {
			if market := context.Market; market != nil {
				if acceptance := market.Acceptance; acceptance != nil {
					if price := acceptance.Price; price != nil {
						row.AcceptancePrice = float64(*price) / 100
					}
					if len(acceptance.PriceModelInfos) > 0 {
						row.AcceptancePriceModels = nil
						for _, info := range acceptance.PriceModelInfos {
							row.AcceptancePriceModels = append(row.AcceptancePriceModels, PriceModel{
								Name:   *info.Name,
								Before: float64(*info.Before) / 100,
								After:  float64(*info.After) / 100,
							})
						}
					}
				}
			}
		}
		if features := standard.Features; features != nil {
			row.Float2Features = nil
			for _, value := range features.Float2 {
				row.Float2Features = append(row.Float2Features, float64(value))
			}
			row.Category2Features = nil
			row.Category2Features = append(row.Category2Features, features.Category2...)
		}
		row.DepositTotal = 0
		if standard.UseDeposit != nil && standard.DepositAmount != nil {
			if *standard.UseDeposit {
				row.DepositTotal = float64(*standard.DepositAmount) / 100
			}
		}
	}
	// Extract data for pack offer.
	if pack := offer.PackOffer; pack != nil {
		if pack.PackPrice != nil {
			row.PackPrice = float64(*pack.PackPrice) / 100
		}
		if pack.MileageLimit != nil {
			row.MileageLimit = float64(*pack.MileageLimit)
		}
		if pack.Duration != nil {
			row.DurationLimit = int64(*pack.Duration)
		}
		if pack.ReturningDuration != nil {
			row.ReturnDurationLimit = int64(*pack.ReturningDuration)
		}
		if pack.RerunPriceKM != nil {
			row.OverrunPrice = float64(*pack.RerunPriceKM) / 100
		}
		if len(pack.DurationModelInfo) > 0 {
			row.DurationLimitModels = nil
			for _, info := range pack.DurationModelInfo {
				row.DurationLimitModels = append(row.DurationLimitModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before),
					After:  float64(*info.After),
				})
			}
		}
		if len(pack.PackPriceModelInfo) > 0 {
			row.PackPriceModels = nil
			for _, info := range pack.PackPriceModelInfo {
				row.PackPriceModels = append(row.PackPriceModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before) / 100,
					After:  float64(*info.After) / 100,
				})
			}
		}
		if len(pack.MileageLimitModelInfo) > 0 {
			row.MileageLimitModels = nil
			for _, info := range pack.MileageLimitModelInfo {
				row.MileageLimitModels = append(row.MileageLimitModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before),
					After:  float64(*info.After),
				})
			}
		}
		if len(pack.OverrunKmModelInfo) > 0 {
			row.OverrunPriceModels = nil
			for _, info := range pack.OverrunKmModelInfo {
				row.OverrunPriceModels = append(row.OverrunPriceModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before) / 100,
					After:  float64(*info.After) / 100,
				})
			}
		}
	}
	// Extract data for fix point offer.
	if fixPoint := offer.FixPointOffer; fixPoint != nil {
		if fixPoint.RouteDuration != nil {
			row.FixPointRouteDuration = int64(*fixPoint.RouteDuration)
		}
		if fixPoint.RouteLength != nil {
			row.FixPointRouteLength = float64(*fixPoint.RouteLength) / 1000
		}
		if fixPoint.AdditionalRouteDuration != nil {
			row.FixPointAddRouteDuration =
				int64(*fixPoint.AdditionalRouteDuration)
		}
		if fixPoint.AdditionalRouteLength != nil {
			row.FixPointAddRouteLength =
				float64(*fixPoint.AdditionalRouteLength) / 1000
		}
		if fixPoint.FinishAdditionalPrice != nil {
			row.FinishAddPrice = float64(*fixPoint.FinishAdditionalPrice) / 100
		}
		if fixPoint.FinishMinPrice != nil {
			row.FinishMinPrice = float64(*fixPoint.FinishMinPrice) / 100
		}
		if fixPoint.UseMapsRouter != nil {
			row.UseMapsRouter = *fixPoint.UseMapsRouter
		}
		if fixPoint.Finish != nil {
			pos := castSimpleCoord(fixPoint.Finish)
			row.FixPointFinishLongitude = pos[0]
			row.FixPointFinishLatitude = pos[1]
		}
		if len(fixPoint.RouteDurationModelInfo) > 0 {
			row.RouteDurationModels = nil
			for _, info := range fixPoint.RouteDurationModelInfo {
				row.RouteDurationModels = append(row.RouteDurationModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before),
					After:  float64(*info.After),
				})
			}
		}
		if len(fixPoint.FinishArea) > 0 {
			row.FixPointFinishArea = nil
			for _, coord := range fixPoint.FinishArea {
				row.FixPointFinishArea = append(
					row.FixPointFinishArea, castSimpleCoord(coord),
				)
			}
		}
		if len(fixPoint.AcceptancePriceModelInfo) > 0 {
			row.FixPointAcceptancePriceModels = nil
			for _, info := range fixPoint.AcceptancePriceModelInfo {
				row.FixPointAcceptancePriceModels = append(row.FixPointAcceptancePriceModels, PriceModel{
					Name:   *info.Name,
					Before: float64(*info.Before) / 100,
					After:  float64(*info.After) / 100,
				})
			}
		}
	}
	// Extract data for standard with discount area offer.
	if discounted := offer.StandardWithDiscountAreaOffer; discounted != nil {
		if discounted.Discount != nil {
			row.SWDADiscount = ptrFloat64(float64(*discounted.Discount) / 10000)
		}
		if len(discounted.FinishArea) > 0 {
			row.SWDAFinishArea = nil
			for _, coord := range discounted.FinishArea {
				row.SWDAFinishArea = append(row.SWDAFinishArea, castSimpleCoord(coord))
			}
		}
	}
	// Build discounts.
	if info := offer.DiscountsInfo; info != nil {
		row.Discounts = nil
		for _, discount := range info.Discounts {
			if discount.Identifier == nil {
				// This is unknown issue. We should fail to investigate this.
				if *discount.Discount < -1e-6 || *discount.Discount > 1e-6 {
					panic(fmt.Errorf(
						"empty discount with non zero cost (session_id = %s)",
						row.ID,
					))
				}
				continue
			}
			// Skip empty discounts.
			if *discount.Discount > -1e-6 && *discount.Discount < 1e-6 {
				continue
			}
			sessionDiscount := Discount{}
			if discount.Identifier != nil {
				sessionDiscount.ID = *discount.Identifier
			}
			if discount.Title != nil {
				sessionDiscount.Title = *discount.Title
			}
			if discount.Discount != nil {
				sessionDiscount.Discount = *discount.Discount
			}
			// Add discounts to session.
			row.Discounts = append(row.Discounts, sessionDiscount)
		}
	}
	// Extract constructor ID.
	if offer.BehaviourConstructorId != nil {
		row.BehaviourConstructorID = *offer.BehaviourConstructorId
		row.PriceConstructorID = *offer.PriceConstructorId
	} else if offer.ConstructorId != nil {
		row.BehaviourConstructorID = *offer.ConstructorId
		row.PriceConstructorID = *offer.ConstructorId
	}
	// Detect selected wallet.
	if offer.SelectedCharge != nil {
		row.Wallet = *offer.SelectedCharge
	}
	// External info.
	if offer.ExternalUserId != nil {
		row.ExternalUserID = *offer.ExternalUserId
	}
	if offer.Origin != nil {
		row.Origin = *offer.Origin
	}
	if offer.CashbackPercent != nil {
		row.Cashback = float64(*offer.CashbackPercent) / 100
	}
	if offer.ParentId != nil {
		row.ParentID = *offer.ParentId
	}
	if offer.IsPlusUser != nil {
		row.IsPlusUser = *offer.IsPlusUser
	}
}

func ptrString(v string) *string {
	return &v
}

func ptrFloat64(v float64) *float64 {
	return &v
}

func ptrBool(v bool) *bool {
	return &v
}

func castSimpleCoord(point *gp.TSimpleCoord) geom.Vec2 {
	x := float64(0)
	if point.XHP != nil {
		x = *point.XHP
	} else if point.X != nil {
		x = float64(*point.X)
	}
	y := float64(0)
	if point.YHP != nil {
		y = *point.YHP
	} else if point.Y != nil {
		y = float64(*point.Y)
	}
	return geom.Vec2{x, y}
}

// updateOrderWithOrder updates compiled ride with order
// built only on tags history.
func updateOrderWithOrder(row *Order, order Order) {
	row.Steps = order.Steps
	row.City = order.City
	row.StartCity = order.StartCity
	row.FinishCity = order.FinishCity
	row.StartPtags = order.StartPtags
	row.FinishPtags = order.FinishPtags
	if order.TariffName != "" {
		row.TariffName = order.TariffName
	}
	if order.GroupName != "" {
		row.GroupName = order.GroupName
	}
	row.FromScanner = row.FromScanner || order.FromScanner
	if order.ParkingPrice != 0 {
		row.ParkingPrice = order.ParkingPrice
	}
	if order.RidingPrice != 0 {
		row.RidingPrice = order.RidingPrice
	}
	row.OfferType = order.OfferType
	row.PriceModel = order.PriceModel
	row.FixPointFinishArea = order.FixPointFinishArea
	row.ReservationDuration = 0
	row.AcceptanceDuration = 0
	row.RidingDuration = 0
	row.RidingCount = 0
	row.ParkingDuration = 0
	row.ParkingCount = 0
	row.ServicingDuration = 0
	for _, step := range order.Steps {
		duration := step.FinishTime - step.StartTime
		switch step.Type {
		case "old_state_reservation":
			row.ReservationDuration += duration
		case "old_state_acceptance":
			row.AcceptanceDuration += duration
		case "old_state_riding":
			row.RidingDuration += duration
			row.RidingCount++
		case "old_state_parking":
			row.ParkingDuration += duration
			row.ParkingCount++
		case "servicing":
			row.ServicingDuration += duration
		}
	}
	row.StartOdoMileage = order.StartOdoMileage
}

func getSensorFloat64(s *tp.TSensor) float64 {
	if s.ValueUI64 != nil {
		return float64(*s.ValueUI64)
	}
	if s.ValueDouble != nil {
		return *s.ValueDouble
	}
	if s.ValueFloat != nil {
		return float64(*s.ValueFloat)
	}
	if s.ValueString != nil {
		v, err := strconv.ParseFloat(*s.ValueString, 64)
		if err == nil {
			return v
		}
	}
	return 0
}

func getSessionCity(lon, lat float64, ptags []string) string {
	city := helpers.CityByPtags(ptags)
	if city == "" {
		city = helpers.CityByLocation(lon, lat)
	}
	if city == "" {
		city = helpers.Moscow
	}
	return city
}

func (r *ordersReducer) finishSession(e models.CarTagEvent) {
	session, ok := r.sessions[e.ID]
	if !ok {
		// In this case session already dropped.
		// This happens when after drop_performer goes remove.
		return
	}
	// Update session with last tag change
	session.FinishTime = e.HistoryTimestamp
	// Parse snapshot.
	if snapshot, err := unmarshalDeviceSnapshot(e.CarTag); err == nil {
		if snapshot.Location != nil {
			if snapshot.Location.Latitude != nil {
				session.FinishPointLatitude = *snapshot.Location.Latitude
			}
			if snapshot.Location.Longitude != nil {
				session.FinishPointLongitude = *snapshot.Location.Longitude
			}
		}
		session.FinishPtags = snapshot.TagInPoint
		session.FinishCity = getSessionCity(
			session.FinishPointLongitude,
			session.FinishPointLatitude,
			session.FinishPtags,
		)
	}
	// Finish current step.
	session.CurrentStep.FinishTime = e.HistoryTimestamp
	session.Steps = append(session.Steps, session.CurrentStep)
	// Remove session.
	delete(r.sessions, e.ID)
	// Old sessions does not have ID so we should ignore this sessions.
	if session.ID != "" {
		r.orders[session.ID] = session.Order
	}
}

func (r *ordersReducer) updateSessionStep(e models.CarTagEvent) {
	session, ok := r.sessions[e.ID]
	if !ok {
		return
	}
	// Check that event is not initial.
	if session.CurrentStep.Type != "" {
		// Sometimes there is too many equal steps.
		// Ignore change, if we face this situation.
		if session.CurrentStep.Type == string(e.Tag) {
			return
		}
		session.CurrentStep.FinishTime = e.HistoryTimestamp
		session.Steps = append(session.Steps, session.CurrentStep)
	}
	session.CurrentStep = Step{
		Type:      string(e.Tag),
		StartTime: e.HistoryTimestamp,
	}
}

const (
	StandartOfferType = "standart_offer"
	FixPointOfferType = "fix_point"
)

func (r *ordersReducer) startSession(e models.CarTagEvent) {
	// Sometimes previous session is not finished yet due to backend bug.
	r.finishSession(e)
	var data tags.ChargableTagData
	if err := e.ScanTagData(&data); err != nil {
		return
	}
	if data.Offer == nil || data.Offer.OfferId == nil {
		return
	}
	session := session{
		Order: Order{
			ID:        *data.Offer.OfferId,
			CarID:     e.ObjectID,
			UserID:    e.HistoryUserID,
			StartTime: e.HistoryTimestamp,
			OfferType: StandartOfferType,
			City:      "msc_area",
			StartCity: helpers.Moscow,
		},
	}
	updateOrderWithOffer(&session.Order, data.Offer)
	// Sometimes there is no snapshot.
	// Check that snapshot exists and get all available information.
	if snapshot, err := unmarshalDeviceSnapshot(e.CarTag); err == nil {
		if snapshot.Location != nil {
			if snapshot.Location.Latitude != nil {
				session.StartPointLatitude = *snapshot.Location.Latitude
			}
			if snapshot.Location.Longitude != nil {
				session.StartPointLongitude = *snapshot.Location.Longitude
			}
		}
		session.StartPtags = snapshot.TagInPoint
		session.StartCity = getSessionCity(
			session.StartPointLongitude,
			session.StartPointLatitude,
			session.StartPtags,
		)
		switch session.StartCity {
		case helpers.Moscow:
			session.City = "msc_area"
		case helpers.Petersburg:
			session.City = "spb_area"
		case helpers.Kazan:
			session.City = "kazan_area"
		case helpers.Sochi:
			session.City = "sochi_area"
		}
		for _, sensor := range snapshot.Sensor {
			if sensor.Id == nil {
				continue
			}
			switch *sensor.Id {
			case 2103:
				session.StartOdoMileage = getSensorFloat64(sensor)
			}
		}
	}
	// Register new session.
	r.sessions[e.ID] = &session
	// Update sessions step.
	r.updateSessionStep(e)
}

func unmarshalDeviceSnapshot(
	tag models.CarTag,
) (*bp.THistoryDeviceSnapshot, error) {
	bwh, err := tags.ParseBlobWithTagHeader(tag.Data)
	if err != nil {
		return &bp.THistoryDeviceSnapshot{}, err
	}
	if len(tag.Snapshot) == 0 {
		return &bp.THistoryDeviceSnapshot{}, nil
	}
	if bwh.Header.GetIsTagProto() {
		sbwh, err := tags.ParseBlobWithSnapshotHeader(string(tag.Snapshot))
		if err != nil {
			return &bp.THistoryDeviceSnapshot{}, err
		}
		if sbwh.Header.Type != nil {
			if *sbwh.Header.Type == "device_snapshot" {
				var snapshot bp.THistoryDeviceSnapshot
				if err := proto.Unmarshal(sbwh.Blob, &snapshot); err != nil {
					return &bp.THistoryDeviceSnapshot{}, err
				}
				return &snapshot, nil
			}
		}
	}
	return &bp.THistoryDeviceSnapshot{}, fmt.Errorf("unsupported snapshot")
}

type walletInfo struct {
	Name       string
	ParentName string
}

type enrichMapper struct {
	Parts       []int
	Areas       []api.Area
	WalletInfos map[string]walletInfo
}

func (m enrichMapper) InputTypes() []interface{} {
	return []interface{}{Order{}, models.CompiledBill{}}
}

func (m enrichMapper) OutputTypes() []interface{} {
	return []interface{}{enrichMapperRow{}}
}

type enrichMapperRow struct {
	Type      int           `yson:"type"`
	SessionID string        `yson:"session_id"`
	Row       yson.RawValue `yson:"row"`
}

type tmSpeedEvent struct {
	SessionID      string `yson:"session_id"`
	RidingDuration int64  `yson:"tm_riding_duration"`
}

type orderUserAgent struct {
	SessionID      string `yson:"session_id"`
	OSFamily       string `yson:"os_family"`
	DeviceIDAccept string `yson:"device_id_accept"`
}

type orderCostTotal struct {
	SessionID              string   `yson:"session_id"`
	CostPerMinuteOfRiding  *float64 `yson:"cost_per_minute_of_riding"`
	CostPerMinuteOfParking *float64 `yson:"cost_per_minute_of_parking"`
	CostPerMinuteOfLeasing *float64 `yson:"cost_per_minute_of_leasing"`
	Cost                   *float64 `yson:"cost"`
	Profit                 *float64 `yson:"profit"`
	DynamicCost            *float64 `yson:"dynamic_cost"`
	DynamicProfit          *float64 `yson:"dynamic_profit"`
	DynamicCostNew         *float64 `yson:"dynamic_cost_new"`
	DynamicProfitNew       *float64 `yson:"dynamic_profit_new"`
	LeasingDynamicCost     *float64 `yson:"leasing_dynamic_cost"`
}

type orderCarDistribution struct {
	SessionID   string   `yson:"session_id"`
	IsGoodOrder *int32   `yson:"car_distribution_is_good_order"`
	MseBefore   *float64 `yson:"car_distribution_mse_before"`
	MseAfter    *float64 `yson:"car_distribution_mse_after"`
	MseDiff     *float64 `yson:"car_distribution_mse_diff"`
}

func (m *enrichMapper) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		rawRow := enrichMapperRow{
			Type: m.Parts[in.TableIndex()],
		}
		switch rawRow.Type {
		case latestPart: // Order.
			order := Order{}
			if err := in.Scan(&order); err != nil {
				return err
			}
			startPoint := geom.Vec2{
				order.StartPointLongitude, order.StartPointLatitude,
			}
			finishPoint := geom.Vec2{
				order.FinishPointLongitude, order.FinishPointLatitude,
			}
			if len(order.SWDAFinishArea) > 0 {
				order.SWDAIsDiscounted = ptrBool(order.SWDAFinishArea.Contains(finishPoint))
			}
			for _, area := range m.Areas {
				if area.Polygon.Contains(startPoint) {
					order.StartAreas = append(order.StartAreas, Area{ID: area.ID})
				}
				if area.Polygon.Contains(finishPoint) {
					order.FinishAreas = append(order.FinishAreas, Area{ID: area.ID})
				}
			}
			if info, ok := m.WalletInfos[order.Wallet]; ok {
				order.ParentWallet = info.ParentName
			}
			var err error
			if rawRow.Row, err = yson.Marshal(order); err != nil {
				return err
			}
			rawRow.SessionID = order.ID
		case compiledBillsPart: // CompiledBill.
			bill := models.CompiledBill{}
			if err := in.Scan(&bill); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(bill); err != nil {
				return err
			}
			rawRow.SessionID = bill.SessionID
		case ordersTmStatsPart: // tmSpeedEvent.
			var row tmSpeedEvent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.SessionID = row.SessionID
		case ordersCostProfitPart: // orderCostTotal.
			var row orderCostTotal
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.SessionID = row.SessionID
		case ordersCarDistributionPart: // OrdersCarDistributionTable.
			var row orderCarDistribution
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.SessionID = row.SessionID
		case sessionsUserAgentPart: // orderUserAgent.
			var row orderUserAgent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.SessionID = row.SessionID
		}
		if err := out[0].Write(rawRow); err != nil {
			return err
		}
	}
	return nil
}

type enrichReducer struct{}

func (r enrichReducer) InputTypes() []interface{} {
	return []interface{}{enrichMapperRow{}}
}

func (r enrichReducer) OutputTypes() []interface{} {
	return []interface{}{Order{}}
}

func (r *enrichReducer) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out)
	})
}

const testWallet = "test_wallet"

var invalidOrders = map[string]struct{}{
	"4416b3e6-bdc640b-4db9a14a-743d1ae8":  struct{}{},
	"6c92b32d-5afd2b7f-aca4f87b-16646f92": struct{}{},
	"fbbbb9dd-7e31540b-480facf-3364da8d":  struct{}{},
}

func (r *enrichReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	var order Order
	var bills []models.CompiledBill
	var tmSpeed tmSpeedEvent
	var costTotal orderCostTotal
	var carDistribution orderCarDistribution
	var userAgent orderUserAgent
	for in.Next() {
		var rawRow enrichMapperRow
		if err := in.Scan(&rawRow); err != nil {
			return err
		}
		switch rawRow.Type {
		case latestPart: // Order.
			var orderRow Order
			if err := yson.Unmarshal(rawRow.Row, &orderRow); err != nil {
				return err
			}
			if order.ID == "" || len(orderRow.Steps) > 0 {
				order = orderRow
			}
		case compiledBillsPart: // CompiledBill.
			var bill models.CompiledBill
			if err := yson.Unmarshal(rawRow.Row, &bill); err != nil {
				return err
			}
			bills = append(bills, bill)
		case ordersTmStatsPart: // tmSpeedEvent.
			if err := yson.Unmarshal(rawRow.Row, &tmSpeed); err != nil {
				return err
			}
		case ordersCostProfitPart: // orderCostTotal.
			if err := yson.Unmarshal(rawRow.Row, &costTotal); err != nil {
				return err
			}
		case ordersCarDistributionPart: // OrdersCarDistributionTable.
			if err := yson.Unmarshal(rawRow.Row, &carDistribution); err != nil {
				return err
			}
		case sessionsUserAgentPart: // orderUserAgent.
			if err := yson.Unmarshal(rawRow.Row, &userAgent); err != nil {
				return err
			}
		}
	}
	if order.ID == "" {
		return nil
	}
	order.WalletTotals = map[string]float64{}
	accounts := map[uint32]struct{}{}
	for _, bill := range bills {
		if bill.SessionID == "" || bill.BillingType != "car_usage" {
			continue
		}
		details, err := bill.ParseDetails()
		if err != nil {
			return err
		}
		for _, item := range details.Items {
			if item.UniqueName != nil && item.Sum != nil {
				name := *item.UniqueName
				if order.Wallet == "" {
					order.Wallet = name
				}
				order.WalletTotals[name] += float64(*item.Sum) / 100
			}
			if item.AccountId != nil {
				if _, ok := accounts[*item.AccountId]; !ok {
					order.AccountIDs = append(order.AccountIDs, *item.AccountId)
					accounts[*item.AccountId] = struct{}{}
				}
			}
		}
		order.DebugInfo.HasCompiledBill = true
	}
	if tmSpeed.SessionID != "" {
		order.TmRidingDuration = tmSpeed.RidingDuration
		order.DebugInfo.HasTmStats = true
	}
	if val, ok := order.WalletTotals[testWallet]; ok && val > 100 {
		fixInvalidOrder(&order)
	} else if _, ok := invalidOrders[order.ID]; ok {
		fixInvalidOrder(&order)
	}
	if costTotal.SessionID != "" {
		if costTotal.CostPerMinuteOfRiding != nil {
			order.RidingCost = *costTotal.CostPerMinuteOfRiding
		}
		if costTotal.CostPerMinuteOfParking != nil {
			order.ParkingCost = *costTotal.CostPerMinuteOfParking
		}
		if costTotal.CostPerMinuteOfLeasing != nil {
			order.LeasingCost = *costTotal.CostPerMinuteOfLeasing
		}
		if costTotal.Cost != nil {
			order.CostTotal = *costTotal.Cost
		}
		if costTotal.Profit != nil {
			order.ProfitTotal = *costTotal.Profit
		}
		if costTotal.DynamicCost != nil {
			order.DynamicCostTotal = *costTotal.DynamicCost
		}
		if costTotal.DynamicCostNew != nil {
			order.DynamicCostNewTotal = *costTotal.DynamicCostNew
		}
		if costTotal.DynamicProfit != nil {
			order.DynamicProfitTotal = *costTotal.DynamicProfit
		}
		if costTotal.DynamicProfitNew != nil {
			order.DynamicProfitNewTotal = *costTotal.DynamicProfitNew
		}
		if costTotal.LeasingDynamicCost != nil {
			order.LeasingDynamicCostTotal = *costTotal.LeasingDynamicCost
		}
		order.DebugInfo.HasCostProfit = true
	}
	if carDistribution.SessionID != "" {
		order.CarDistributionIsGoodOrder = carDistribution.IsGoodOrder
		order.CarDistributionMseBefore = carDistribution.MseBefore
		order.CarDistributionMseAfter = carDistribution.MseAfter
		order.CarDistributionMseDiff = carDistribution.MseDiff
		order.DebugInfo.HasCarDistribution = true
	}
	if userAgent.SessionID != "" {
		order.OSFamily = userAgent.OSFamily
		order.DeviceIDAccept = userAgent.DeviceIDAccept
		order.DebugInfo.HasUserAgent = true
	}
	return out[0].Write(order)
}

func fixInvalidOrder(order *Order) {
	// Reset totals.
	order.Total = 0
	order.ReservationTotal = 0
	order.AcceptanceTotal = 0
	order.RidingTotal = 0
	order.ParkingTotal = 0
	order.DiscountTotal = 0
	order.OvertimeTotal = 0
	order.OverrunTotal = 0
	order.BonusTotal = 0
	// Reset prices.
	order.RidingPrice = 0
	order.ParkingPrice = 0
	order.PackPrice = 0
}

func getBillingAccountDescriptions(
	ctx *gotasks.Context, yc yt.Client,
) ([]models.BillingDescription, error) {
	in, err := yc.ReadTable(
		ctx.Context,
		ctx.Config.YTPaths.BillingAccountDescriptionsTable,
		nil,
	)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = in.Close()
	}()
	var rows []models.BillingDescription
	for in.Next() {
		var row models.BillingDescription
		if err := in.Scan(&row); err != nil {
			return nil, err
		}
		rows = append(rows, row)
	}
	return rows, in.Err()
}

func getWalletInfos(ctx *gotasks.Context, yc yt.Client) (map[string]walletInfo, error) {
	if len(ctx.Config.YTPaths.BillingAccountsTable.String()) == 0 {
		ctx.Logger.Warn("Disable wallet infos, no billing accounts")
		return map[string]walletInfo{}, nil
	}
	if len(ctx.Config.YTPaths.BillingAccountDescriptionsTable.String()) == 0 {
		ctx.Logger.Warn("Disable wallet infos, no billing account descriptions")
		return map[string]walletInfo{}, nil
	}
	descs, err := getBillingAccountDescriptions(ctx, yc)
	if err != nil {
		return nil, err
	}
	byID := map[int64]int{}
	byParent := map[int64]int{}
	for i, desc := range descs {
		meta, err := desc.GetMeta()
		if err != nil {
			return nil, err
		}
		byID[desc.ID] = i
		byParent[meta.ParentID] = i
	}
	in, err := yc.ReadTable(
		ctx.Context,
		ctx.Config.YTPaths.BillingAccountsTable,
		nil,
	)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = in.Close()
	}()
	infos := map[string]walletInfo{}
	for in.Next() {
		var row models.BillingAccount
		if err := in.Scan(&row); err != nil {
			return nil, err
		}
		if pos, ok := byParent[row.ID]; ok {
			if parentPos, ok := byID[row.TypeID]; ok {
				infos[descs[pos].Name] = walletInfo{
					Name:       descs[pos].Name,
					ParentName: descs[parentPos].Name,
				}
			}
		}
	}
	return infos, in.Close()
}
