package runsql

import (
	"fmt"
	"reflect"

	"github.com/spf13/cobra"

	bmodels "a.yandex-team.ru/drive/analytics/goback/models"
	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/gotasks/models/users"
	"a.yandex-team.ru/drive/library/go/gosql"
	rmodels "a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	obmodels "a.yandex-team.ru/zootopia/analytics/drive/models"
)

func init() {
	// Register subcommands.
	dumpModelCmd := cobra.Command{
		Use: "yt-dump-model",
		Run: gotasks.WrapMain(dumpModelMain),
	}
	dumpModelCmd.Flags().String("yt-proxy", "hahn", "YT proxy")
	dumpModelCmd.Flags().String("yt-path", "", "Path to YT table")
	dumpModelCmd.Flags().String("table", "", "Name of table")
	dumpModelCmd.Flags().String("model", "", "Name of model")
	SQLCmd.AddCommand(&dumpModelCmd)
}

type garageFieldValueView struct {
	ID            int64       `db:"id" yson:"id"`
	VIN           string      `db:"vin" yson:"vin"`
	FieldName     string      `db:"field_name" yson:"field_name"`
	Value         models.JSON `db:"value" yson:"value"`
	BeginTime     *int64      `db:"begin_time" yson:"begin_time"`
	EndTime       *int64      `db:"end_time" yson:"end_time"`
	CreateTime    int64       `db:"create_time" yson:"create_time"`
	CreateComment string      `db:"create_comment" yson:"create_comment"`
	CreateUserID  string      `db:"create_user_id" yson:"create_user_id"`
}

var modelTypes = map[string]interface{}{
	"garage_field_value_view":               garageFieldValueView{},
	"user_robot_state":                      users.RobotState{},
	"runner/planner":                        rmodels.Planner{},
	"runner/action":                         rmodels.Action{},
	"runner/secret":                         rmodels.Secret{},
	"runner/config":                         rmodels.Config{},
	"backend/trace_tag":                     bmodels.Tag{},
	"backend/car_document_event":            obmodels.CarDocumentEvent{},
	"backend/car_document_assignment_event": obmodels.CarDocumentAssignmentEvent{},
	"backend/car_event":                     obmodels.CarEvent{},
	"backend/billing_account":               obmodels.BillingAccount{},
	"backend/billing_account_description":   obmodels.BillingDescription{},
}

func dumpModelMain(ctx *gotasks.Context) (err error) {
	dbName, err := ctx.Cmd.Flags().GetString("db")
	if err != nil {
		return err
	}
	ytProxy, err := ctx.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return err
	}
	ytPath, err := ctx.Cmd.Flags().GetString("yt-path")
	if err != nil {
		return err
	}
	dbTable, err := ctx.Cmd.Flags().GetString("table")
	if err != nil {
		return err
	}
	modelName, err := ctx.Cmd.Flags().GetString("model")
	if err != nil {
		return err
	}
	if modelName == "" {
		modelName = dbTable
	}
	db, ok := ctx.DBs[dbName]
	if !ok {
		return fmt.Errorf("db %q does not exists", dbName)
	}
	yc, ok := ctx.YTs[ytProxy]
	if !ok {
		return fmt.Errorf("yt %q does not exists", ytProxy)
	}
	model, ok := modelTypes[modelName]
	if !ok {
		return fmt.Errorf("model %q does not exists", modelName)
	}
	modelCopy := reflect.New(reflect.TypeOf(model)).Interface()
	query, values := db.Select(dbTable).Names(gosql.StructNames(modelCopy)...).Build()
	rows, err := db.Query(query, values...)
	if err != nil {
		return err
	}
	defer func() {
		_ = rows.Close()
	}()
	ytSchema, err := schema.Infer(modelCopy)
	if err != nil {
		return err
	}
	path := ypath.NewRich(ytPath).SetSchema(ytSchema)
	opts := yt.CreateNodeOptions{
		IgnoreExisting: true,
		Attributes: map[string]interface{}{
			"optimize_for": "scan",
			"schema":       ytSchema,
		},
	}
	if _, err := yc.CreateNode(ctx.Context, path, yt.NodeTable, &opts); err != nil {
		return err
	}
	out, err := yc.WriteTable(ctx.Context, path, nil)
	if err != nil {
		return err
	}
	defer func() {
		if err != nil {
			_ = out.Rollback()
			return
		}
		err = out.Commit()
	}()
	for rows.Next() {
		modelCopy := reflect.New(reflect.TypeOf(model)).Interface()
		if err := rows.Scan(gosql.StructValues(modelCopy, true)...); err != nil {
			return err
		}
		if err := out.Write(modelCopy); err != nil {
			return err
		}
	}
	return rows.Err()
}
