package service

import (
	"database/sql"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"github.com/spf13/cobra"
)

func init() {
	// Register subcommands.
	ytStatsCmd := cobra.Command{
		Use: "yt-stats",
		Run: gotasks.WrapMain(ytStatsMain),
	}
	ytStatsCmd.Flags().String("yt", "hahn", "Name of YT connection")
	ytStatsCmd.Flags().String("db", "clickhouse", "Name of DB connection")
	ytStatsCmd.Flags().String("root", "//home/carsharing", "YT root path")
	ytStatsCmd.Flags().String(
		"table", "yt_hahn_carsharing_stats", "YT stats table",
	)
	ServiceCmd.AddCommand(&ytStatsCmd)
}

type ytNodeInfo struct {
	Value      interface{}     `yson:",value"`
	Type       yt.NodeType     `yson:"type,attr"`
	Stats      ytResourceUsage `yson:"resource_usage,attr"`
	RecStats   ytResourceUsage `yson:"recursive_resource_usage,attr"`
	UpdateTime time.Time       `yson:"modification_time,attr"`
	Codec      string          `yson:"compression_codec,attr"`
}

type ytResourceUsage struct {
	DiskSpace  int64 `yson:"disk_space"`
	ChunkCount int64 `yson:"chunk_count"`
	NodeCount  int64 `yson:"node_count"`
}

type ytNodeStats struct {
	Time  int64
	Path  string
	Stats ytResourceUsage
}

func ytStatsMain(ctx *gotasks.Context) error {
	ytName, err := ctx.Cmd.Flags().GetString("yt")
	if err != nil {
		return err
	}
	dbName, err := ctx.Cmd.Flags().GetString("db")
	if err != nil {
		return err
	}
	root, err := ctx.Cmd.Flags().GetString("root")
	if err != nil {
		return err
	}
	table, err := ctx.Cmd.Flags().GetString("table")
	if err != nil {
		return err
	}
	yc, ok := ctx.YTs[ytName]
	if !ok {
		return fmt.Errorf("yt %q does not exists", ytName)
	}
	db, ok := ctx.DBs[dbName]
	if !ok {
		return fmt.Errorf("db %q does not exists", dbName)
	}
	save := func(stats []ytNodeStats) error {
		if len(stats) == 0 {
			return nil
		}
		if err := gosql.WithTxContext(
			ctx.Context, db, nil, func(tx *sql.Tx) error {
				stmt, err := tx.Prepare(fmt.Sprintf(
					`INSERT INTO %q `+
						`("time", "path", "disk_space", `+
						`"chunk_count", "node_count") `+
						`VALUES (?, ?, ?, ?, ?)`,
					table,
				))
				if err != nil {
					return err
				}
				for _, row := range stats {
					if _, err := stmt.Exec(
						row.Time, row.Path, row.Stats.DiskSpace,
						row.Stats.ChunkCount, row.Stats.NodeCount,
					); err != nil {
						return err
					}
				}
				return nil
			},
		); err != nil {
			return err
		}
		return nil
	}
	var stats []ytNodeStats
	var visit func(path ypath.Path) error
	visit = func(path ypath.Path) error {
		ctx.Logger.Debug("Visit path", log.String("path", path.String()))
		var infos []ytNodeInfo
		err := yc.ListNode(ctx.Context, path, &infos, &yt.ListNodeOptions{
			Attributes: []string{"type", "recursive_resource_usage"},
		})
		if err != nil {
			return err
		}
		ts := time.Now().Unix()
		for _, info := range infos {
			child := path.Child(info.Value.(string))
			stats = append(stats, ytNodeStats{
				Time:  ts,
				Path:  child.String(),
				Stats: info.RecStats,
			})
			if len(stats) > 500 {
				if err := save(stats); err != nil {
					return err
				}
				stats = nil
			}
			if info.Type == yt.NodeMap {
				if err := visit(child); err != nil {
					return err
				}
			}
		}
		return nil
	}
	if err := visit(ypath.Path(root)); err != nil {
		return err
	}
	return save(stats)
}
