package taxidwh

import (
	"fmt"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/goback/models"
	"a.yandex-team.ru/drive/library/go/clients/taxidwh"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/events"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

func init() {
	exportCompiledBillsCmd := cobra.Command{
		Use: "export-compiled-bills",
		Run: deliveryMain(compiledBillsDelivery{}),
	}
	TaxiDWHCmd.AddCommand(&exportCompiledBillsCmd)
	ytExportCompiledBillsCmd := cobra.Command{
		Use: "export-compiled-bills",
		Run: ytDeliveryMain(compiledBillsDelivery{}),
	}
	YTTaxiDWHCmd.AddCommand(&ytExportCompiledBillsCmd)
}

type compiledBillsDelivery struct{}

func (d compiledBillsDelivery) RuleName() string {
	return "drive_analytics_compiled_bill"
}

func (d compiledBillsDelivery) EventStore(db *gosql.DB) events.ROStore {
	return events.NewStore(
		models.CompiledBill{}, "history_event_id", "compiled_bills",
		getDialect(db),
	)
}

func (d compiledBillsDelivery) EventDocument(
	event events.Event,
) (taxidwh.Document, error) {
	bill := event.(models.CompiledBill)
	data := map[string]interface{}{
		"history_event_id":      bill.HistoryEventID,
		"history_action":        bill.HistoryAction,
		"history_timestamp":     bill.HistoryTimestamp,
		"history_user_id":       bill.HistoryUserID,
		"history_originator_id": bill.HistoryOriginatorID,
		"history_comment":       bill.HistoryComment,
		"session_id":            bill.SessionID,
		"user_id":               bill.UserID,
		"bill":                  bill.Bill,
		"billing_type":          bill.BillingType,
	}
	billDetails, err := bill.ParseDetails()
	if err != nil {
		return taxidwh.Document{}, err
	}
	data["parsed_details"] = mongoFix(billDetails)
	return taxidwh.Document{
		ID:   fmt.Sprint(bill.HistoryEventID),
		Data: data,
	}, nil
}

func (d compiledBillsDelivery) ScanEvent(
	r goyt.TableReader,
) (events.Event, error) {
	var row models.CompiledBill
	if err := r.Scan(&row); err != nil {
		return nil, err
	}
	return row, nil
}
