package taxidwh

import (
	"fmt"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/goback/models"
	"a.yandex-team.ru/drive/library/go/clients/taxidwh"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/events"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

func init() {
	exportUsersCmd := cobra.Command{
		Use: "export-users",
		Run: deliveryMain(usersDelivery{}),
	}
	TaxiDWHCmd.AddCommand(&exportUsersCmd)
	exportUsersSnapshotCmd := cobra.Command{
		Use: "export-users-snapshot",
		Run: objectDeliveryMain(usersDelivery{}),
	}
	TaxiDWHCmd.AddCommand(&exportUsersSnapshotCmd)
}

type usersDelivery struct{}

func (d usersDelivery) RuleName() string {
	return "drive_analytics_user_wo_pd"
}

func (d usersDelivery) ObjectStore(db *gosql.DB) objects.ROStore {
	return objects.NewStore(
		models.User{}, "id", "user",
		getDialect(db),
	)
}

func (d usersDelivery) ObjectDocument(
	object objects.Object,
) (taxidwh.Document, error) {
	user := object.(models.User)
	data := map[string]interface{}{}
	data["id"] = user.ID
	data["uid"] = user.UID
	data["username"] = user.Username
	data["email"] = user.Email
	if user.Email == "" {
		data["email"] = nil
	}
	data["phone"] = user.Phone
	data["status"] = user.Status
	data["registered_at"] = user.RegisterTime
	data["registration_geo"] = user.RegisterGeo
	data["driving_license_ds_revision"] = user.LicenseRevision
	data["passport_ds_revision"] = user.PassportRevision
	data["updated_at"] = 0
	if ts := user.RegisterTime.Unix(); ts >= 0 {
		data["updated_at"] = ts
	}
	data["deleted_at"] = nil
	return taxidwh.Document{
		ID:   fmt.Sprint(user.ID),
		Data: data,
	}, nil
}

func (d usersDelivery) EventStore(db *gosql.DB) events.ROStore {
	return events.NewStore(
		models.UserEvent{}, "history_event_id", "drive_user_data_history",
		getDialect(db),
	)
}

func (d usersDelivery) EventDocument(
	event events.Event,
) (taxidwh.Document, error) {
	user := event.(models.UserEvent)
	document, err := d.ObjectDocument(user.User)
	if err != nil {
		return taxidwh.Document{}, err
	}
	data := document.Data.(map[string]interface{})
	if ts := user.EventTime().Unix(); ts >= 0 {
		data["updated_at"] = user.EventTime().Unix()
	}
	if user.EventAction() == "remove" {
		data["deleted_at"] = user.EventTime().Unix()
	}
	return taxidwh.Document{
		ID:   fmt.Sprint(user.ID),
		Data: data,
	}, nil
}

func (d usersDelivery) ScanEvent(
	r goyt.TableReader,
) (events.Event, error) {
	var row models.UserEvent
	if err := r.Scan(&row); err != nil {
		return nil, err
	}
	return row, nil
}
