package tracks

import (
	"bufio"
	"context"
	"fmt"
	"io"
	"os"
	"os/exec"
	"sync"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

type TracksWrapper struct {
	BinaryPath          string
	BeginTime           time.Time
	EndTime             time.Time
	InputTables         []ypath.Path
	ObjectIDsTable      ypath.Path
	SessionIDsTable     ypath.Path
	SortedTable         ypath.Path
	Config              config.Tracks
	Logger              log.Logger
	CarTagsHistoryTable ypath.Path
}

func (w *TracksWrapper) streamLogs(group *sync.WaitGroup, reader io.Reader) {
	defer group.Done()
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		w.Logger.Info(scanner.Text())
	}
	if err := scanner.Err(); err != nil {
		w.Logger.Error("Scanner error", log.Error(err))
	}
}

func (w *TracksWrapper) Exec(ctx context.Context) error {
	args := []string{
		"--begin_ts", fmt.Sprint(w.BeginTime.Unix()),
		"--end_ts", fmt.Sprint(w.EndTime.Unix()),
		"--ydbEndpoint", w.Config.YDBEndpoint,
		"--ydbDatabase", w.Config.YDBDatabase,
		"--ydbTable", w.Config.YDBTable,
		"--objectIds", w.ObjectIDsTable.String(),
		"--sessionIds", w.SessionIDsTable.String(),
		"--sorted", w.SortedTable.String(),
		"--history", w.CarTagsHistoryTable.String(),
		"--matched", w.Config.TracksTable + "-" + w.BeginTime.Format("2006-01-02"),
		"--matcher", w.Config.MatcherConfigPath,
		"--filter", w.Config.FilterConfigPath,
		"--needMatching",
		"--useYDB",
	}
	for _, table := range w.InputTables {
		args = append(args, "--input", table.String())
	}
	cmd := exec.CommandContext(ctx, w.BinaryPath, args...)
	cmd.Env = os.Environ()
	var group sync.WaitGroup
	defer group.Wait()
	stderr, err := cmd.StderrPipe()
	if err != nil {
		return err
	}
	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return err
	}
	group.Add(2)
	if err := cmd.Start(); err != nil {
		return err
	}
	go w.streamLogs(&group, stderr)
	go w.streamLogs(&group, stdout)
	return cmd.Wait()
}

func init() {
	// Register subcommands.
	buildYDBCmd := cobra.Command{
		Use: "build-ydb",
		Run: gotasks.WrapMain(buildYDBMain),
	}
	buildYDBCmd.Flags().String("date", "", "")
	buildYDBCmd.Flags().Int("days", 2, "")
	buildYDBCmd.Flags().Int("safe-days", 1, "")
	buildYDBCmd.Flags().String("yt-proxy", "hahn", "Name of YT connection")
	TracksCmd.AddCommand(&buildYDBCmd)
}

func getDayStart(ts time.Time) time.Time {
	return time.Date(ts.Year(), ts.Month(), ts.Day(), 0, 0, 0, 0, ts.Location())
}

const dayDuration = time.Hour * 24

func buildYDBMain(ctx *gotasks.Context) error {
	date, err := ctx.Cmd.Flags().GetString("date")
	if err != nil {
		return err
	}
	days, err := ctx.Cmd.Flags().GetInt("days")
	if err != nil {
		return err
	}
	safeDays, err := ctx.Cmd.Flags().GetInt("safe-days")
	if err != nil {
		return err
	}
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	endTime := time.Now()
	if date != "" {
		ts, err := time.Parse("2006-01-02", date)
		if err != nil {
			return err
		}
		endTime = ts
	}
	endTime = getDayStart(endTime)
	beginTime := endTime.Add(-dayDuration * time.Duration(days))
	safeBeginTime := beginTime.Add(-dayDuration * time.Duration(safeDays))
	ctx.Logger.Info(
		"Building tracks for period",
		log.Time("safe_begin_time", safeBeginTime),
		log.Time("begin_time", beginTime),
		log.Time("end_time", endTime),
	)
	tableNames, err := goyt.ListTableNames(yc, ctx.Config.YTPaths.TelematicsLog1dDir)
	if err != nil {
		return err
	}
	var inputs []ypath.Path
	var safeInputs []ypath.Path
	for _, tableName := range tableNames {
		tableTime, err := time.Parse("2006-01-02", tableName)
		if err != nil {
			ctx.Logger.Warn(
				"Unable to parse date in table",
				log.String("table_name", tableName),
				log.Error(err),
			)
			continue
		}
		if !tableTime.Before(endTime) {
			continue
		}
		if !tableTime.Before(beginTime) {
			inputs = append(
				inputs,
				ctx.Config.YTPaths.TelematicsLog1dDir.Child(tableName),
			)
		} else if !tableTime.Before(safeBeginTime) {
			safeInputs = append(
				safeInputs,
				ctx.Config.YTPaths.TelematicsLog1dDir.Child(tableName),
			)
		}
	}
	if len(inputs) != days {
		return fmt.Errorf("expected %d tables but found %d", days, len(inputs))
	}
	tempDir := ctx.Config.YTPaths.TempDir
	sortedTable, err := goyt.TempTable(ctx.Context, yc, tempDir)
	if err != nil {
		return err
	}
	objectIDsTable, err := goyt.TempTable(ctx.Context, yc, tempDir)
	if err != nil {
		return err
	}
	sessionIDsTable, err := goyt.TempTable(ctx.Context, yc, tempDir)
	if err != nil {
		return err
	}
	inputs = append(inputs, safeInputs...)
	wrapper := TracksWrapper{
		BinaryPath:          ctx.Config.Tracks.BinaryPath,
		Logger:              ctx.Logger,
		BeginTime:           beginTime,
		EndTime:             endTime,
		InputTables:         inputs,
		ObjectIDsTable:      objectIDsTable,
		SessionIDsTable:     sessionIDsTable,
		Config:              *ctx.Config.Tracks,
		CarTagsHistoryTable: ctx.Config.YTPaths.CarTagsHistoryTable,
		SortedTable:         sortedTable,
	}
	return wrapper.Exec(ctx.Context)
}
