package users

import (
	"encoding/json"
	"fmt"
	"sort"
	"strings"
	"time"

	"github.com/dchest/siphash"
	"github.com/gofrs/uuid"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/gotasks/models/users"
	"a.yandex-team.ru/drive/analytics/gotasks/orders"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
	"a.yandex-team.ru/zootopia/analytics/drive/helpers"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/goyt"

	bmt "a.yandex-team.ru/drive/analytics/goback/models/tags"
)

func init() {
	updateExtendedCmd := cobra.Command{
		Use:  "update-extended",
		Long: "Update 'user/extended' YT table",
		Run:  gotasks.WrapMain(updateExtendedMain),
	}
	updateExtendedCmd.Flags().String("yt-proxy", "hahn", "YT proxy")
	updateExtendedCmd.Flags().String("drive", "prestable", "Name of Drive client")
	UsersCmd.AddCommand(&updateExtendedCmd)
	mapreduce.Register(&extendedUserMapper{})
	mapreduce.Register(&extendedUserReducer{})
}

const (
	userPart = iota
	userAppsPart
	userTagsHistoryPart
	userOrdersPart
	userParamsPart
	userFeaturesPart
	userPlusPart
	userLandingsPart
	userSocialProfilesPart
	userTagsPart
	userRolesPart
	userLastActivePart
	userDevicesPart
	userDocumentPhotosPart
	userRenisKBMPart
	userLicenseCachePart
	userPassportCachePart
)

func updateExtendedMain(ctx *gotasks.Context) (errMain error) {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	driveName, err := ctx.Cmd.Flags().GetString("drive")
	if err != nil {
		return err
	}
	drive, ok := ctx.Drives[driveName]
	if !ok {
		return fmt.Errorf("invalid drive name %q", driveName)
	}
	tags, err := drive.GetTagDescriptions()
	if err != nil {
		return err
	}
	tagDescriptions := map[string]api.TagDescription{}
	for _, tag := range tags {
		tagDescriptions[tag.Tag] = tag
	}
	var tables []ypath.YPath
	var tableParts []int
	addPart := func(path ypath.Path, kind int) {
		if len(path.String()) == 0 {
			ctx.Logger.Warn(
				"Table part disabled",
				log.String("path", path.String()),
				log.Int("kind", kind),
			)
			return
		}
		tables = append(tables, path)
		tableParts = append(tableParts, kind)
	}
	addDirPart := func(dir ypath.Path, kind int) error {
		if len(dir.String()) == 0 {
			ctx.Logger.Warn(
				"Table part disabled",
				log.String("path", dir.String()),
				log.Int("kind", kind),
			)
			return nil
		}
		paramNodes, err := goyt.ListDir(ctx.Context, yc, dir)
		if err != nil {
			return err
		}
		for _, node := range paramNodes {
			if node.Type == yt.NodeTable {
				addPart(dir.Child(node.Name), kind)
			}
		}
		return nil
	}
	addPart(ctx.Config.YTPaths.SimpleUsersTable, userPart)
	addPart(ctx.Config.YTPaths.UserAppsTable, userAppsPart)
	addPart(ctx.Config.YTPaths.UserTagsHistoryTable, userTagsHistoryPart)
	addPart(ctx.Config.YTPaths.LatestOrdersTable, userOrdersPart)
	addPart(ctx.Config.YTPaths.UserLandingsTable, userLandingsPart)
	addPart(ctx.Config.YTPaths.UserPlusTable, userPlusPart)
	addPart(ctx.Config.YTPaths.UserTagsTable, userTagsPart)
	addPart(ctx.Config.YTPaths.UserRolesTable, userRolesPart)
	addPart(ctx.Config.YTPaths.UserLastActiveTable, userLastActivePart)
	addPart(ctx.Config.YTPaths.UserDevicesTable, userDevicesPart)
	addPart(ctx.Config.YTPaths.UserDocumentPhotosTable, userDocumentPhotosPart)
	addPart(ctx.Config.YTPaths.UserRenisKBMTable, userRenisKBMPart)
	addPart(ctx.Config.YTPaths.UserSocialProfilesTable, userSocialProfilesPart)
	addPart(ypath.Path(ctx.Config.Exports.LicenseCacheTable), userLicenseCachePart)
	addPart(ypath.Path(ctx.Config.Exports.PassportCacheTable), userPassportCachePart)
	if err := addDirPart(ctx.Config.YTPaths.UserParamsDir, userParamsPart); err != nil {
		return err
	}
	if err := addDirPart(ctx.Config.YTPaths.UserFeaturesDir, userFeaturesPart); err != nil {
		return err
	}
	outputTable := ctx.Config.YTPaths.ExtendedUsersTable
	tx, err := yc.BeginTx(ctx.Context, nil)
	if err != nil {
		return err
	}
	defer func() {
		if errMain == nil {
			errMain = tx.Commit()
			return
		}
		_ = tx.Abort()
	}()
	opSpec := spec.Spec{
		InputTablePaths: tables,
		OutputTablePaths: []ypath.YPath{
			outputTable.Rich().
				SetSchema(schema.MustInfer(users.ExtendedUser{})),
		},
		ReduceBy: []string{"user_id"},
		SortBy:   []string{"user_id", "type"},
		Pool:     "carsharing",
	}
	now := time.Now().Unix()
	mr := mapreduce.New(yc).WithTx(tx)
	op, err := mr.MapReduce(
		&extendedUserMapper{Parts: tableParts, NowTime: now},
		&extendedUserReducer{
			NowTime:         now,
			TagDescriptions: tagDescriptions,
		},
		opSpec.MapReduce(),
	)
	if err != nil {
		return err
	}
	if err := op.Wait(); err != nil {
		return err
	}
	sortOpSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{outputTable},
		OutputTablePath: outputTable,
		SortBy:          []string{"user_id"},
		Pool:            "carsharing",
	}
	sortOp, err := mr.Sort(sortOpSpec.Sort())
	if err != nil {
		return err
	}
	return sortOp.Wait()
}

func ptrInt64(v int64) *int64 {
	return &v
}

func ptrFloat64(v float64) *float64 {
	return &v
}

const daySeconds = 24 * 60 * 60
const monthSeconds = daySeconds * 30

type userLandingsRow struct {
	LandingID string    `yson:"landing_id"`
	UserID    uuid.UUID `yson:"user_id"`
}

type userLastActiveRow struct {
	UserID string `yson:"user_id"`
	Time   uint64 `yson:"ts"`
	IsPlus bool   `yson:"is_plus"`
}

type extendedUserMapperRow struct {
	UserID string        `yson:"user_id"`
	Type   int           `yson:"type"`
	Row    yson.RawValue `yson:"data"`
}

type extendedUserMapper struct {
	Parts   []int
	NowTime int64
}

func (e *extendedUserMapper) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		rawRow := extendedUserMapperRow{
			Type: e.Parts[in.TableIndex()],
		}
		switch rawRow.Type {
		case userPart:
			var row users.User
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userTagsHistoryPart:
			var row models.TagEvent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.TargetID.String()
		case userAppsPart:
			var row users.UserAppsRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userOrdersPart:
			var row orders.Order
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID
		case userParamsPart:
			var row map[string]interface{}
			if err := in.Scan(&row); err != nil {
				return err
			}
			userIDStr, ok := row["user_id"].(string)
			if !ok {
				continue
			}
			delete(row, "user_id")
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = userIDStr
		case userFeaturesPart:
			var row map[string]interface{}
			if err := in.Scan(&row); err != nil {
				return err
			}
			userIDStr, ok := row["user_id"].(string)
			if !ok {
				continue
			}
			delete(row, "user_id")
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = userIDStr
		case userPlusPart:
			var row users.PlusRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userSocialProfilesPart:
			var row users.SocialProfilesRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.Type = userSocialProfilesPart
			rawRow.UserID = row.UserID.String()
		case userLandingsPart:
			var row userLandingsRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userTagsPart:
			var row models.Tag
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.TargetID.String()
		case userRolesPart:
			var row models.UserRole
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userDevicesPart:
			var row models.UserDevice
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userLastActivePart:
			var row userLastActiveRow
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID
		case userDocumentPhotosPart:
			var row models.UserDocumentPhoto
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userRenisKBMPart:
			var row users.RenisKBM
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userLicenseCachePart:
			var row models.UserLicense
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		case userPassportCachePart:
			var row models.UserPassport
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.UserID = row.UserID.String()
		}
		if err := out[0].Write(rawRow); err != nil {
			return err
		}
	}
	return nil
}

var knownApps = map[string]string{
	"delimobil":            "delimobil",
	"com.carshering":       "delimobil",
	"belkacar":             "belkacar",
	"ru.belkacar.belkacar": "belkacar",
	"youdrive.today":       "youdrive",
	"today.youdrive.lite":  "youdrive",
	"today.youdrive.yd4b":  "youdrive",
	"ru.matreshcar.app":    "matreshcar",
	"com.pkdeveloper.carsharingwhitelabel.carusel.club": "carusel",
}

func (e extendedUserMapper) InputTypes() []interface{} {
	return []interface{}{
		users.User{}, users.UserAppsRow{}, models.TagEvent{},
		orders.Order{}, users.SocialProfilesRow{}, users.PlusRow{},
		userLandingsRow{},
	}
}

func (e extendedUserMapper) OutputTypes() []interface{} {
	return []interface{}{extendedUserMapperRow{}}
}

func extractDevicePlatform(description string) string {
	var platform struct {
		Headers struct {
			ICAppBuild string `json:"IC_AppBuild"`
			ACAppBuild string `json:"AC_AppBuild"`
		} `json:"headers"`
	}
	if err := json.Unmarshal([]byte(description), &platform); err != nil {
		return "?"
	}
	if platform.Headers.ICAppBuild != "" {
		return "i"
	}
	if platform.Headers.ACAppBuild != "" {
		return "a"
	}
	return "?"
}

func hashDeviceID(deviceID string) uint64 {
	var key [16]byte
	h := siphash.New(key[:])
	_, _ = h.Write([]byte(deviceID))
	return h.Sum64()
}

func getYearDay(t int64) int {
	_, m, d := time.Unix(t, 0).UTC().Date()
	return int(m)*31 + d
}

const oneYearSeconds = 365.25 * 24 * 60 * 60

type advTokenSorter []users.AdvToken

func (a advTokenSorter) Len() int {
	return len(a)
}

func (a advTokenSorter) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a advTokenSorter) Less(i, j int) bool {
	return a[i].Time > a[j].Time
}

type extendedUserReducer struct {
	NowTime int64
	// TagDescriptions contains descriptions for all tags.
	TagDescriptions map[string]api.TagDescription
}

func (e extendedUserReducer) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return e.reduceGroup(in, out)
	})
}

func (e *extendedUserReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	var user users.ExtendedUser
	apps := map[string]struct{}{}
	carsharingApps := map[string]struct{}{}
	shownLandings := map[string]struct{}{}
	tagsCount := map[string]int{}
	tagsCount1Month := map[string]int{}
	params := map[string]interface{}{}
	features := map[string]interface{}{}
	bbTags1Month := 0
	refundTags1Month := 0
	birthday := false
	cityOrderCounts := map[string]int{}
	walletTotals := map[string]float64{}
	orderNameCounts := map[string]int{}
	orderTypeCounts := map[string]int{}
	vehicleSearchTimes := map[string]int64{}
	licensePhotosFlags := 0
	passportPhotosFlags := 0
	var activeRoles []string
	var passiveRoles []string
	var tags []string
	var userTags []users.Tag
	for in.Next() {
		var rawRow extendedUserMapperRow
		if err := in.Scan(&rawRow); err != nil {
			return err
		}
		switch rawRow.Type {
		case userPart:
			var row users.User
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.User = row
		case userAppsPart:
			var row users.UserAppsRow
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			for _, app := range row.Apps {
				apps[app] = struct{}{}
				if name, ok := knownApps[app]; ok {
					carsharingApps[name] = struct{}{}
				}
			}
			for _, ts := range row.TaxiAcceptedSurgeTimes {
				if user.LastTaxiAcceptedSurgeTime == nil ||
					*user.LastTaxiAcceptedSurgeTime < ts {
					user.LastTaxiAcceptedSurgeTime = ptrInt64(ts)
				}
			}
			for _, event := range row.AutoVehicleSearchEvents {
				ts := vehicleSearchTimes[event.Nameplate]
				if ts < event.Time {
					vehicleSearchTimes[event.Nameplate] = event.Time
				}
			}
		case userTagsHistoryPart:
			var row models.TagEvent
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			tagName := string(row.Tag.Tag)
			if row.HistoryAction == models.AddAction {
				tagsCount[tagName]++
				if row.HistoryTimestamp+monthSeconds >= e.NowTime {
					tagsCount1Month[tagName]++
					if strings.HasPrefix(tagName, "bb_") {
						bbTags1Month++
					} else if strings.HasPrefix(tagName, "refund_") {
						refundTags1Month++
					}
				}
			}
		case userOrdersPart:
			var row orders.Order
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			if user.FirstOrderTime == nil ||
				*user.FirstOrderTime > row.FinishTime {
				user.FirstOrderTime = ptrInt64(row.FinishTime)
			}
			if user.LastOrderTime == nil ||
				*user.LastOrderTime < row.FinishTime {
				user.LastOrderTime = ptrInt64(row.FinishTime)
			}
			user.OrdersCount++
			if row.Total > 0 {
				if user.FirstPaidOrderTime == nil ||
					*user.FirstPaidOrderTime > row.FinishTime {
					user.FirstPaidOrderTime = ptrInt64(row.FinishTime)
				}
				if user.LastPaidOrderTime == nil ||
					*user.LastPaidOrderTime < row.FinishTime {
					user.LastPaidOrderTime = ptrInt64(row.FinishTime)
				}
				user.PaidOrdersCount++
			}
			for name, total := range row.WalletTotals {
				walletTotals[name] += total
			}
			if row.FinishCity != "" {
				cityOrderCounts[row.FinishCity]++
			}
			orderNameCounts[row.TariffName]++
			orderTypeCounts[row.OfferType]++
		case userParamsPart:
			var row map[string]interface{}
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			for name, value := range row {
				params[name] = value
			}
		case userFeaturesPart:
			var row map[string]interface{}
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			for name, value := range row {
				features[name] = value
			}
		case userLandingsPart:
			var row userLandingsRow
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			shownLandings[row.LandingID] = struct{}{}
		case userPlusPart:
			var row users.PlusRow
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			if row.PlusBeginTime != nil {
				user.PlusBeginTime = ptrInt64(*row.PlusBeginTime)
			}
			if row.PlusEndTime != nil {
				user.IsPlus = e.NowTime < *row.PlusEndTime
				user.PlusEndTime = ptrInt64(*row.PlusEndTime)
			}
			user.PlusSubs = row.PlusSubs
		case userSocialProfilesPart:
			var row users.SocialProfilesRow
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.SocialProfiles = row.Profiles
		case userTagsPart:
			var row models.Tag
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			tags = append(tags, string(row.Tag))
			if row.Tag == "user_app_settings" {
				var data bmt.UserDictionaryTagData
				if err := row.ScanData(&data); err != nil {
					continue
				}
				user.AppSettings = map[string]string{}
				for _, field := range data.Fields {
					user.AppSettings[*field.Key] = *field.Value
				}
			}
			userTag := users.Tag{Tag: string(row.Tag)}
			if description, ok := e.TagDescriptions[string(row.Tag)]; ok {
				data, err := row.ParseData(description.Type)
				if err == nil {
					userTag.Data = data
				}
			}
			userTags = append(userTags, userTag)
		case userRolesPart:
			var row models.UserRole
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			if row.Active {
				activeRoles = append(activeRoles, row.RoleID)
			} else {
				passiveRoles = append(passiveRoles, row.RoleID)
			}
		case userDevicesPart:
			var row models.UserDevice
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			if row.Enabled && row.Verified {
				if user.LastDeviceID < row.DeviceID {
					user.LastDeviceID = row.DeviceID
					user.LastDeviceIDHash = hashDeviceID(row.DeviceID)
					user.LastPlatform = extractDevicePlatform(
						string(row.Description),
					)
				}
			}
			user.DeviceIDs = append(user.DeviceIDs, row.DeviceID)
			user.DeviceIDHashes = append(
				user.DeviceIDHashes, hashDeviceID(row.DeviceID),
			)
			user.Platforms += extractDevicePlatform(string(row.Description))
			user.AdvTokens = append(user.AdvTokens, users.AdvToken{
				Token: string(row.AdvToken),
				Type:  string(row.AdvTokenType),
				Time:  int64(row.AdvTokenTime),
			})
		case userLastActivePart:
			var row userLastActiveRow
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.LastActiveTime = ptrInt64(int64(row.Time))
		case userDocumentPhotosPart:
			var row models.UserDocumentPhoto
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			if row.Status != "o" {
				continue
			}
			switch row.Type {
			case "lf":
				licensePhotosFlags |= 1
			case "lb":
				licensePhotosFlags |= 2
			case "pb":
				passportPhotosFlags |= 1
			case "pr":
				passportPhotosFlags |= 2
			case "ps":
				passportPhotosFlags |= 4
			}
		case userRenisKBMPart:
			var row users.RenisKBM
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.RenisKBM = ptrFloat64(row.KBM)
		case userLicenseCachePart:
			var row models.UserLicense
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.LicenseNumber = row.Number
			user.LicenseNumberFront = row.NumberFront
			user.LicenseNumberBack = row.NumberBack
			user.LicenseFirstName = row.FirstNameHash
			user.LicenseLastName = row.LastNameHash
			user.LicenseMiddleName = row.MiddleNameHash
			user.LicenseCountry = row.Country
			user.LicenseCountryFront = row.CountryFront
			user.LicenseCountryBack = row.CountryBack
			user.LicenseIssueDate = toUnix(row.IssueDate)
			user.LicenseFirstIssueDate = toUnix(row.FirstIssueDate)
			user.LicenseExpireDate = toUnix(row.ExpireDate)
			user.LicenseBirthDate = toUnix(row.BirthDate)
		case userPassportCachePart:
			var row models.UserPassport
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			user.PassportNumber = row.NumberHash
			user.PassportFirstName = row.FirstNameHash
			user.PassportLastName = row.LastNameHash
			user.PassportMiddleName = row.MiddleNameHash
			user.PassportGender = fixGender(row.Gender)
			user.PassportBirthPlace = row.BirthPlace
			user.PassportCitizenship = row.Citizenship
			user.PassportBioCountry = row.BioCountry
			user.PassportRegCountry = row.RegCountry
			user.PassportRegApartment = row.RegApartment
			user.PassportRegHousing = row.RegHousing
			user.PassportRegLetter = row.RegLetter
			user.PassportRegHouse = row.RegHouse
			user.PassportRegStreet = row.RegStreet
			user.PassportRegArea = row.RegArea
			user.PassportRegLocality = row.RegLocality
			user.PassportRegRegion = row.RegRegion
			user.PassportRegType = row.RegType
			user.PassportBirthDate = toUnix(row.BirthDate)
			user.PassportIssueDate = toUnix(row.IssueDate)
			user.PassportExpireDate = toUnix(row.ExpireDate)
			user.PassportRegExpireDate = toUnix(row.RegExpireDate)
		}
	}
	if user.UserID == uuid.Nil {
		return nil
	}
	for app := range apps {
		user.Apps = append(user.Apps, app)
	}
	for app := range carsharingApps {
		user.CarsharingApps = append(user.CarsharingApps, app)
	}
	for landing := range shownLandings {
		user.ShownLandings = append(user.ShownLandings, landing)
	}
	if user.PassportBirthDate != nil {
		t := time.Unix(*user.PassportBirthDate, 0)
		ts := time.Since(t).Seconds() / oneYearSeconds
		user.Age = &ts
		if getYearDay(*user.PassportBirthDate) == getYearDay(e.NowTime) {
			birthday = true
		}
	}
	if user.LicenseBirthDate != nil {
		t := time.Unix(*user.LicenseBirthDate, 0)
		ts := time.Since(t).Seconds() / oneYearSeconds
		user.LicenseAge = &ts
	}
	if user.LicenseFirstIssueDate != nil {
		t := time.Unix(*user.LicenseFirstIssueDate, 0)
		ts := time.Since(t).Seconds() / oneYearSeconds
		user.Experience = &ts
	}
	params["bb_tags_1month"] = bbTags1Month
	params["refund_tags_1month"] = refundTags1Month
	params["birthday"] = birthday
	user.Features = features
	user.Params = params
	user.TagsCount = tagsCount
	user.TagsCount1Month = tagsCount1Month
	user.CityOrderCounts = cityOrderCounts
	user.WalletTotals = walletTotals
	user.OrderNameCounts = orderNameCounts
	user.OrderTypeCounts = orderTypeCounts
	user.LastAutoVehicleSearchTimes = vehicleSearchTimes
	user.ActiveRoles = activeRoles
	user.PassiveRoles = passiveRoles
	user.Tags = tags
	user.UserTags = userTags
	user.GoodLicensePhotos = licensePhotosFlags == 3
	user.GoodPassportPhotos = passportPhotosFlags == 7
	switch user.RegisterDBGeo {
	case "spb":
		user.CurrentCity = helpers.Petersburg
	case "kazan":
		user.CurrentCity = helpers.Kazan
	case "sochi":
		user.CurrentCity = helpers.Sochi
	default:
		user.CurrentCity = helpers.Moscow
	}
	for city, count := range user.CityOrderCounts {
		if count > user.CityOrderCounts[user.CurrentCity] {
			user.CurrentCity = city
		}
	}
	sort.Sort(advTokenSorter(user.AdvTokens))
	return out[0].Write(user)
}

func (e extendedUserReducer) InputTypes() []interface{} {
	return []interface{}{extendedUserMapperRow{}}
}

func (e extendedUserReducer) OutputTypes() []interface{} {
	return []interface{}{users.ExtendedUser{}}
}

func fixGender(gender string) string {
	switch gender {
	case "FEMALE", "F", "ЖЕН", "Ж":
		return "f"
	case "MALE", "M", "МУЖ", "М":
		return "m"
	}
	return gender
}

func toUnix(t time.Time) *int64 {
	if t.IsZero() {
		return nil
	}
	ts := t.Unix()
	return &ts
}
