package users

import (
	"context"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptrace"
	"net/url"
	"os/exec"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/services/licensechecks"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
)

func init() {
	fetchLicenseChecksCmd := cobra.Command{
		Use: "fetch-license-checks",
		Run: gotasks.WrapMain(fetchLicenseChecksMain),
	}
	fetchLicenseChecksCmd.Flags().Int("limit", 500, "Limit of users to load")
	UsersCmd.AddCommand(&fetchLicenseChecksCmd)
	processLicenseChecksCmd := cobra.Command{
		Use: "process-license-checks",
		Run: gotasks.WrapMain(processLicenseChecksMain),
	}
	processLicenseChecksCmd.Flags().Int("limit", 100, "Limit of tasks to process")
	processLicenseChecksCmd.Flags().String("model", "./gibddcaptcha", "Path to Tensorflow model")
	processLicenseChecksCmd.Flags().String("db", "analytics", "DB name")
	UsersCmd.AddCommand(&processLicenseChecksCmd)
}

func trySendUserLicenseCheck(
	ctx *gotasks.Context, client *licensechecks.Client, userID string,
) (licensechecks.Task, error) {
	ctx.Logger.Debug("Processing user", log.String("user_id", userID))
	tags, err := ctx.Drive.GetUserTags(userID)
	if err != nil {
		ctx.Logger.Error(
			"Unable to get user tags",
			log.String("user_id", userID),
			log.Error(err),
		)
		ctx.Signal(
			"user_license_check_error",
			map[string]string{"type": "fetch_tags_error"},
		).Add(1)
		return licensechecks.Task{}, err
	}
	var checkTag api.UserTag
	var taskTag api.UserTag
	for _, tag := range tags {
		switch tag.Tag {
		case "user_license_check":
			checkTag = tag
		case "user_license_check_task":
			taskTag = tag
		}
	}
	if checkTag.Tag == "" {
		ctx.Logger.Error(
			"Unable to find check tag",
			log.String("user_id", userID),
		)
		return licensechecks.Task{}, fmt.Errorf("user does not need check")
	}
	if taskTag.Tag != "" {
		ctx.Logger.Debug("Skipping user", log.String("user_id", userID))
		if err := ctx.Drive.RemoveUserTag(checkTag.ID); err != nil {
			ctx.Logger.Error(
				"Unable to remove user tag",
				log.String("user_id", userID),
				log.String("tag_id", checkTag.ID),
				log.Error(err),
			)
			return licensechecks.Task{}, err
		}
		ctx.Logger.Debug("Removed check tag", log.String("user_id", userID))
		return licensechecks.Task{}, fmt.Errorf("user already sent")
	}
	user, err := ctx.Drive.GetUser(userID)
	if err != nil {
		ctx.Logger.Error(
			"Unable to get user",
			log.String("user_id", userID),
			log.Error(err),
		)
		ctx.Signal(
			"user_license_check_error",
			map[string]string{"type": "fetch_user_error"},
		).Add(1)
		return licensechecks.Task{}, err
	}
	if user.License == nil {
		return licensechecks.Task{}, fmt.Errorf("user license does not exists")
	}
	callbackData, err := json.Marshal(userID)
	if err != nil {
		return licensechecks.Task{}, err
	}
	task, err := client.Request(licensechecks.RequestForm{
		Callback:         "drive",
		CallbackData:     models.JSON(callbackData),
		LicenseNumber:    user.License.FrontNumber,
		LicenseIssueDate: user.License.IssueDate.Format("2006-01-02"),
	})
	if err != nil {
		return licensechecks.Task{}, err
	}
	ctx.Logger.Debug("Created task", log.Any("task", task))
	if err := ctx.Drive.AddUserTag(api.UserTag{
		UserID: userID,
		Tag:    "user_license_check_task",
		Data: &api.SimpleUserTagData{
			Comment: fmt.Sprintf("ID проверки = %d", task.ID),
		},
	}); err != nil {
		ctx.Logger.Error(
			"Unable to add user tag",
			log.String("user_id", userID),
			log.Error(err),
		)
		ctx.Signal(
			"user_license_check_error",
			map[string]string{"type": "add_tag_error"},
		).Add(1)
		return licensechecks.Task{}, err
	}
	ctx.Logger.Debug("Added task tag", log.String("user_id", userID))
	if err := ctx.Drive.RemoveUserTag(checkTag.ID); err != nil {
		ctx.Logger.Error(
			"Unable to remove user tag",
			log.String("user_id", userID),
			log.String("tag_id", checkTag.ID),
			log.Error(err),
		)
		ctx.Signal(
			"user_license_check_error",
			map[string]string{"type": "remove_tag_error"},
		).Add(1)
		return licensechecks.Task{}, err
	}
	ctx.Logger.Debug("Removed check tag", log.String("user_id", userID))
	ctx.Signal("user_license_check", nil).Add(1)
	return task, nil
}

func fetchLicenseChecksMain(ctx *gotasks.Context) error {
	limit := must(ctx.Cmd.Flags().GetInt("limit"))
	ticket, err := ctx.GetServiceTicket("analytics", "analytics")
	if err != nil {
		return err
	}
	client, err := licensechecks.NewClient(licensechecks.WithAuth(ticket))
	if err != nil {
		return err
	}
	users, err := ctx.Drive.FindUsers(api.FindUsersOptions{
		HasAllOf:  []string{"user_license_check"},
		HasNoneOf: []string{"user_deleted_finally"},
		Limit:     limit,
		// TODO(iudovin@): Remove this when timeout will be fixed on backend.
		HasOneOf: []string{"user_license_check"},
	})
	if err != nil {
		return err
	}
	totalCount := 0
	errorsCount := 0
	func() {
		for _, user := range users {
			select {
			case <-ctx.Context.Done():
				return
			default:
			}
			totalCount++
			_, err := trySendUserLicenseCheck(ctx, client, user.ID)
			if err != nil {
				errorsCount++
			}
		}
	}()
	ctx.Logger.Info(
		"Fetch of license checks finished",
		log.Int("total", totalCount),
		log.Int("errors", errorsCount),
	)
	return nil
}

func must[T any](val T, err error) T {
	if err != nil {
		panic(err)
	}
	return val
}

type licenseCheckTaskImpl struct {
	ctx     *gotasks.Context
	client  http.Client
	wrapper CaptchaWrapper
}

type licenseCheckResultDoc struct {
	Date     string `json:"date"`
	BDate    string `json:"bdate"`
	Num      string `json:"num"`
	Type     string `json:"type"`
	Srok     string `json:"srok"`
	Division string `json:"division"`
	Stag     string `json:"stag"`
	Cat      string `json:"cat"`
	VUch     string `json:"v_uch"`
	StKart   string `json:"st_kart"`
	Divid    string `json:"divid"`
	Status   string `json:"status"`
}

type licenseCheckResultDep struct {
	Date    string `json:"date"`
	FisID   string `json:"fis_id"`
	BPlace  string `json:"bplace"`
	Comment string `json:"comment"`
	RegName string `json:"reg_name"`
	State   string `json:"state"`
	Srok    int    `json:"srok"`
	RegCode string `json:"reg_code"`
}

type licenseCheckResult struct {
	Code    int    `json:"code"`
	Message string `json:"message"`
	Count   int    `json:"count"`
	// Doc.
	Doc *licenseCheckResultDoc `json:"doc"`
	// Decis.
	Decis []licenseCheckResultDep `json:"decis"`
}

const userAgentHeader = "Mozilla/5.0 (iPhone; CPU iPhone OS 15_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/15.3 Mobile/15E148 Safari/604.1"

func (t *licenseCheckTaskImpl) getClientContext(ctx context.Context) context.Context {
	clientTrace := &httptrace.ClientTrace{
		GetConn: func(hostPort string) {
			t.ctx.Logger.Debug("Get connection", log.String("host_port", hostPort))
		},
		GotConn: func(info httptrace.GotConnInfo) {
			t.ctx.Logger.Debug(
				"Got connection",
				log.Bool("reused", info.Reused),
				log.Bool("was_idle", info.WasIdle),
			)
		},
		PutIdleConn: func(err error) {
			t.ctx.Logger.Debug("Put idle connection", log.Error(err))
		},
		Got100Continue: func() {
			t.ctx.Logger.Debug("Got 100 continue")
		},
		ConnectStart: func(network, addr string) {
			t.ctx.Logger.Debug(
				"Connect start",
				log.String("network", network),
				log.String("addr", addr),
			)
		},
		ConnectDone: func(network, addr string, err error) {
			t.ctx.Logger.Debug(
				"Connect done",
				log.String("network", network),
				log.String("addr", addr),
				log.Error(err),
			)
		},
		TLSHandshakeStart: func() {
			t.ctx.Logger.Debug("TLS handshake start")
		},
		TLSHandshakeDone: func(state tls.ConnectionState, err error) {
			t.ctx.Logger.Debug(
				"TLS handshake done",
				log.Error(err),
			)
		},
	}
	return httptrace.WithClientTrace(ctx, clientTrace)
}

func (t *licenseCheckTaskImpl) doRunCheck(ctx context.Context, task *licensechecks.Task, token, code string) (licenseCheckResult, error) {
	form := url.Values{}
	form.Add("num", task.LicenseNumber)
	form.Add("date", task.LicenseIssueDate.Format("2006-01-02"))
	form.Add("captchaToken", token)
	form.Add("captchaWord", code)
	req, err := http.NewRequestWithContext(
		t.getClientContext(ctx),
		"POST", "https://xn--b1afk4ade.xn--90adear.xn--p1ai/proxy/check/driver",
		strings.NewReader(form.Encode()),
	)
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	req.Header.Set("User-Agent", userAgentHeader)
	if err != nil {
		return licenseCheckResult{}, err
	}
	resp, err := t.client.Do(req)
	if err != nil {
		return licenseCheckResult{}, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			t.ctx.Logger.Warn("Unable to close body", log.Error(err))
		}
	}()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return licenseCheckResult{}, err
	}
	task.RawResult = string(body)
	t.ctx.Logger.Debug(
		"Check response",
		log.Int("status", resp.StatusCode),
		log.String("license_number", task.LicenseNumber),
		log.String("date", task.LicenseIssueDate.Format("2006-01-02")),
		log.String("body", string(body)),
	)
	if resp.StatusCode != http.StatusOK {
		return licenseCheckResult{}, fmt.Errorf("invalid status %d", resp.StatusCode)
	}
	var result licenseCheckResult
	if err := json.Unmarshal(body, &result); err != nil {
		return licenseCheckResult{}, err
	}
	return result, nil
}

type captchaResult struct {
	Base64JPG string `json:"base64jpg"`
	Token     string `json:"token"`
}

func (t *licenseCheckTaskImpl) doRunCaptcha(ctx context.Context) (string, string, error) {
	req, err := http.NewRequestWithContext(
		t.getClientContext(ctx),
		"GET", "https://check.gibdd.ru/captcha", nil,
	)
	if err != nil {
		return "", "", err
	}
	req.Header.Set("User-Agent", userAgentHeader)
	resp, err := t.client.Do(req)
	if err != nil {
		return "", "", err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			t.ctx.Logger.Warn("Unable to close body", log.Error(err))
		}
	}()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", "", err
	}
	t.ctx.Logger.Debug(
		"Captcha response",
		log.Int("status", resp.StatusCode),
		log.String("body", string(body)),
	)
	var result captchaResult
	if err := json.Unmarshal(body, &result); err != nil {
		return "", "", err
	}
	captchas, err := t.wrapper.Exec(ctx, result.Base64JPG)
	if err != nil {
		return "", "", err
	}
	if len(captchas) != 1 {
		return "", "", fmt.Errorf("expected 1 captcha but got %d", len(captchas))
	}
	return result.Token, captchas[0], nil
}

func (t *licenseCheckTaskImpl) RunTask(ctx context.Context, task *licensechecks.Task) error {
	token, code, err := t.doRunCaptcha(ctx)
	if err != nil {
		t.ctx.Logger.Error("Run check error", log.Error(err))
		t.ctx.Signal("users.license_check.error_sum", nil).Add(1)
		return err
	}
	rawResult, err := t.doRunCheck(ctx, task, token, code)
	if err != nil {
		t.ctx.Logger.Error("Run check error", log.Error(err))
		t.ctx.Signal("users.license_check.error_sum", nil).Add(1)
		return err
	}
	if rawResult.Code != 100 && rawResult.Code != 200 {
		t.ctx.Logger.Error("Run check error", log.Any("result", rawResult))
		t.ctx.Signal("users.license_check.error_sum", nil).Add(1)
		return fmt.Errorf("code %d: %s", rawResult.Code, rawResult.Message)
	}
	t.ctx.Signal("users.license_check.ok_sum", nil).Add(1)
	result := licensechecks.Result{
		Deps: []licensechecks.Deprivation{},
	}
	if doc := rawResult.Doc; doc != nil {
		result.BirthDate = doc.BDate
		result.IssueDate = doc.Date
		result.ExpireDate = doc.Srok
		result.Number = doc.Num
		result.Status = "success"
	} else {
		result.Status = "no_data"
		return task.SetResult(result)
	}
	for _, decis := range rawResult.Decis {
		dep := licensechecks.Deprivation{
			Birthplace: decis.BPlace,
			Period:     licensechecks.Period{Months: decis.Srok},
			StartDate:  decis.Date,
			RawStatus:  fmt.Sprintf("%s: %s", decis.State, decis.Comment),
			Status:     "no_data",
		}
		switch decis.State {
		case "42":
			dep.Status = "started"
		case "60":
			dep.Status = "started"
		case "68":
			dep.Status = "paused"
		case "71":
			dep.Status = "finished"
		case "73":
			dep.Status = "started"
		case "76":
			dep.Status = "started"
		case "78":
			dep.Status = "started"
		case "79":
			dep.Status = "finished"
		case "82":
			dep.Status = "started"
		}
		result.Deps = append(result.Deps, dep)
	}
	t.ctx.Logger.Debug(
		"Gibdd parsed result",
		log.Any("raw_result", rawResult),
		log.Any("result", result),
	)
	return task.SetResult(result)
}

func processLicenseChecksMain(ctx *gotasks.Context) error {
	limit := must(ctx.Cmd.Flags().GetInt("limit"))
	dbName := must(ctx.Cmd.Flags().GetString("db"))
	model := must(ctx.Cmd.Flags().GetString("model"))
	db, ok := ctx.DBs[dbName]
	if !ok {
		return fmt.Errorf("invalid DB name %q", dbName)
	}
	store := licensechecks.NewTaskStore(db, ctx.Logger)
	proxyURL := must(url.Parse(ctx.Config.Animals.URL))
	proxyURL.User = url.UserPassword(ctx.Config.Animals.User, ctx.Config.Animals.Password)
	impl := licenseCheckTaskImpl{
		ctx:     ctx,
		client:  http.Client{Transport: &http.Transport{Proxy: http.ProxyURL(proxyURL)}},
		wrapper: CaptchaWrapper{BinaryPath: model},
	}
	return store.ProcessTasks(ctx.Context, limit, impl.RunTask)
}

type CaptchaWrapper struct {
	BinaryPath string
}

// ExecParse executes binary with specified arguments.
func (w CaptchaWrapper) Exec(
	ctx context.Context, images ...string,
) ([]string, error) {
	args := append([]string{"--base64"}, images...)
	cmd := exec.CommandContext(ctx, w.BinaryPath, args...)
	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return nil, fmt.Errorf("unable to open stdout: %w", err)
	}
	defer func() {
		_ = stdout.Close()
	}()
	if err := cmd.Start(); err != nil {
		return nil, fmt.Errorf("unable to start: %w", err)
	}
	var outputs []string
	if err := json.NewDecoder(stdout).Decode(&outputs); err != nil && err != io.EOF {
		return nil, err
	}
	return outputs, cmd.Wait()
}
