package users

import (
	"context"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/drive/library/go/clients/renis"
	"a.yandex-team.ru/drive/library/go/solomon"
	"github.com/gofrs/uuid"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/gotasks/models/users"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

func init() {
	updateRenisKBMCmd := cobra.Command{
		Use: "update-renis-kbm",
		Run: gotasks.WrapMain(updateRenisKBMMain),
	}
	updateRenisKBMCmd.Flags().Int("req-limit", 50000, "")
	updateRenisKBMCmd.Flags().Int("workers", 4, "")
	UsersCmd.AddCommand(&updateRenisKBMCmd)
	// Register MapReduce.
	mapreduce.Register(joinMapper{})
	mapreduce.Register(updateRenisKBMReducer{})
	mapreduce.Register(snapshotRenisKBMReducer{})
}

func ensureTable(
	ctx *gotasks.Context, path ypath.Path, rowSchema schema.Schema,
) error {
	_, err := ctx.YT.CreateNode(
		ctx.Context, path, yt.NodeTable,
		&yt.CreateNodeOptions{
			Attributes: map[string]interface{}{
				"schema": rowSchema,
			},
			IgnoreExisting: true,
		},
	)
	return err
}

type joinMapper struct {
	mapreduce.Untyped
}

func (joinMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var row map[string]interface{}
		if err := in.Scan(&row); err != nil {
			return err
		}
		row["_table_index"] = in.TableIndex()
		if err := out[0].Write(row); err != nil {
			return err
		}
	}
	return nil
}

func getTableIndex(r mapreduce.Reader) (int, error) {
	var row struct {
		TableIndex int `yson:"_table_index"`
	}
	if err := r.Scan(&row); err != nil {
		return 0, err
	}
	return row.TableIndex, nil
}

func sortedBy(s schema.Schema, col string) schema.Schema {
	s = s.Copy()
	for i := range s.Columns {
		if s.Columns[i].Name == col {
			s.Columns[i].SortOrder = schema.SortAscending
			s.Columns[0], s.Columns[i] = s.Columns[i], s.Columns[0]
			break
		}
	}
	return s
}

func readLastRow(
	tx *goyt.Tx, path ypath.Path, row interface{},
) (bool, error) {
	var count int64
	if err := tx.Get(path.Attr("row_count"), &count); err != nil {
		return false, err
	}
	if count == 0 {
		return false, nil
	}
	count--
	r, err := tx.Raw().ReadTable(
		context.TODO(), path.Rich().AddRange(ypath.Range{
			Exact: &ypath.ReadLimit{RowIndex: &count},
		}), nil,
	)
	if err != nil {
		return false, err
	}
	defer func() {
		_ = r.Close()
	}()
	if !r.Next() {
		return false, nil
	}
	if err := r.Scan(&row); err != nil {
		return false, err
	}
	return true, nil
}

type updateRenisKBMReducer struct {
	mapreduce.Untyped
}

func (r updateRenisKBMReducer) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out)
	})
}

type renisKBMQueueRow struct {
	// UserID.
	UserID uuid.UUID `yson:"user_id"`
	// UpdateTime.
	UpdateTime int64 `yson:"update_time"`
}

func (r updateRenisKBMReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	var renisKBM users.RenisKBM
	var user users.ExtendedUser
	for in.Next() {
		tableIndex, err := getTableIndex(in)
		if err != nil {
			return err
		}
		switch tableIndex {
		case 0: // UserKBMTable.
			if err := in.Scan(&renisKBM); err != nil {
				return err
			}
		case 1: // ExtendedUsersTable.
			if err := in.Scan(&user); err != nil {
				return err
			}
		}
	}
	if user.UserID == uuid.Nil || user.Status != "active" {
		return nil
	}
	return out[0].Write(renisKBMQueueRow{
		UserID:     user.UserID,
		UpdateTime: renisKBM.UpdateTime,
	})
}

func readRenisKBMQueue(
	ctx *gotasks.Context, tx *goyt.Tx,
) (yt.TableReader, error) {
	ctx.Logger.Info("Preparing KBM update queue")
	defer ctx.Logger.Info("KBM queue prepared")
	temp, err := tx.TempTable("")
	if err != nil {
		return nil, err
	}
	joinSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{
			ctx.Config.YTPaths.UserRenisKBMTable,
			ctx.Config.YTPaths.ExtendedUsersTable.Rich().
				SetColumns([]string{"user_id", "status"}),
		},
		OutputTablePaths: []ypath.YPath{temp},
		SortBy:           []string{"user_id"},
		ReduceBy:         []string{"user_id"},
		Pool:             "carsharing",
	}
	joinOp, err := tx.RawMR().MapReduce(
		joinMapper{},
		updateRenisKBMReducer{},
		joinSpec.MapReduce(),
	)
	if err != nil {
		return nil, err
	}
	if err := joinOp.Wait(); err != nil {
		return nil, err
	}
	sortSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{temp},
		OutputTablePath: temp,
		SortBy:          []string{"update_time"},
		Pool:            "carsharing",
	}
	sortOp, err := tx.RawMR().Sort(sortSpec.Sort())
	if err != nil {
		return nil, err
	}
	if err := sortOp.Wait(); err != nil {
		return nil, err
	}
	return tx.ReadTable(temp)
}

type snapshotRenisKBMReducer struct {
	mapreduce.Untyped
}

func (r snapshotRenisKBMReducer) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return r.reduceGroup(in, out)
	})
}

func (r snapshotRenisKBMReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	var last users.RenisKBM
	for in.Next() {
		var row users.RenisKBM
		if err := in.Scan(&row); err != nil {
			return err
		}
		if row.UpdateTime > last.UpdateTime {
			last = row
		}
	}
	return out[0].Write(last)
}

func updateRenisKBMSnapshot(ctx *gotasks.Context) error {
	return goyt.New(ctx.YT).WithTx(func(tx *goyt.Tx) error {
		table := ctx.Config.YTPaths.UserRenisKBMTable
		var dirty bool
		if err := tx.Get(table.Attr("_dirty"), &dirty); err != nil {
			if yterrors.ContainsErrorCode(err, yterrors.CodeResolveError) {
				return nil
			}
			return err
		}
		var updateTime int64
		if err := tx.Get(table.Attr("_update_time"), &updateTime); err != nil &&
			!yterrors.ContainsErrorCode(err, yterrors.CodeResolveError) {
			return err
		}
		snapshotSpec := spec.Spec{
			InputTablePaths: []ypath.YPath{
				table,
				ctx.Config.YTPaths.UserRenisKBMHistoryTable.Rich().
					AddRange(ypath.Range{
						Lower: &ypath.ReadLimit{Key: []interface{}{updateTime}},
					}),
			},
			OutputTablePaths: []ypath.YPath{table},
			ReduceBy:         []string{"user_id"},
			SortBy:           []string{"user_id"},
			Pool:             "carsharing",
		}
		snapshotOp, err := tx.RawMR().MapReduce(
			nil, snapshotRenisKBMReducer{},
			snapshotSpec.MapReduce(),
		)
		if err != nil {
			return err
		}
		if err := snapshotOp.Wait(); err != nil {
			return err
		}
		var lastRow users.RenisKBM
		if _, err := readLastRow(
			tx, ctx.Config.YTPaths.UserRenisKBMHistoryTable, &lastRow,
		); err != nil {
			return err
		}
		if err := tx.Set(
			table.Attr("_update_time"), lastRow.UpdateTime,
		); err != nil {
			return err
		}
		if err := goyt.MergeTable(
			tx, ctx.Config.YTPaths.UserRenisKBMHistoryTable, "sorted",
		); err != nil {
			return err
		}
		return tx.Set(table.Attr("_dirty"), false)
	})
}

type richRenisKBMQueueRow struct {
	renisKBMQueueRow
	api.User
}

func toUnifiedRus(s string) string {
	a := []rune(strings.ToUpper(s))
	for i, c := range a {
		switch c {
		case 'A':
			a[i] = 'А'
		case 'B':
			a[i] = 'В'
		case 'C':
			a[i] = 'С'
		case 'E':
			a[i] = 'Е'
		case 'H':
			a[i] = 'Н'
		case 'M':
			a[i] = 'М'
		case 'O':
			a[i] = 'О'
		case 'P':
			a[i] = 'Р'
		case 'T':
			a[i] = 'Т'
		case 'X':
			a[i] = 'Х'
		case 'Y':
			a[i] = 'У'
		}
	}
	return string(a)
}

func renisKBMWorker(
	ctx *gotasks.Context,
	waiter *sync.WaitGroup,
	mutex *sync.Mutex,
	client *renis.Client,
	input <-chan []richRenisKBMQueueRow,
) {
	defer waiter.Done()
	defer ctx.Logger.Info("Worker stopped")
	ctx.Logger.Info("Worker started")
	for batch := range input {
		ctx.Logger.Info("Process batch")
		var persons []renis.PhysicalPerson
		for _, user := range batch {
			licenseNumber := toUnifiedRus(user.License.FrontNumber)
			passportNumber := toUnifiedRus(user.Passport.Number)
			persons = append(persons, renis.PhysicalPerson{
				FirstName:     user.License.FirstName,
				LastName:      user.License.LastName,
				MiddleName:    user.License.MiddleName,
				Serial:        licenseNumber[:4],
				Number:        licenseNumber[4:],
				ExpDate:       user.License.ExpDate,
				BirthDate:     user.License.BirthDate,
				DocFirstName:  user.Passport.FirstName,
				DocLastName:   user.Passport.LastName,
				DocMiddleName: user.Passport.MiddleName,
				DocIssueDate:  user.Passport.IssueDate,
				DocSerial:     passportNumber[:4],
				DocNumber:     passportNumber[4:],
				Country:       "Россия",
				Province:      "Москва",
				Region:        "Москва",
				City:          "Москва",
			})
		}
		reqID, err := uuid.NewV4()
		if err != nil {
			ctx.Logger.Error("Unable to get reqID", log.Error(err))
			continue
		}
		resp, err := client.CalcKBM(persons, reqID.String())
		if err != nil {
			ctx.Logger.Error("Unable to get kbm", log.Error(err))
			ctx.SignalV(
				"renis_kbm_error", 1, solomon.Label("type", "calc"),
			)
			continue
		}
		if len(resp) != len(batch) {
			ctx.Logger.Errorf(
				"Expected %d length for req %q, got %d",
				len(batch), reqID.String(), len(resp),
			)
			ctx.SignalV(
				"renis_kbm_error", 1, solomon.Label("type", "calc"),
			)
			continue
		}
		if err := func() error {
			mutex.Lock()
			defer mutex.Unlock()
			writer, err := ctx.YT.WriteTable(
				ctx.Context,
				ctx.Config.YTPaths.UserRenisKBMHistoryTable.Rich().SetAppend(),
				&yt.WriteTableOptions{},
			)
			if err != nil {
				return err
			}
			defer writer.Commit()
			for i, user := range batch {
				number := toUnifiedRus(user.License.FrontNumber)
				if number != resp[i].Serial+resp[i].Number {
					ctx.Logger.Error("Response does not match input")
					continue
				}
				result := users.RenisKBM{
					UserID:     user.UserID,
					UpdateTime: time.Now().Unix(),
					KBM:        resp[i].KBM,
					FirstKBM:   resp[i].FirstKBM,
				}
				if err := writer.Write(result); err != nil {
					return err
				}
			}
			return nil
		}(); err != nil {
			ctx.SignalV(
				"renis_kbm_error", 1, solomon.Label("type", "write"),
			)
			continue
		}
		ctx.SignalV(
			"renis_kbm", 1, solomon.Label("type", "history"),
		)
	}
}

func updateRenisKBMMain(ctx *gotasks.Context) error {
	cfg := ctx.Config.RenisKBM
	reqLimit, _ := ctx.Cmd.Flags().GetInt("req-limit")
	workers, _ := ctx.Cmd.Flags().GetInt("workers")
	reqCount := 0
	rowSchema, err := schema.Infer(users.RenisKBM{})
	if err != nil {
		return err
	}
	client := renis.NewClient(
		cfg.Endpoint, cfg.ClientName.Secret(), cfg.PartnerUID.Secret(),
	)
	if err := ensureTable(
		ctx, ctx.Config.YTPaths.UserRenisKBMTable, rowSchema,
	); err != nil {
		return err
	}
	if err := ensureTable(
		ctx, ctx.Config.YTPaths.UserRenisKBMHistoryTable,
		sortedBy(rowSchema, "update_time"),
	); err != nil {
		return err
	}
	if err := updateRenisKBMSnapshot(ctx); err != nil {
		return err
	}
	ctx.SignalV(
		"renis_kbm", 1,
		solomon.Label("type", "snapshot"),
	)
	var waiter sync.WaitGroup
	var mutex sync.Mutex
	input := make(chan []richRenisKBMQueueRow)
	for i := 0; i < workers; i++ {
		waiter.Add(1)
		go renisKBMWorker(ctx, &waiter, &mutex, client, input)
	}
	yc := goyt.New(ctx.YT)
	var batch []richRenisKBMQueueRow
	if err := yc.WithTx(func(tx *goyt.Tx) error {
		reader, err := readRenisKBMQueue(ctx, tx)
		if err != nil {
			return err
		}
		defer reader.Close()
		if err := yc.Set(
			ctx.Config.YTPaths.UserRenisKBMTable.Attr("_dirty"), true,
		); err != nil {
			return err
		}
		for reader.Next() {
			var row renisKBMQueueRow
			if err := reader.Scan(&row); err != nil {
				return err
			}
			user, err := ctx.Drive.GetUser(row.UserID.String())
			if err != nil {
				ctx.Logger.Error("Unable to get user", log.Error(err))
				continue
			}
			if user.Passport == nil || user.License == nil {
				continue
			}
			if c := user.Passport.RegCountry; c != "" &&
				strings.ToLower(c) != "rus" {
				continue
			}
			if len(user.License.FrontNumber) != 10 {
				continue
			}
			if len(user.Passport.Number) != 10 {
				continue
			}
			if user.License.FirstName == "" {
				continue
			}
			if user.License.LastName == "" {
				continue
			}
			if user.License.MiddleName == "" {
				continue
			}
			if user.Passport.FirstName == "" {
				continue
			}
			if user.Passport.LastName == "" {
				continue
			}
			if user.Passport.MiddleName == "" {
				continue
			}
			batch = append(batch, richRenisKBMQueueRow{row, user})
			if len(batch) == 30 {
				if reqCount++; reqCount > reqLimit {
					break
				}
				input <- batch
				batch = nil
			}
		}
		return nil
	}); err != nil {
		close(input)
		waiter.Wait()
		return err
	}
	if len(batch) > 0 {
		input <- batch
		batch = nil
	}
	close(input)
	waiter.Wait()
	if err := updateRenisKBMSnapshot(ctx); err != nil {
		return err
	}
	return goyt.MergeTable(
		yc, ctx.Config.YTPaths.UserRenisKBMHistoryTable, "sorted",
	)
}
