# -*- coding: utf-8 -*-

from logging import getLogger
from datetime import timedelta
from yt.wrapper import TablePath, with_context
from drive.library.py.time import to_timestamp, from_string, TZ_MSK
from drive.analytics.pybase.helpers import get_yt


logger = getLogger(__name__)


def is_fueling_tag(name):
    return name.find("fueling") >= 0 and name not in ("user_fueling_tag", "fueling_washing_liquid")


fuel_tank_volumes = {
    'audi_a3': 56,
    'audi_q3_quattro': 67,
    'audi_q3': 70,
    'bmw_520i_w': 70,
    'bmw_520i': 70,
    'bmw_320d': 60,  # нужен настоящий
    'bmw_X1': 60,
    'citroen_jumpy': 81,  #
    'genesis_g70_w': 69,
    'genesis_g70': 69,
    'hyundai_creta': 68,
    'hyundai_solaris': 50,
    'kia_rio_xline': 51,
    'kia_rio': 50,
    'mercedes_c180': 55,
    'mercedes_e200_w': 75,
    'mercedes_e200': 75,
    'mustang_65': 50,  #
    'mustang_69': 80,  #
    'nissan_leaf': 0,  #
    'nissan_qashqai': 67,
    'petrol_tanker_2': 50,
    'petrol_tanker_3': 0,
    'petrol_tanker': 50,
    'peugeot_expert': 79,  #
    'porsche_carrera': 68,  #
    'porsche_macan': 60,
    'porsche_panamera': 0,  #
    'range_rover_velar': 91,
    'renault_kaptur': 62,
    'skoda_octavia': 55,
    'skoda_rapid': 65,
    'toyota_rav4': 60,  # нужно актуальный
    'volkswagen_caddy': 60,  #
    'volkswagen_caddy_maxi': 60,
    'volkswagen_caravelle': 88,  #
    'volkswagen_transporter_kombi': 87,  #
    'volkswagen_transporter': 88,  #
    'volvo_xc60_diesel': 70,
    'volvo_xc40': 70,  # нужно актуальный
    'volvo_xc60': 70,
    'vw_polo': 65,
    'renault_duster': 50,
    'skoda_rapid_2': 55,
    'volkswagen_polo_6': 55,
    'mercedes_e200_restyling_2020': 66,
    'mercedes_e200d_restyling_2020': 50,
    'ford_transit': 80,
}


def get_liters_by_level_diff(model, after, before):
    if not after or not before:
        return None
    volume = fuel_tank_volumes.get(model, 0)
    return round((float(after - before) / 100 * volume), 2)


def get_imei_map(yt):
    imei_map = {}
    for row in yt.read_table("//home/carsharing/production/data/cars/extended"):
        for imei in row["imei_log"]:
            key = int(imei["imei"])
            imei_map[key] = imei_map.get(key, [])
            items = imei_map[key]
            items.append({
                "object_id": row["car_id"],
                "model": row["model"],
                "begin_time": imei.get("begin_time"),
                "end_time": imei.get("end_time"),
            })
    return imei_map


def get_daily_table_range(yt, dir, begin_dt, end_dt):
    tables = []
    for path in yt.list(dir):
        path_date = from_string(path, "%Y-%m-%d").replace(tzinfo=TZ_MSK)
        if path_date >= begin_dt - timedelta(hours=48) and path_date <= end_dt + timedelta(hours=24):
            tables.append("{}/{}".format(dir, path))
    return tables


# TODO: Verify this implementation.
def get_sensor_by_id(data, sensor_id=None, sensor_name=None):
    if not data:
        return
    records = data.get("records")
    if not records:
        return
    for rec in records:
        if not rec.get("subrecords"):
            continue
        for subrecord in rec["subrecords"]:
            if sensor_name and subrecord.get(sensor_name):
                return subrecord.get(sensor_name)
            if sensor_id:
                if subrecord.get("type") != "custom_parameters" or subrecord.get("params", {}).get(sensor_id) is None:
                    continue
            return subrecord.get("params", {}).get(sensor_id)


def build_sensors_data(yt, begin_dt, end_dt, sensors_table):
    def mapper(row):
        if row["event"] == "incoming" and row["type"] == "BLACKBOX_RECORDS":
            fuel_level = get_sensor_by_id(row["data"], sensor_id="2107")
            if not fuel_level or fuel_level == 255:
                return
            yield {
                "timestamp": row["unixtime"],
                "imei": row["imei"],
                "fuel_level_sensor": fuel_level,
                "sensor": "2107",
            }
        elif row["event"] == "sensors" and row["type"] == "OnSensorDerived":
            fuel_level = None
            for record in row["data"]:
                if record["id"] == 2107 and record["subid"] == 2211:
                    fuel_level = record["value"]
            if not fuel_level or fuel_level == 255:
                return
            yield {
                "timestamp": row["unixtime"],
                "imei": row["imei"],
                "fuel_level_sensor": fuel_level,
                "sensor": "2107/2211",
            }

    logger.info("Build sensors begin time: {}".format(begin_dt.strftime("%Y-%m-%d %H:%M")))
    logger.info("Build sensors end time: {}".format(end_dt.strftime("%Y-%m-%d %H:%M")))
    telematics_tables = get_daily_table_range(yt, "//logs/carsharing-telematics-events-log/1d", begin_dt, end_dt)
    yt.run_map(
        mapper,
        source_table=telematics_tables,
        destination_table=sensors_table,
    )


def build_mapped_sensors_data(yt, imei_map, sensors_table, mapped_sensors_table):
    def sensor_mapper(row):
        if row["timestamp"] is None:
            return
        for item in imei_map.get(int(row["imei"]), []):
            begin_time = item["begin_time"] or 0
            end_time = item["end_time"] or 1e18
            if begin_time <= row["timestamp"] and row["timestamp"] <= end_time:
                yield {"object_id": item["object_id"], "model": item["model"], **row}

    yt.run_map(
        sensor_mapper,
        source_table=sensors_table,
        destination_table=mapped_sensors_table,
    )


def build_closed_tags(yt, begin_dt, end_dt, mapped_sensors_table, closed_tags_table):
    safe_begin_ts = to_timestamp(begin_dt - timedelta(hours=3))
    safe_end_ts = to_timestamp(end_dt + timedelta(hours=3))

    @with_context
    def mapper(row, context):
        row["_table_index"] = context.table_index
        if row["_table_index"] == 0:  # Sensor.
            row["_key"] = str(row["object_id"])
            row["_subkey"] = int(row["timestamp"])
        elif row["_table_index"] == 1:  # Tag.
            row["_key"] = str(row["object_id"])
            row["_subkey"] = int(row["history_timestamp"])
            if not is_fueling_tag(row["tag"]):
                return
            if row["history_action"] not in ("set_performer", "remove"):
                return
            if row["history_timestamp"] < safe_begin_ts or row["history_timestamp"] > safe_end_ts:
                return
        yield row

    def reducer(_, rows):
        fuelings = {}  # Contains current fuelings.
        closed_fuelings = []
        last_level = None
        last_model = "unknown"
        for row in rows:
            if row["_table_index"] == 0:  # Sensor.
                last_level = row["fuel_level_sensor"]
                last_model = row["model"]
                for key in fuelings:
                    fueling = fuelings[key]
                    fueling["min_level"] = min(last_level, fueling["min_level"] or 1000)
                    fueling["max_level"] = max(last_level, fueling["max_level"] or -1000)
                # Update closed fuelings.
                new_closed_fuelings = []
                for fueling in closed_fuelings:
                    # Wait for sensor for next 5 minutes or before next performed fueling.
                    if row["timestamp"] <= fueling["end_time"] + 5*60 or len(fuelings) == 0:
                        fueling["max_level"] = max(last_level, fueling["max_level"] or -1000)
                        new_closed_fuelings.append(fueling)
                    else:
                        # No more sensors for this fueling in future.
                        fueling["liters"] = get_liters_by_level_diff(
                            fueling["model"],
                            fueling["max_level"],
                            fueling["min_level"],
                        )
                        yield fueling
                closed_fuelings = new_closed_fuelings
            elif row["_table_index"] == 1:  # Tag.
                if row["history_action"] == "set_performer":
                    fuelings[row["tag_id"]] = {
                        "tag": row,
                        "min_level": last_level,
                        "max_level": last_level,
                    }
                elif row["history_action"] == "remove":
                    if row["tag_id"] in fuelings:
                        fueling = fuelings[row["tag_id"]]
                        closed_fuelings.append({
                            "tag": fueling["tag"]["tag"],
                            "model": last_model,
                            "tag_id": fueling["tag"]["tag_id"],
                            "object_id": fueling["tag"]["object_id"],
                            "user_id": fueling["tag"]["performer"],
                            "min_level": fueling["min_level"],
                            "max_level": fueling["max_level"],
                            "begin_time": int(fueling["tag"]["history_timestamp"]),
                            "end_time": int(row["history_timestamp"]),
                        })
                        del fuelings[row["tag_id"]]
        for fueling in closed_fuelings:
            fueling["liters"] = get_liters_by_level_diff(
                fueling["model"],
                fueling["max_level"],
                fueling["min_level"],
            )
            yield fueling

    yt.run_map_reduce(
        mapper,
        reducer,
        source_table=(
            mapped_sensors_table,
            TablePath(
                "//home/carsharing/production/data/exports/car_tags_history",
                columns=("history_timestamp", "history_action", "tag_id", "tag", "object_id", "performer"),
            ),
        ),
        destination_table=closed_tags_table,
        reduce_by=("_key"),
        sort_by=("_key", "_subkey"),
    )


def append_closed_tags(yt, full_table, delta_table, split_dt):
    if not yt.exists(full_table):
        yt.run_sort(
            source_table=delta_table,
            destination_table=full_table,
            sort_by=("begin_time",),
        )
    else:
        split_key = to_timestamp(split_dt)
        with yt.Transaction():
            with yt.TempTable() as tmp_tabe:
                yt.run_sort(
                    source_table=delta_table,
                    destination_table=tmp_tabe,
                    sort_by=("begin_time"),
                )
                yt.run_merge(
                    source_table=(
                        TablePath(full_table, upper_key=int(split_key) - 1),
                        TablePath(tmp_tabe, lower_key=int(split_key)),
                    ),
                    destination_table=full_table,
                    mode="sorted",
                )


def closed_tags_main(args):
    yt = get_yt()
    sensors_table = "//home/carsharing/production/data/fuelings/_raw_sensor"
    mapped_sensors_table = "//home/carsharing/production/data/fuelings/_mapped_sensor"
    closed_tags_table = "//home/carsharing/production/data/fuelings/_closed_tags_history"
    full_closed_tags_table = "//home/carsharing/production/data/fuelings/closed_tags_history"
    imei_map = get_imei_map(yt)
    build_sensors_data(yt, args.from_date, args.to_date, sensors_table)
    build_mapped_sensors_data(yt, imei_map, sensors_table, mapped_sensors_table)
    build_closed_tags(yt, args.from_date, args.to_date, mapped_sensors_table, closed_tags_table)
    append_closed_tags(yt, full_closed_tags_table, closed_tags_table, args.from_date)
