# -*- coding: UTF-8 -*-

# from copy import deepcopy
from os import environ
from logging import getLogger
from argparse import ArgumentParser
from requests import get
from multiprocessing.dummy import Pool
from yt.wrapper import TablePath
from drive.analytics.pybase.helpers import get_yt


_log = getLogger(__name__)


schema = [
	{"name":"agreement_number", "type":"any"},
	{"name":"agreement_partner_number", "type":"any"},
	{"name":"beacon", "type":"any"},
	{"name":"car_id", "type":"string"},
	{"name":"car_insurance_attr", "type":"any"},
	{"name":"first_order", "type":"int64"},
	{"name":"last_order", "type":"int64"},
	{"name":"head_id", "type":"any"},
	{"name":"imei", "type":"any"},
	{"name":"mileage", "type":"int64"},
	{"name":"model", "type":"string"},
	{"name":"modem", "type":"any"},
	{"name":"number", "type":"string"},
	{"name":"parking_permit_start_date", "type":"any"},
	{"name":"tags", "type":"any"},
	{"name":"transfer_date", "type":"any"},
	{"name":"transponder", "type":"any"},
	{'name':'airport', 'type':'any'},
	{"name":"vin", "type":"string"},
	{"name":"city", "type":"string"},
	{"name":"segment", "type":"string"},
	{"name": "car_tags", "type": "any"},
	{"name": "filters", "type": "any"},
	{"name": "lessor", "type": "string"},
]


def get_mapper(cars, models):
	def mapper(row):
		vin = None
		number = None
		lessor = None
		transfer_date = None
		parking_permit_start_date = None
		for document in row["unsafe_documents"].get("car_registry_document", []):
			vin = document.get("vin", vin)
			number = document.get("number", number)
			lessor = document.get("lessor", lessor)
			transfer_date = document.get("transfer_date", transfer_date)
			parking_permit_start_date = document.get("parking_permit_start_date", parking_permit_start_date)
		agreement_number = None
		agreement_partner_number = None
		for document in row["unsafe_documents"].get("car_insurance_policy", []):
			agreement_number = document.get("agreement_number", agreement_number)
			agreement_partner_number = document.get("agreement_partner_number", agreement_partner_number)
		imeis = []
		for event in row["imei_log"] or []:
			imei = event.get("imei")
			imeis.append({
				"add": event.get("begin_time"),
				"remove": event.get("end_time"),
				"imei": str(imei) if imei else None,
			})
		head_ids = []
		for event in row["head_log"] or []:
			head_ids.append({
				"add": event.get("begin_time"),
				"remove": event.get("end_time"),
				"head_id": event.get("head_id"),
			})
		model = row["model"]
		segment = None
		tags = None
		mileage = None
		if row["car_id"] in cars:
			car = cars[row["car_id"]]
			number = car["number"]
			model = car["model_id"]
			segment = models.get(model, {}).get("segment")
			tags = (car.get("location") or {}).get("tags")
			mileage = int(car.get("telematics", {}).get("mileage") or 0)
		return dict(
			car_id=row["car_id"],
			model=model,
			imei=imeis,
			head_id=head_ids,
			vin=row["vin"] or vin,
			number=number,
			lessor=lessor,
			car_tags=row["tags"],
			tags=tags,
			modem=row["unsafe_documents"].get("car_hardware_modem", []),
			car_insurance_attr=row["unsafe_documents"].get("car_insurance_policy", []),
			transponder=row["unsafe_documents"].get("car_transponder_spb", []),
			transfer_date=transfer_date,
			agreement_number=agreement_number,
			agreement_partner_number=agreement_partner_number,
			parking_permit_start_date=parking_permit_start_date,
			segment=segment,
			mileage=mileage,
			# beacon
			# first_order
			# last_order
			# airport
			# city
			# filters
		)

	return mapper


def get_cars(endpoint):
	resp = get(
		"{}/api/staff/car/list?sensors=fuel_level,fuel_distance,mileage&traits=ReportIMEI,ReportVIN,ReportModels".format(endpoint),
		headers={
			"Authorization": "OAuth {}".format(environ["DRIVE_TOKEN"]),
		}
	)
	resp.raise_for_status()
	return resp.json()


def main():
	cmd = ArgumentParser()
	cmd.add_argument("--extended-path")
	cmd.add_argument("--yt-path")
	cmd.add_argument("--workers", default=16)
	cmd.add_argument("--endpoint", default="https://leasing-cabinet.carsharing.yandex.net")
	args = cmd.parse_args()
	yc = get_yt()
	cars_resp = get_cars(args.endpoint)
	cars = {}
	for car in cars_resp["cars"]:
		cars[car["id"]] = car
	models = cars_resp["models"]
	with Pool(args.workers) as pool:
		rows = yc.read_table(args.extended_path)
		mapper = get_mapper(cars, models)
		yc.write_table(
			TablePath(args.yt_path, schema=schema),
			pool.map(mapper, rows),
		)


if __name__ == "__main__":
	main()
