package adjust

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/library/go/gosql"
)

// View represents API view.
type View struct {
	core   *core.Core
	db     *gosql.DB
	config *config.Adjust
}

// callbackFrom represents form for Adjust callback handler.
type callbackFrom struct {
	Type               string `query:"type"`
	CreatedAt          int64  `query:"created_at"`
	AdjustID           string `query:"adjust_id"`
	IDFA               string `query:"idfa"`
	AndroidID          string `query:"android_id"`
	AppID              string `query:"app_id"`
	AppVersion         string `query:"app_version"`
	InstalledAt        *int64 `query:"installed_at"`
	UninstalledAt      *int64 `query:"uninstalled_at"`
	ReinstalledAt      *int64 `query:"reinstalled_at"`
	CampaignName       string `query:"campaign_name"`
	AdgroupName        string `query:"adgroup_name"`
	CreativeName       string `query:"creative_name"`
	Label              string `query:"label"`
	ConversionDuration *int64 `query:"conversion_duration"`
	ClickTime          *int64 `query:"click_time"`
	Country            string `query:"country"`
	LastTrackerName    string `query:"last_tracker_name"`
	ConnectionType     string `query:"connection_type"`
	DeviceName         string `query:"device_name"`
	UserAgent          string `query:"user_agent"`
	IPAddress          string `query:"ip_address"`
	EngagementTime     int64  `query:"engagement_time"`
	ImpressionBased    bool   `query:"impression_based"`
	TrackerName        string `query:"tracker_name"`
	Params             string `query:"params"`
	Referrer           string `query:"referrer"`
	SecretToken        string `query:"secret_token"`
}

func (v *View) getCallback(c echo.Context) error {
	var form callbackFrom
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return err
	}
	if form.SecretToken != v.config.Secret.Secret() {
		return c.NoContent(http.StatusForbidden)
	}
	var params struct {
		UserID string `json:"user_id"`
	}
	if form.Params != "" {
		if err := json.Unmarshal([]byte(form.Params), &params); err != nil {
			c.Logger().Error(err)
			return err
		}
	}
	referrerQuery := ""
	referrerParts := strings.SplitN(form.Referrer, "|", 2)
	if len(referrerParts) == 2 && len(referrerParts[1]) > 0 {
		referrerURL, err := url.Parse(referrerParts[1])
		if err != nil {
			c.Logger().Error(err)
			return err
		}
		referrerQuery = referrerURL.RawQuery
	}
	referrer, err := url.ParseQuery(referrerQuery)
	if err != nil {
		c.Logger().Error(err)
		return err
	}
	utmSource := referrer.Get("utm_source")
	utmMedium := referrer.Get("utm_medium")
	utmCampaign := referrer.Get("utm_campaign")
	utmContent := referrer.Get("utm_content")
	utmTerm := referrer.Get("utm_term")
	if err := gosql.WithTx(v.db, func(tx *sql.Tx) error {
		stmt, err := tx.Prepare(fmt.Sprintf(
			`INSERT INTO %q `+
				`("type", "created_at", "adjust_id", "idfa", "android_id", `+
				`"app_id", "app_version", "installed_at", "uninstalled_at", `+
				`"reinstalled_at", "campaign_name", "adgroup_name", `+
				`"creative_name", "label", "conversion_duration", `+
				`"click_time", "country", "last_tracker_name", `+
				`"connection_type", "device_name", "user_agent", `+
				`"ip_address", "engagement_time", "impression_based", `+
				`"tracker_name", "user_id", "utm_source", "utm_medium", `+
				`"utm_campaign", "utm_content", "utm_term") `+
				`VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, `+
				`?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
			v.config.Table,
		))
		if err != nil {
			c.Logger().Error(err)
			return err
		}
		_, err = stmt.Exec(
			form.Type, form.CreatedAt, form.AdjustID, form.IDFA,
			form.AndroidID, form.AppID, form.AppVersion, form.InstalledAt,
			form.UninstalledAt, form.ReinstalledAt, form.CampaignName,
			form.AdgroupName, form.CreativeName, form.Label,
			form.ConversionDuration, form.ClickTime, form.Country,
			form.LastTrackerName, form.ConnectionType, form.DeviceName,
			form.UserAgent, form.IPAddress, form.EngagementTime,
			form.ImpressionBased, form.TrackerName, params.UserID,
			utmSource, utmMedium, utmCampaign, utmContent, utmTerm,
		)
		return err
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.NoContent(http.StatusCreated)
}

// Register registers API handlers.
func (v *View) Register(g *echo.Group) {
	g.GET("/callback", v.getCallback)
}

// NewView creates a new instance of view.
func NewView(c *core.Core) *View {
	db, ok := c.DBs[c.Config.Adjust.DB]
	if !ok {
		panic("DB not found")
	}
	return &View{
		core:   c,
		db:     db,
		config: c.Config.Adjust,
	}
}
