package backendlog

import (
	"bufio"
	"bytes"
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"net/url"
	"strconv"
	"sync"
	"time"

	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/logbroker"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/zootopia/analytics/drive/helpers"
	"a.yandex-team.ru/zootopia/library/go/geom"
)

// Config represents backend log configurtion.
type Config struct {
	DB       string `json:"db"`
	Consumer string `json:"consumer"`
	Topic    string `json:"topic"`
	Target   string `json:"target"`
	// Source contains TVM source alias.
	//
	// TODO(iudovin@): Add support for source configuration.
	Source string `json:"source"`
	// Table contains name of log table.
	Table string `json:"table"`
	// PriceModelTable contains name of price model table.
	PriceModelTable string `json:"price_model_table"`
	// Workers contains amount of workers.
	Workers int `json:"workers"`
	// Endpoints contains list of logbroker endpoints.
	Endpoints []string `json:"endpoints"`
}

type Service struct {
	core   *core.Core
	db     *gosql.DB
	tvm    tvm.Client
	logger log.Logger
	config Config
}

func (s *Service) Start() {
	s.core.StartDaemon(s.daemon)
}

func (s *Service) daemon(ctx context.Context) error {
	endpoints := logbroker.WithProduction
	if len(s.config.Endpoints) > 0 {
		endpoints = logbroker.WithEndpoint(s.config.Endpoints...)
	}
	reader, err := logbroker.NewReader(
		endpoints,
		logbroker.WithConsumer(s.config.Consumer),
		logbroker.WithTopic(s.config.Topic),
		logbroker.WithTVM(s.tvm, s.config.Target),
		logbroker.WithLogger(s.logger),
	)
	if err != nil {
		return err
	}
	s.logger.Info("Daemon started")
	defer func() {
		if err := reader.Close(); err != nil {
			s.logger.Error("LogBroker close error", log.Error(err))
		}
		s.logger.Warn("Daemon stopped")
	}()
	var waiter sync.WaitGroup
	defer waiter.Wait()
	readerCtx, cancel := context.WithCancel(ctx)
	defer cancel()
	waiter.Add(1)
	go func() {
		defer waiter.Done()
		defer cancel()
		if err := s.daemonReader(readerCtx, reader); err != nil {
			s.logger.Error("Daemon error", log.Error(err))
		}
	}()
	for i := 1; i < s.config.Workers; i++ {
		waiter.Add(1)
		go func() {
			defer waiter.Done()
			defer cancel()
			if err := s.daemonReader(readerCtx, reader); err != nil {
				s.logger.Error("Daemon error", log.Error(err))
			}
		}()
	}
	<-readerCtx.Done()
	return nil
}

func (s *Service) daemonReader(
	ctx context.Context, reader logbroker.Reader,
) error {
	// Buffer size should be at least 128 megabytes.
	buffer := make([]byte, 128*1024*1024)
	for {
		select {
		case event := <-reader.C():
			var err error
			for retry := 0; retry < 5; retry++ {
				if err = s.onLogEvent(event, buffer); err != nil {
					s.logger.Warn("Unable to process event", log.Error(err))
				} else {
					break
				}
				time.Sleep(time.Duration(retry) * time.Second)
			}
			if err != nil {
				s.logger.Error("Unable to process event", log.Error(err))
				return err
			}
		case <-ctx.Done():
			return nil
		}
	}
}

type Event struct {
	// UnixTime contains time of backend event.
	UnixTime int64 `json:"unixtime"`
	// Event contains type of backend event.
	Event string `json:"event"`
	// UserID contains user ID.
	UserID *string `json:"user_id"`
	// Source contains source url.
	Source *string `json:"source"`
	// Query contains query.
	Query *string `json:"query"`
	// Data contains backend event data.
	Data json.RawMessage `json:"data"`
}

func (s *Service) onLogEvent(event persqueue.Event, buffer []byte) error {
	switch e := event.(type) {
	case *persqueue.Data:
		var events []Event
		var offerCreated []Event
		// Process all batches for current event.
		for _, batch := range e.Batches() {
			for _, message := range batch.Messages {
				scanner := bufio.NewScanner(bytes.NewBuffer(message.Data))
				scanner.Buffer(buffer, len(buffer))
				for scanner.Scan() {
					var event Event
					if err := json.Unmarshal(
						scanner.Bytes(), &event,
					); err != nil {
						s.logger.Warn(
							"Unable to unmarshal JSON", log.Error(err),
						)
						continue
					}
					switch event.Event {
					case "BookOffer", "CompiledRiding", "CarList":
						events = append(events, event)
					case "OfferCreated":
						events = append(events, event)
						offerCreated = append(offerCreated, event)
					}
				}
				if err := scanner.Err(); err != nil {
					return err
				}
			}
		}
		// Process all extracted events.
		if err := gosql.WithTx(s.db, func(tx *sql.Tx) error {
			return s.processEvents(tx, events)
		}); err != nil {
			return err
		}
		if err := gosql.WithTx(s.db, func(tx *sql.Tx) error {
			return s.processOfferCreated(tx, offerCreated)
		}); err != nil {
			return err
		}
		e.Commit()
	}
	return nil
}

func (s *Service) processOfferCreated(tx *sql.Tx, events []Event) error {
	// Do not prepare any statement if there are no rows.
	if len(events) == 0 {
		return nil
	}
	stmt, err := tx.Prepare(
		fmt.Sprintf(
			`INSERT INTO %q `+
				`("time", "offer_id", "offer_type", "object_id", `+
				`"price_model", "price_model_group", "riding_price", `+
				`"before_price", "after_price") `+
				`VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
			s.config.PriceModelTable,
		),
	)
	if err != nil {
		return err
	}
	for _, event := range events {
		if err := s.onOfferCreated(stmt, event); err != nil {
			return err
		}
	}
	return nil
}

type PriceModel struct {
	Name   string  `json:"Name"`
	Before float64 `json:"Before"`
	After  float64 `json:"After"`
}

type StandardOffer struct {
	RidingPrice int64        `json:"PriceRiding"`
	PriceModels []PriceModel `json:"PriceModelInfo"`
}

type ProtoLocation struct {
	Longitude float64 `json:"XHP"`
	Latitude  float64 `json:"YHP"`
}

type Offer struct {
	OfferID       string        `json:"OfferId"`
	StandardOffer StandardOffer `json:"StandartOffer"`
	InstanceType  string        `json:"InstanceType"`
	ObjectID      string        `json:"ObjectId"`
	UserID        string        `json:"UserId"`
	Origin        string        `json:"Origin"`
	RidingStart   ProtoLocation `json:"OriginalRidingStart"`
}

type BookOffer struct {
	OfferID string `json:"offer_id"`
	Origin  string `json:"origin"`
	Type    string `json:"type"`
}

type RidingPriceModelEvent struct {
	Time            int64   `json:"time"`
	OfferID         string  `json:"offer_id"`
	OfferType       string  `json:"offer_type"`
	ObjectID        string  `json:"object_id"`
	PriceModel      string  `json:"price_model"`
	PriceModelGroup string  `json:"price_model_group"`
	RidingPrice     float64 `json:"riding_price"`
	BeforePrice     float64 `json:"before_price"`
	AfterPrice      float64 `json:"after_price"`
}

func (s *Service) onOfferCreated(stmt *sql.Stmt, e Event) error {
	var offer Offer
	if err := json.Unmarshal(e.Data, &offer); err != nil {
		s.logger.Error("Unable to unmarshal offer", log.Error(err))
		return nil
	}
	for _, model := range offer.StandardOffer.PriceModels {
		row := RidingPriceModelEvent{
			Time:            e.UnixTime,
			OfferID:         offer.OfferID,
			OfferType:       offer.InstanceType,
			ObjectID:        offer.ObjectID,
			PriceModel:      model.Name,
			PriceModelGroup: "",
			RidingPrice:     float64(offer.StandardOffer.RidingPrice) / 100,
			BeforePrice:     model.Before / 100,
			AfterPrice:      model.After / 100,
		}
		if _, err := stmt.Exec(
			row.Time, row.OfferID, row.OfferType, row.ObjectID,
			row.PriceModel, row.PriceModelGroup, row.RidingPrice,
			row.BeforePrice, row.AfterPrice,
		); err != nil {
			return err
		}
	}
	return nil
}

type ParsedEvent struct {
	Time            int64   `db:"time"`
	EventType       int8    `db:"event_type"`
	UserID          string  `db:"user_id"`
	ObjectID        string  `db:"object_id"`
	SessionID       string  `db:"session_id"`
	Total           float64 `db:"total"`
	StartTime       int64   `db:"start_time"`
	Duration        int     `db:"duration"`
	RidingDuration  int     `db:"riding_duration"`
	ParkingDuration int     `db:"parking_duration"`
	Origin          string  `db:"origin"`
	Path            string  `db:"path"`
	OfferType       string  `db:"offer_type"`
	City            string  `db:"city"`
}

const (
	CreateOffer   = 1
	CreateSession = 2
	FinishSession = 3
	ListCars      = 4
)

func (s *Service) processEvents(tx *sql.Tx, events []Event) error {
	// Do not prepare any statement if there are no rows.
	if len(events) == 0 {
		return nil
	}
	stmt, err := tx.Prepare(
		fmt.Sprintf(
			`INSERT INTO %q`+
				` ("time", "event_type", "user_id", "object_id",`+
				` "session_id", "total", "start_time", "duration",`+
				` "riding_duration", "parking_duration", "origin",`+
				` "path", "offer_type", "city")`+
				` VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
			s.config.Table,
		),
	)
	if err != nil {
		return err
	}
	for _, event := range events {
		if err := s.onEvent(stmt, event); err != nil {
			return err
		}
	}
	return nil
}

func castToFloat64(v interface{}) float64 {
	switch t := v.(type) {
	case string:
		if x, err := strconv.ParseFloat(t, 64); err == nil {
			return x
		}
	}
	return 0
}

func (s *Service) onEvent(stmt *sql.Stmt, e Event) error {
	row := ParsedEvent{Time: e.UnixTime}
	if e.Source != nil {
		row.Path = *e.Source
	}
	if e.UserID != nil {
		row.UserID = *e.UserID
	}
	switch e.Event {
	case "OfferCreated":
		var offer Offer
		if err := json.Unmarshal(e.Data, &offer); err != nil {
			s.logger.Error("Unable to unmarshal offer", log.Error(err))
			return nil
		}
		row.EventType = CreateOffer
		row.SessionID = offer.OfferID
		row.UserID = offer.UserID
		row.ObjectID = offer.ObjectID
		row.Origin = offer.Origin
		row.OfferType = offer.InstanceType
		row.City = helpers.CityByLocation(
			offer.RidingStart.Longitude, offer.RidingStart.Latitude,
		)
	case "BookOffer":
		var offer BookOffer
		if err := json.Unmarshal(e.Data, &offer); err != nil {
			s.logger.Error("Unable to unmarshal offer", log.Error(err))
			return nil
		}
		if e.Query != nil {
			if values, err := url.ParseQuery(*e.Query); err != nil {
				s.logger.Warn("Unable to parse query", log.Error(err))
			} else {
				row.ObjectID = values.Get("car_id")
			}
		}
		row.EventType = CreateSession
		row.SessionID = offer.OfferID
		row.Origin = offer.Origin
		row.OfferType = offer.Type
	case "CompiledRiding":
		var order CompiledRiding
		if err := json.Unmarshal(e.Data, &order); err != nil {
			s.logger.Error("Unable to unmarshal order", log.Error(err))
			return nil
		}
		var prevEvent TagEvent
		for _, event := range order.Events {
			if prevEvent.Tag == "old_state_riding" {
				row.RidingDuration += int(event.Time - prevEvent.Time)
			}
			if prevEvent.Tag == "old_state_parking" {
				row.ParkingDuration += int(event.Time - prevEvent.Time)
			}
			prevEvent = event
		}
		row.EventType = FinishSession
		row.UserID = order.OfferProto.UserID
		row.ObjectID = order.OfferProto.ObjectID
		row.SessionID = order.OfferProto.OfferID
		row.Total = float64(order.Total) / 100
		row.StartTime = order.Start
		row.Duration = int(order.Finish - order.Start)
		row.Origin = order.OfferProto.Origin
		row.OfferType = order.OfferProto.InstanceType
		row.City = helpers.CityByLocation(
			order.Diff.Finish.Longitude, order.Diff.Finish.Latitude,
		)
	case "CarList":
		if e.UserID == nil {
			s.logger.Error("Unable to extract user")
			return nil
		}
		var cars carListData
		if err := json.Unmarshal(e.Data, &cars); err != nil {
			s.logger.Error("Unable to unmarshal car list", log.Error(err))
			return nil
		}
		row.EventType = ListCars
		for _, car := range cars.Cars {
			if len(car) < 2 {
				s.logger.Warn("Invalid car in car list")
				continue
			}
			var pos geom.Vec2
			if err := json.Unmarshal(car[1], &pos); err != nil {
				s.logger.Warn("Invalid car pos", log.Error(err))
				continue
			}
			if city := helpers.CityByLocation(pos[0], pos[1]); city != "" {
				row.City = city
				break
			}
		}
	default:
		return fmt.Errorf("unsupported event %q", e.Event)
	}
	_, err := stmt.Exec(
		row.Time, row.EventType, row.UserID, row.ObjectID, row.SessionID,
		row.Total, row.StartTime, row.Duration, row.RidingDuration,
		row.ParkingDuration, row.Origin, row.Path, row.OfferType, row.City,
	)
	return err
}

type carListData struct {
	Cars [][]json.RawMessage `json:"cars"`
}

type Bill struct {
	Cost     int    `json:"cost"`
	Duration int    `json:"duration"`
	Type     string `json:"type"`
}

type TagEvent struct {
	EventID int64  `json:"event_id"`
	Action  string `json:"action"`
	Tag     string `json:"tag_name"`
	Time    int64  `json:"timestamp"`
}

type DiffLocation struct {
	Latitude  float64 `json:"latitude"`
	Longitude float64 `json:"longitude"`
}

type Diff struct {
	Start  DiffLocation `json:"start"`
	Finish DiffLocation `json:"finish"`
}

type CompiledRiding struct {
	Diff       Diff       `json:"diff"`
	Bills      []Bill     `json:"bill"`
	Events     []TagEvent `json:"events"`
	Offer      BookOffer  `json:"offer"`
	OfferProto Offer      `json:"offer_proto"`
	Start      int64      `json:"start"`
	Finish     int64      `json:"finish"`
	Total      int64      `json:"total_price"`
	SessionID  string     `json:"session_id"`
}

func NewService(c *core.Core, cfg Config) *Service {
	db, ok := c.DBs[cfg.DB]
	if !ok {
		panic("database not found")
	}
	tvm, ok := c.TVMs[cfg.Source]
	if !ok {
		panic("tvm source not found")
	}
	return &Service{
		core:   c,
		db:     db,
		tvm:    tvm,
		logger: c.Logger("backend_log"),
		config: cfg,
	}
}
