package garage

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/library/go/gosql"
)

type FieldType int

const (
	IntegerField FieldType = 1
	FloatField   FieldType = 2
	StringField  FieldType = 3
)

func (t FieldType) String() string {
	switch t {
	case IntegerField:
		return "integer"
	case FloatField:
		return "float"
	case StringField:
		return "string"
	default:
		return fmt.Sprintf("FieldType(%d)", t)
	}
}

type EventType int

const (
	CreateEvent EventType = 1
	DeleteEvent EventType = 2
)

func (t EventType) String() string {
	switch t {
	case CreateEvent:
		return "create"
	case DeleteEvent:
		return "delete"
	default:
		return fmt.Sprintf("EventType(%d)", t)
	}
}

type Field struct {
	ID   int       `db:"id"`
	Name string    `db:"name"`
	Type FieldType `db:"type"`
	// Title represents display name of field.
	Title string `db:"title"`
}

func (o Field) CheckValue(v models.JSON) bool {
	switch o.Type {
	case IntegerField:
		var val int64
		if err := json.Unmarshal(v, &val); err != nil {
			return false
		}
		return true
	case FloatField:
		var val float64
		if err := json.Unmarshal(v, &val); err != nil {
			return false
		}
		return true
	case StringField:
		var val string
		if err := json.Unmarshal(v, &val); err != nil {
			return false
		}
		return true
	default:
		return false
	}
}

type Mutation struct {
	ID         int64     `db:"id"`
	UserID     uuid.UUID `db:"user_id"`
	Comment    string    `db:"comment"`
	CreateTime int64     `db:"create_time"`
}

type FieldValue struct {
	ID        int64         `db:"id"`
	FieldID   int           `db:"field_id"`
	VIN       string        `db:"vin"`
	Value     models.JSON   `db:"value"`
	BeginTime models.NInt64 `db:"begin_time"`
	EndTime   models.NInt64 `db:"end_time"`
	// CreateID contains ID of create mutation.
	CreateID int64 `db:"create_id"`
	// DeleteID contains ID of delete mutation.
	DeleteID models.NInt64 `db:"delete_id"`
}

type FieldStore struct {
	db    *gosql.DB
	table string
}

func (s *FieldStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]Field, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(Field{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var fields []Field
	for rows.Next() {
		var field Field
		if err := rows.Scan(gosql.StructValues(&field, true)...); err != nil {
			return nil, err
		}
		fields = append(fields, field)
	}
	return fields, rows.Err()
}

func (s *FieldStore) All() ([]Field, error) {
	return s.FindTx(s.db, nil)
}

func NewFieldStore(db *gosql.DB, table string) *FieldStore {
	return &FieldStore{db: db, table: table}
}

type MutationStore struct {
	db    *gosql.DB
	table string
}

func (s *MutationStore) CreateTx(
	tx gosql.Runner, mutation *Mutation,
) error {
	mutation.CreateTime = time.Now().Unix()
	names, values := gosql.StructNameValues(mutation, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if err := row.Scan(&mutation.ID); err != nil {
		return err
	}
	return nil
}

func (s *MutationStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]Mutation, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(Mutation{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var mutations []Mutation
	for rows.Next() {
		var mutation Mutation
		if err := rows.Scan(
			gosql.StructValues(&mutation, true)...,
		); err != nil {
			return nil, err
		}
		mutations = append(mutations, mutation)
	}
	return mutations, rows.Err()
}

func (s *MutationStore) All() ([]Mutation, error) {
	return s.FindTx(s.db, nil)
}

func NewMutationStore(db *gosql.DB, table string) *MutationStore {
	return &MutationStore{db: db, table: table}
}

type FieldValueStore struct {
	db    *gosql.DB
	table string
}

func (s *FieldValueStore) CreateTx(
	tx gosql.Runner, fieldValue *FieldValue,
) error {
	names, values := gosql.StructNameValues(fieldValue, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if err := row.Scan(&fieldValue.ID); err != nil {
		return err
	}
	return nil
}

func (s *FieldValueStore) UpdateTx(
	tx gosql.Runner, fieldValue FieldValue,
) error {
	names, values := gosql.StructNameValues(fieldValue, false, "id")
	query, values := s.db.Update(s.table).
		Names(names...).Values(values...).
		Where(gosql.Column("id").Equal(fieldValue.ID)).Build()
	res, err := tx.Exec(query, values...)
	if err != nil {
		return err
	}
	count, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if count != 1 {
		return fmt.Errorf("row with ID %d does not exist", fieldValue.ID)
	}
	return nil
}

func (s *FieldValueStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]FieldValue, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(FieldValue{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var fieldValues []FieldValue
	for rows.Next() {
		var fieldValue FieldValue
		if err := rows.Scan(
			gosql.StructValues(&fieldValue, true)...,
		); err != nil {
			return nil, err
		}
		fieldValues = append(fieldValues, fieldValue)
	}
	return fieldValues, rows.Err()
}

func (s *FieldValueStore) FindByVIN(vin string) ([]FieldValue, error) {
	return s.FindTx(s.db, gosql.Column("vin").Equal(vin))
}

func (s *FieldValueStore) All() ([]FieldValue, error) {
	return s.FindTx(s.db, gosql.Column("delete_id").Equal(nil))
}

func NewFieldValueStore(db *gosql.DB, table string) *FieldValueStore {
	return &FieldValueStore{db: db, table: table}
}

type GarageManager struct {
	// fields contains store for fields.
	fields *FieldStore
	// mutations contains store for mutations.
	mutations *MutationStore
	// fieldValues contains store for field values.
	fieldValues *FieldValueStore
}

func (m *GarageManager) CreateValuesTx(
	tx gosql.Runner, values []FieldValue, user uuid.UUID, comment string,
) (Mutation, error) {
	if len(values) == 0 {
		return Mutation{}, fmt.Errorf("empty mutation")
	}
	fields, err := m.fields.FindTx(tx, nil)
	if err != nil {
		return Mutation{}, err
	}
	fieldPos := map[int]int{}
	for i, field := range fields {
		fieldPos[field.ID] = i
	}
	for _, value := range values {
		pos, ok := fieldPos[value.FieldID]
		if !ok {
			return Mutation{}, fmt.Errorf(
				"unknown field with ID %d", value.FieldID,
			)
		}
		if value.BeginTime != 0 && value.EndTime != 0 &&
			value.BeginTime >= value.EndTime {
			return Mutation{}, fmt.Errorf(
				"begin time should be less than end time",
			)
		}
		if !fields[pos].CheckValue(value.Value) {
			return Mutation{}, fmt.Errorf(
				"field %q has invalid value: %q",
				fields[pos].Name, value.Value,
			)
		}
	}
	mutation := Mutation{UserID: user, Comment: comment}
	if err := m.mutations.CreateTx(tx, &mutation); err != nil {
		return Mutation{}, err
	}
	for _, value := range values {
		value.CreateID = mutation.ID
		if err := m.fieldValues.CreateTx(tx, &value); err != nil {
			return Mutation{}, err
		}
	}
	return mutation, nil
}

func (m *GarageManager) RollbackMutationTx(
	tx gosql.Runner, id int64, user uuid.UUID,
) (Mutation, error) {
	createValues, err := m.fieldValues.FindTx(tx, gosql.Column("create_id").Equal(id))
	if err != nil {
		return Mutation{}, err
	}
	deleteValues, err := m.fieldValues.FindTx(tx, gosql.Column("delete_id").Equal(id))
	if err != nil {
		return Mutation{}, err
	}
	if len(createValues) == 0 && len(deleteValues) == 0 {
		return Mutation{}, fmt.Errorf("empty mutation")
	}
	mutation := Mutation{
		UserID:  user,
		Comment: fmt.Sprintf("Rollback #%d", id),
	}
	if err := m.mutations.CreateTx(tx, &mutation); err != nil {
		return Mutation{}, err
	}
	for _, value := range createValues {
		if value.DeleteID != 0 {
			continue
		}
		value.DeleteID = models.NInt64(mutation.ID)
		if err := m.fieldValues.UpdateTx(tx, value); err != nil {
			return Mutation{}, err
		}
	}
	for _, value := range deleteValues {
		value.ID = 0
		value.CreateID = mutation.ID
		value.DeleteID = 0
		if err := m.fieldValues.CreateTx(tx, &value); err != nil {
			return Mutation{}, err
		}
	}
	return mutation, nil
}

func (m *GarageManager) Fields() *FieldStore {
	return m.fields
}

func (m *GarageManager) FieldValues() *FieldValueStore {
	return m.fieldValues
}

func (m *GarageManager) Mutations() *MutationStore {
	return m.mutations
}

func NewGarageManager(db *gosql.DB) *GarageManager {
	return &GarageManager{
		fields:      NewFieldStore(db, "garage_field"),
		mutations:   NewMutationStore(db, "garage_mutation"),
		fieldValues: NewFieldValueStore(db, "garage_field_value"),
	}
}

type Document struct {
	ID         int64        `db:"id"`
	UserID     models.NUUID `db:"user_id"`
	Format     string       `db:"format"`
	FileName   string       `db:"file_name"`
	CreateTime int64        `db:"create_time"`
}

type DocumentStore struct {
	db    *gosql.DB
	table string
}

// DB returns store databse.
func (s DocumentStore) DB() *gosql.DB {
	return s.db
}

func (s *DocumentStore) CreateTx(
	tx gosql.Runner, document *Document,
) error {
	names, values := gosql.StructNameValues(document, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if err := row.Scan(&document.ID); err != nil {
		return err
	}
	return nil
}

func (s *DocumentStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]Document, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(Document{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var documents []Document
	for rows.Next() {
		var document Document
		if err := rows.Scan(
			gosql.StructValues(&document, true)...,
		); err != nil {
			return nil, err
		}
		documents = append(documents, document)
	}
	return documents, rows.Err()
}

func NewDocumentStore(db *gosql.DB, table string) *DocumentStore {
	return &DocumentStore{db: db, table: table}
}
