package garage

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"os/exec"
)

type ParseDocsWrapper struct {
	BinaryPath string
	BinaryEnv  map[string]string
}

type ParseStatus struct {
	Message string `json:"message"`
}

type Column struct {
	Type string `json:"type"`
}

type Scheme struct {
	Columns map[string]Column `json:"columns"`
}

type Format struct {
	Name   string `json:"name"`
	Scheme Scheme `json:"scheme"`
}

type Formats struct {
	Formats []Format `json:"formats"`
}

// ExecParse executes binary with specified arguments.
func (w ParseDocsWrapper) ExecParse(
	ctx context.Context, src, dst, format string, meta interface{},
) (ParseStatus, error) {
	rawMeta, err := json.Marshal(meta)
	if err != nil {
		return ParseStatus{}, fmt.Errorf("unable to marshal meta: %w", err)
	}
	cmd := exec.CommandContext(
		ctx, w.BinaryPath, "parse",
		"--format", format,
		"--input-file", src,
		"--output-file", dst,
		"--meta", string(rawMeta),
	)
	for key, value := range w.BinaryEnv {
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", key, value))
	}
	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return ParseStatus{}, fmt.Errorf("unable to open stdout: %w", err)
	}
	defer func() {
		_ = stdout.Close()
	}()
	if err := cmd.Start(); err != nil {
		return ParseStatus{}, fmt.Errorf("unable to start: %w", err)
	}
	var status ParseStatus
	if err := json.NewDecoder(stdout).Decode(&status); err != nil && err != io.EOF {
		return ParseStatus{}, err
	}
	return status, cmd.Wait()
}

func (w ParseDocsWrapper) ExecUpload(ctx context.Context, src, format string) (ParseStatus, error) {
	cmd := exec.CommandContext(
		ctx, w.BinaryPath, "upload",
		"--format", format,
		"--parsed-file", src,
	)
	for key, value := range w.BinaryEnv {
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", key, value))
	}
	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return ParseStatus{}, fmt.Errorf("unable to open stdout: %w", err)
	}
	defer func() {
		_ = stdout.Close()
	}()
	if err := cmd.Start(); err != nil {
		return ParseStatus{}, fmt.Errorf("unable to start: %w", err)
	}
	var status ParseStatus
	if err := json.NewDecoder(stdout).Decode(&status); err != nil && err != io.EOF {
		return ParseStatus{}, err
	}
	return status, cmd.Wait()
}

// ExecFormats executes binary with specified arguments.
func (w ParseDocsWrapper) ExecFormats(ctx context.Context) (Formats, error) {
	cmd := exec.CommandContext(ctx, w.BinaryPath, "formats")
	for key, value := range w.BinaryEnv {
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", key, value))
	}
	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return Formats{}, fmt.Errorf("unable to open stdout: %w", err)
	}
	defer func() {
		_ = stdout.Close()
	}()
	if err := cmd.Start(); err != nil {
		return Formats{}, fmt.Errorf("unable to start: %w", err)
	}
	var formats Formats
	if err := json.NewDecoder(stdout).Decode(&formats); err != nil && err != io.EOF {
		return Formats{}, err
	}
	return formats, cmd.Wait()
}
