package garage

import (
	"database/sql"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"sort"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/secret"
)

type Config struct {
	DB         string                   `json:"db"`
	BinaryPath string                   `json:"binary_path"`
	BinaryEnv  map[string]secret.Secret `json:"binary_env"`
}

type View struct {
	core      *core.Core
	cfg       Config
	db        *gosql.DB
	garage    *GarageManager
	documents *DocumentStore
	wrapper   *ParseDocsWrapper
}

type ErrorResponse struct {
	Message string `json:"message"`
}

type fieldValue struct {
	Value     models.JSON `json:"value"`
	BeginTime int64       `json:"begin_time,omitempty"`
	EndTime   int64       `json:"end_time,omitempty"`
}

type fieldInfo struct {
	Values []fieldValue `json:"values"`
}

type vinFieldsInfo struct {
	VIN    string                `json:"vin"`
	Fields map[string]*fieldInfo `json:"fields"`
}

type ListVinsInfoForm struct {
	Begin string `query:"begin"`
	Limit int    `query:"limit"`
	Time  int64  `query:"time"`
}

type fieldResp struct {
	Name  string `json:"name"`
	Type  string `json:"type"`
	Title string `json:"title"`
}

type ListVinsInfoResp struct {
	Infos     []vinFieldsInfo `json:"infos"`
	Fields    []fieldResp     `json:"fields"`
	NextBegin string          `json:"next_begin,omitempty"`
}

type fieldValueSorter []FieldValue

func (a fieldValueSorter) Len() int {
	return len(a)
}

func (a fieldValueSorter) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a fieldValueSorter) Less(i, j int) bool {
	if a[i].FieldID != a[j].FieldID {
		return a[i].FieldID < a[j].FieldID
	}
	if a[i].BeginTime != a[j].BeginTime {
		return a[i].BeginTime < a[j].BeginTime
	}
	return a[i].EndTime < a[j].EndTime
}

type fieldRespSorter []fieldResp

func (a fieldRespSorter) Len() int {
	return len(a)
}

func (a fieldRespSorter) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a fieldRespSorter) Less(i, j int) bool {
	return a[i].Name < a[j].Name
}

func (v *View) listVinsInfo(c echo.Context) error {
	form := ListVinsInfoForm{}
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("Unable to parse request: %q", err.Error()),
		})
	}
	if form.Limit <= 0 {
		form.Limit = 100
	}
	fields := map[int]Field{}
	{
		allFields, err := v.garage.Fields().All()
		if err != nil {
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				Message: fmt.Sprintf("Unable to load fields: %q", err.Error()),
			})
		}
		for _, field := range allFields {
			fields[field.ID] = field
		}
	}
	fieldValues, err := v.garage.FieldValues().All()
	if err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("Unable to load values: %q", err.Error()),
		})
	}
	vinInfos := map[string][]FieldValue{}
	for _, value := range fieldValues {
		if form.Begin != "" && value.VIN < form.Begin {
			continue
		}
		vinInfos[value.VIN] = append(vinInfos[value.VIN], value)
	}
	for _, infos := range vinInfos {
		sort.Sort(fieldValueSorter(infos))
	}
	var vins []string
	for vin := range vinInfos {
		vins = append(vins, vin)
	}
	sort.Strings(vins)
	var resp ListVinsInfoResp
	for _, field := range fields {
		resp.Fields = append(resp.Fields, fieldResp{
			Name:  field.Name,
			Type:  field.Type.String(),
			Title: field.Title,
		})
	}
	sort.Sort(fieldRespSorter(resp.Fields))
	for _, vin := range vins {
		if len(resp.Infos) >= form.Limit {
			resp.NextBegin = vin
			break
		}
		fieldsInfo := vinFieldsInfo{
			VIN:    vin,
			Fields: map[string]*fieldInfo{},
		}
		for _, value := range vinInfos[vin] {
			field, ok := fields[value.FieldID]
			if !ok {
				continue
			}
			info, ok := fieldsInfo.Fields[field.Name]
			if !ok {
				info = &fieldInfo{}
				fieldsInfo.Fields[field.Name] = info
			}
			if form.Time == 0 {
				fValue := fieldValue{
					Value:     value.Value,
					BeginTime: int64(value.BeginTime),
					EndTime:   int64(value.EndTime),
				}
				info.Values = append(info.Values, fValue)
			} else {
				if value.EndTime != 0 && int64(value.EndTime) <= form.Time {
					continue
				}
				info.Values = []fieldValue{{
					Value:     value.Value,
					BeginTime: int64(value.BeginTime),
					EndTime:   int64(value.EndTime),
				}}
			}
		}
		resp.Infos = append(resp.Infos, fieldsInfo)
	}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) getVinInfo(c echo.Context) error {
	return nil
}

type CreateVinsInfoForm struct {
	VINs map[string]struct {
		Fields    map[string]models.JSON `json:"fields"`
		BeginTime *int64                 `json:"begin_time"`
		EndTime   *int64                 `json:"end_time"`
	} `json:"vins"`
	Comment string `json:"comment"`
}

func (v *View) createVinsInfo(c echo.Context) error {
	user, ok := c.Get(core.AuthDriveUser).(core.DriveUser)
	if !ok {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message: "Unable to get Drive user",
		})
	}
	var form CreateVinsInfoForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("Unable to parse payload: %q", err.Error()),
		})
	}
	fields := map[string]Field{}
	{
		allFields, err := v.garage.Fields().All()
		if err != nil {
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				Message: fmt.Sprintf("Unable to load fields: %q", err.Error()),
			})
		}
		for _, field := range allFields {
			fields[field.Name] = field
		}
	}
	hasChanges := false
	for vin, values := range form.VINs {
		for name, value := range values.Fields {
			if field, ok := fields[name]; ok {
				if !field.CheckValue(value) {
					return c.JSON(http.StatusBadRequest, ErrorResponse{
						Message: fmt.Sprintf(
							"Invalid value for field %q (vin = %q)",
							name, vin,
						),
					})
				}
				hasChanges = true
			} else {
				return c.JSON(http.StatusBadRequest, ErrorResponse{
					Message: fmt.Sprintf(
						"Field %q does not exists (vin = %q)",
						name, vin,
					),
				})
			}
		}
	}
	if !hasChanges {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: "Nothing was changes",
		})
	}
	if err := gosql.WithTxContext(
		c.Request().Context(), v.db, nil,
		func(tx *sql.Tx) error {
			mutation := Mutation{UserID: user.ID, Comment: form.Comment}
			if err := v.garage.Mutations().CreateTx(tx, &mutation); err != nil {
				return err
			}
			for vin, values := range form.VINs {
				for name, value := range values.Fields {
					fieldValue := FieldValue{
						FieldID:  fields[name].ID,
						VIN:      vin,
						Value:    value,
						CreateID: mutation.ID,
					}
					if values.BeginTime != nil {
						fieldValue.BeginTime = models.NInt64(*values.BeginTime)
					}
					if values.EndTime != nil {
						fieldValue.EndTime = models.NInt64(*values.EndTime)
					}
					if err := v.garage.FieldValues().CreateTx(tx, &fieldValue); err != nil {
						return err
					}
				}
			}
			return nil
		},
	); err != nil {
		c.Logger().Error("Error:", err)
		return err
	}
	return c.NoContent(http.StatusOK)
}

func (v *View) listMutations(c echo.Context) error {
	return c.NoContent(http.StatusOK)
}

func (v *View) getMutation(c echo.Context) error {
	return c.NoContent(http.StatusOK)
}

type RollbackMutationForm struct {
	MutationID int64 `param:"mutation_id"`
}

func (v *View) rollbackMutation(c echo.Context) error {
	user, ok := c.Get(core.AuthDriveUser).(core.DriveUser)
	if !ok {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message: "Unable to get Drive user",
		})
	}
	var form RollbackMutationForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to parse payload: %s", err.Error(),
			),
		})
	}
	var mutation Mutation
	if err := gosql.WithTxContext(
		c.Request().Context(), v.db, nil,
		func(tx *sql.Tx) (err error) {
			mutation, err = v.garage.RollbackMutationTx(
				tx, form.MutationID, user.ID,
			)
			return err
		},
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to rollback mutation: %s", err.Error(),
			),
		})
	}
	return c.JSON(http.StatusCreated, uploadFileResp{
		MutationID: mutation.ID,
		Comment:    mutation.Comment,
	})
}

type UploadFileForm struct {
	Comment string `query:"comment" form:"comment"`
}

type uploadFileResp struct {
	MutationID int64  `json:"mutation_id"`
	Comment    string `json:"comment"`
}

func (v *View) uploadFileOld(c echo.Context) error {
	user, ok := c.Get(core.AuthDriveUser).(core.DriveUser)
	if !ok {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message: "Unable to get Drive user",
		})
	}
	var form UploadFileForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to parse payload: %s", err.Error(),
			),
		})
	}
	fields := map[string]Field{}
	{
		allFields, err := v.garage.Fields().All()
		if err != nil {
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				Message: fmt.Sprintf(
					"Unable to load fields: %s", err.Error(),
				),
			})
		}
		for _, field := range allFields {
			fields[field.Name] = field
		}
	}
	file, err := c.FormFile("file")
	if err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf(
				"File is not specified: %s", err.Error(),
			),
		})
	}
	reader, err := file.Open()
	if err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to open file: %s", err.Error(),
			),
		})
	}
	defer func() {
		_ = reader.Close()
	}()
	fieldValues, err := ParseXLSX(reader, fields)
	if err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to parse file: %s", err.Error(),
			),
		})
	}
	var mutation Mutation
	if err := gosql.WithTxContext(
		c.Request().Context(), v.db, nil,
		func(tx *sql.Tx) (err error) {
			mutation, err = v.garage.CreateValuesTx(
				tx, fieldValues, user.ID, form.Comment,
			)
			return err
		},
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: fmt.Sprintf(
				"Unable to upload file: %s", err.Error(),
			),
		})
	}
	return c.JSON(http.StatusCreated, uploadFileResp{
		MutationID: mutation.ID,
		Comment:    mutation.Comment,
	})
}

func (v *View) observeFormats(c echo.Context) error {
	formats, err := v.wrapper.ExecFormats(c.Request().Context())
	if err != nil {
		return err
	}
	return c.JSON(http.StatusOK, formats)
}

type uploadFileForm struct {
	Format string `form:"format"`
}

type documentResp struct {
	ID         int64        `json:"id"`
	UserID     models.NUUID `json:"user_id,omitempty"`
	Format     string       `json:"format"`
	FileName   string       `json:"file_name"`
	CreateTime int64        `json:"create_time"`
}

func (v *View) uploadFile(c echo.Context) error {
	var form uploadFileForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("unable to parse form: %v", err),
		})
	}
	file, err := c.FormFile("file")
	if err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("unable to load file: %v", err),
		})
	}
	src, err := file.Open()
	if err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf(
				"unable to open file %q: %v", file.Filename, err,
			),
		})
	}
	defer func() {
		_ = src.Close()
	}()
	tmpIn, err := ioutil.TempFile("/tmp", "garage-*.xlsx")
	if err != nil {
		return err
	}
	defer func() {
		_ = os.Remove(tmpIn.Name())
	}()
	if _, err := io.Copy(tmpIn, src); err != nil {
		return err
	}
	if err := tmpIn.Close(); err != nil {
		return err
	}
	tmpOut, err := ioutil.TempFile("/tmp", "garage-*.json")
	if err != nil {
		return err
	}
	defer func() {
		_ = os.Remove(tmpOut.Name())
	}()
	if err := tmpOut.Close(); err != nil {
		return err
	}
	meta := map[string]string{
		"file_name": file.Filename,
		"format":    form.Format,
	}
	parseResp, err := v.wrapper.ExecParse(c.Request().Context(), tmpIn.Name(), tmpOut.Name(), form.Format, meta)
	if err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusBadRequest, parseResp)
	}
	document := Document{
		Format:     form.Format,
		FileName:   file.Filename,
		CreateTime: time.Now().Unix(),
	}
	if user, ok := c.Get(core.AuthDriveUser).(core.DriveUser); ok {
		document.UserID.UUID = user.ID
	}
	createTx := func(tx *sql.Tx) error {
		if err := v.documents.CreateTx(tx, &document); err != nil {
			return err
		}
		uploadResp, err := v.wrapper.ExecUpload(c.Request().Context(), tmpOut.Name(), form.Format)
		if err != nil {
			c.Logger().Error(uploadResp)
		}
		return err
	}
	if err := gosql.WithTxContext(
		c.Request().Context(), v.documents.DB(), nil, createTx,
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: fmt.Sprintf(
				"unable to create document: %v", err,
			),
		})
	}
	//nolint:S1016
	return c.JSON(http.StatusCreated, documentResp{
		ID:         document.ID,
		UserID:     document.UserID,
		Format:     document.Format,
		FileName:   document.FileName,
		CreateTime: document.CreateTime,
	})
}

func (v *View) Register(g *echo.Group) {
	// Old garage API.
	g.GET("/vins-info", v.listVinsInfo)
	g.GET("/vins-info/:vin", v.getVinInfo)
	g.POST("/vins-info", v.createVinsInfo)
	g.POST("/upload-file", v.uploadFileOld)
	g.GET("/mutations", v.listMutations)
	g.GET("/mutations/:mutation_id", v.getMutation)
	g.POST("/mutations/:mutation_id/rollback", v.rollbackMutation)
	// Garage API.
	g.GET(
		"/v0/formats", v.observeFormats,
		v.core.RequireDriveAction("analytics_garage_observe"),
	)
	g.POST(
		"/v0/upload-file", v.uploadFile,
		v.core.RequireDriveAction("analytics_garage_upload_file"),
	)
}

func NewView(c *core.Core, cfg Config) *View {
	db, ok := c.DBs[cfg.DB]
	if !ok {
		panic(fmt.Errorf("db %q does not exist", cfg.DB))
	}
	envs := map[string]string{}
	for key, value := range cfg.BinaryEnv {
		envs[key] = value.Secret()
	}
	return &View{
		core:      c,
		cfg:       cfg,
		db:        db,
		garage:    NewGarageManager(db),
		documents: NewDocumentStore(db, "garage_document"),
		wrapper: &ParseDocsWrapper{
			BinaryPath: cfg.BinaryPath,
			BinaryEnv:  envs,
		},
	}
}
