package garage

import (
	"encoding/json"
	"fmt"
	"io"
	"strconv"
	"time"

	"github.com/xuri/excelize/v2"

	"a.yandex-team.ru/drive/analytics/gobase/models"
)

func parseXLSXDate(s string) (int64, error) {
	if ts, err := time.Parse("01/02/06", s); err == nil {
		return ts.Unix(), nil
	}
	if ts, err := time.Parse("02.01.2006", s); err == nil {
		return ts.Unix(), nil
	}
	if ts, err := time.Parse("2006-01-02", s); err == nil {
		return ts.Unix(), nil
	}
	i, err := strconv.ParseInt(s, 10, 64)
	if err != nil {
		return 0, err
	}
	return (i - 25569) * 86400, nil
}

func ParseXLSX(reader io.Reader, fields map[string]Field) ([]FieldValue, error) {
	xlsx, err := excelize.OpenReader(reader)
	if err != nil {
		return nil, err
	}
	sheet := xlsx.GetSheetName(xlsx.GetActiveSheetIndex())
	rows, err := xlsx.GetRows(sheet)
	if err != nil {
		return nil, err
	}
	var columns []string
	var fieldValues []FieldValue
	for i, row := range rows {
		var vin string
		var beginTime, endTime int64
		values := map[string]interface{}{}
		for j, cell := range row {
			if i == 0 {
				columns = append(columns, cell)
				continue
			}
			if j >= len(columns) {
				break
			}
			switch columns[j] {
			case "VIN":
				vin = cell
			case "Дата начала действия условий":
				beginTime, err = parseXLSXDate(cell)
				if err != nil {
					return nil, err
				}
			case "Дата окончания действия условий":
				endTime, err = parseXLSXDate(cell)
				if err != nil {
					return nil, err
				}
				endTime += 24 * 3600
			case "Платеж по ДС без НДС":
				values["ds_payment"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "Срок аренды":
				values["lease_term_months"], err = strconv.ParseInt(cell, 10, 64)
				if err != nil {
					return nil, err
				}
			case "Контрактный пробег":
				values["contract_mileage"], err = strconv.ParseInt(cell, 10, 64)
				if err != nil {
					return nil, err
				}
			case "Ставка недопробега":
				values["under_run_rate"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "Ставка перепробега":
				values["over_run_rate"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "Дата возврата":
				values["return_date"], err = parseXLSXDate(cell)
				if err != nil {
					return nil, err
				}
			case "ОСАГО сумма":
				values["osago_cost"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "ДАГО сумма":
				values["dago_cost"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "Парковочное":
				values["parking_permit_cost"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			case "Платеж на лизинговых каникулах", "Платёж на лизинговых каникулах":
				values["lease_holiday_payment"], err = strconv.ParseFloat(cell, 64)
				if err != nil {
					return nil, err
				}
			}
		}
		if i == 0 {
			continue
		}
		if beginTime == 0 {
			if vin == "" {
				break
			}
			return nil, fmt.Errorf("begin_time is not specified for VIN %q", vin)
		}
		for key, value := range values {
			fieldValue := FieldValue{
				FieldID: fields[key].ID,
				VIN:     vin,
			}
			fieldValue.Value, err = json.Marshal(value)
			if err != nil {
				return nil, err
			}
			fieldValue.BeginTime = models.NInt64(beginTime)
			fieldValue.EndTime = models.NInt64(endTime)
			fieldValues = append(fieldValues, fieldValue)
		}
	}
	return fieldValues, nil
}
