package licensechecks

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"a.yandex-team.ru/drive/library/go/auth"
)

type Client struct {
	endpoint string
	client   http.Client
	auth     auth.Provider
}

func (c *Client) Request(form RequestForm) (Task, error) {
	reqData, err := json.Marshal(form)
	if err != nil {
		return Task{}, err
	}
	req, err := http.NewRequest(
		"POST",
		c.endpoint+"/request",
		bytes.NewBuffer(reqData),
	)
	if err != nil {
		return Task{}, err
	}
	req.Header.Add("Content-Type", "application/json")
	if c.auth != nil {
		if err := c.auth.UpdateRequest(req); err != nil {
			return Task{}, err
		}
	}
	resp, err := c.client.Do(req)
	if err != nil {
		return Task{}, err
	}
	defer func() {
		_ = resp.Body.Close()
	}()
	if resp.StatusCode != 201 {
		return Task{}, fmt.Errorf("invalid code: %d", resp.StatusCode)
	}
	var task Task
	if err := json.NewDecoder(resp.Body).Decode(&task); err != nil {
		return Task{}, err
	}
	return task, nil
}

const (
	ProductionEndpoint = "https://admin.carsharing.yandex.net/analytics/license-checks"
	PrestableEndpoint  = "https://prestable.carsharing.yandex.net/analytics/license-checks"
)

type ClientOption func(c *Client) error

func WithAuth(provider auth.Provider) ClientOption {
	return func(c *Client) (err error) {
		c.auth = provider
		return
	}
}

func NewClient(options ...ClientOption) (*Client, error) {
	client := Client{
		endpoint: ProductionEndpoint,
	}
	for _, option := range options {
		if err := option(&client); err != nil {
			return nil, err
		}
	}
	return &client, nil
}
