package licensechecks

import (
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/gosql/stores"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
)

type Period struct {
	Months int `json:"months"`
}

type Deprivation struct {
	Birthplace string `json:"birthplace"`
	Period     Period `json:"period"`
	RawStatus  string `json:"raw_status"`
	StartDate  string `json:"start_date"`
	Status     string `json:"status"`
}

type Result struct {
	BirthDate  string        `json:"birth_date"`
	Deps       []Deprivation `json:"deps"`
	ExpireDate string        `json:"expire_date"`
	IssueDate  string        `json:"issue_date"`
	Number     string        `json:"number"`
	Status     string        `json:"status"`
}

// Task represents state of license check task.
type Task struct {
	ID               int64       `db:"id"`
	LicenseNumber    string      `db:"license_number"`
	LicenseIssueDate time.Time   `db:"license_issue_date"`
	Result           models.JSON `db:"result"`
	RawResult        string      `db:"raw_result"`
	CheckRetry       int         `db:"check_retry"`
	CheckTries       int         `db:"check_tries"`
	CheckNextTime    *time.Time  `db:"check_next_time"`
	NotifyRetry      int         `db:"notify_retry"`
	NotifyNextTime   *time.Time  `db:"notify_next_time"`
	Callback         string      `db:"callback"`
	CallbackData     models.JSON `db:"callback_data"`
	CallbackURL      *string     `db:"callback_url"`
	ResultTime       *time.Time  `db:"result_time"`
	Priority         int         `db:"priority"`
}

func (o *Task) SetResult(result Result) error {
	var err error
	o.Result, err = json.Marshal(result)
	return err
}

// ObjectID returns ID of task.
func (o Task) ObjectID() int64 {
	return o.ID
}

func (o *Task) SetObjectID(id int64) {
	o.ID = id
}

// TaskStore represents task store for license check.
type TaskStore struct {
	objects stores.SimpleObjectStore[Task, *Task]
	logger  log.Logger
}

// Create creates task in store.
func (s *TaskStore) Create(ctx context.Context, task *Task) error {
	return s.objects.CreateObject(ctx, task)
}

type TaskFn func(context.Context, *Task) error

var sqlReadWrite = &sql.TxOptions{
	// Force REPEATABLE READ for Postgres.
	Isolation: sql.LevelRepeatableRead,
}

func (s *TaskStore) ProcessTasks(ctx context.Context, limit int, fn TaskFn) error {
	if tx := stores.GetTx(ctx); tx != nil {
		return fmt.Errorf("transaction is not supported")
	}
	var tasks []Task
	if err := gosql.WithTxContext(ctx, s.objects.DB(), sqlReadWrite, func(tx *sql.Tx) error {
		ctx := stores.WithTx(ctx, tx)
		// if err := m.objects.LockStore(ctx); err != nil {
		// 	return err
		// }
		checkNextTime := gosql.Column("check_next_time")
		where := checkNextTime.NotEqual(nil).And(checkNextTime.Less(time.Now()))
		rows, err := s.objects.SelectObjects(
			ctx,
			stores.WithWhere(where),
			stores.WithLimit(limit),
		)
		if err != nil {
			return err
		}
		defer func() {
			_ = rows.Close()
		}()
		var allTasks []Task
		for rows.Next() {
			allTasks = append(allTasks, rows.Row())
		}
		if err := rows.Err(); err != nil {
			return err
		}
		for _, task := range allTasks {
			if task.CheckRetry > task.CheckTries {
				now := time.Now()
				task.CheckNextTime = nil
				task.NotifyNextTime = &now
			} else {
				ts := time.Now().Add(time.Minute * time.Duration(5+task.CheckRetry))
				task.CheckNextTime = &ts
				task.CheckRetry++
				tasks = append(tasks, task)
			}
			if err := s.objects.UpdateObject(ctx, &task); err != nil {
				return err
			}
		}
		return nil
	}); err != nil {
		return err
	}
	for _, task := range tasks {
		select {
		case <-ctx.Done():
			return ctx.Err()
		default:
		}
		if err := fn(ctx, &task); err != nil {
			s.logger.Warn(
				"Error",
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
			if err := s.objects.UpdateObject(ctx, &task); err != nil {
				s.logger.Error(
					"Unable to update task",
					log.Int64("task_id", task.ID),
					log.Error(err),
				)
			}
			continue
		}
		now := time.Now()
		task.CheckNextTime = nil
		task.NotifyNextTime = &now
		task.ResultTime = &now
		if err := s.objects.UpdateObject(ctx, &task); err != nil {
			s.logger.Error(
				"Unable to update task",
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
		}
	}
	return nil
}

// NewTaskStore creates a new instance of TaskStore.
func NewTaskStore(db *gosql.DB, logger log.Logger) *TaskStore {
	return &TaskStore{
		objects: stores.NewSimpleObjectStore[Task, *Task](db, "license_check_task", "id"),
		logger:  logger,
	}
}
