package licensechecks

import (
	"fmt"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

type CallbackConfig struct {
	TvmID int32 `json:"tvm_id"`
}

type Config struct {
	Callbacks map[string]CallbackConfig `json:"callbacks"`
}

// View represents API view for license checks.
type View struct {
	cfg   Config
	tvm   tvm.Client
	tasks *TaskStore
}

// RequestForm represents form for license check request.
type RequestForm struct {
	// Callback contains name of callback service.
	Callback string `json:""`
	// CallbackData contains ID of callback.
	CallbackData models.JSON `json:""`
	// CallbackURL contains URL of callback.
	CallbackURL *string `json:""`
	// LicenseNumber contains number of driver license.
	LicenseNumber string `json:""`
	// LicenseIssueDate contains issue date of driver license.
	LicenseIssueDate string `json:""`
	// Priority contains priority of operations.
	//
	// Empty value treated as 10.
	Priority *int `json:""`
	// CheckTries contains amount of tries.
	//
	// Empty value treated as 5.
	CheckTries *int `json:""`
}

// RequestError represents error of license check request.
type RequestError struct {
	Error error `json:""`
}

func (v *View) request(c echo.Context) error {
	var form RequestForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, RequestError{Error: err})
	}
	issueDate, err := time.Parse("2006-01-02", form.LicenseIssueDate)
	if err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, RequestError{Error: err})
	}
	ticket := c.Get(serviceTicketKey).(*tvm.CheckedServiceTicket)
	callback, ok := v.cfg.Callbacks[form.Callback]
	if !ok {
		return c.NoContent(http.StatusForbidden)
	}
	if callback.TvmID != int32(ticket.SrcID) {
		return c.JSON(http.StatusForbidden, RequestError{
			Error: fmt.Errorf("wrong TvmID"),
		})
	}
	now := time.Now()
	task := Task{
		LicenseNumber:    form.LicenseNumber,
		LicenseIssueDate: issueDate,
		CheckNextTime:    &now,
		Callback:         form.Callback,
		CallbackData:     form.CallbackData,
		CallbackURL:      form.CallbackURL,
		Priority:         10,
		CheckTries:       5,
	}
	if form.Priority != nil {
		task.Priority = *form.Priority
	}
	if form.CheckTries != nil && *form.CheckTries > 0 {
		task.CheckTries = *form.CheckTries
	}
	if err := v.tasks.Create(c.Request().Context(), &task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusCreated, task)
}

const serviceTicketKey = "AuthServiceTicket"

func (v *View) extractServiceTicket(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		ticketString := c.Request().Header.Get("X-Ya-Service-Ticket")
		ticket, err := v.tvm.CheckServiceTicket(
			c.Request().Context(), ticketString,
		)
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusUnauthorized)
		}
		c.Set(serviceTicketKey, ticket)
		return next(c)
	}
}

// Register register API handlers.
func (v *View) Register(g *echo.Group) {
	g.Use(v.extractServiceTicket)
	g.POST("/request", v.request)
}

// NewView creates a new instance of View.
func NewView(c *core.Core, cfg Config) *View {
	db, ok := c.DBs[c.Config.CoreDB]
	if !ok {
		panic("database not found")
	}
	tvm, ok := c.TVMs[c.Config.BB.Source]
	if !ok {
		panic("database not found")
	}
	return &View{
		cfg:   cfg,
		tvm:   tvm,
		tasks: NewTaskStore(db, c.Logger("licensechecks")),
	}
}
