package realtime

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/zootopia/library/go/db"
	"a.yandex-team.ru/zootopia/library/go/db/events"

	bm "a.yandex-team.ru/drive/analytics/goback/models"
	bmt "a.yandex-team.ru/drive/analytics/goback/models/tags"
	dm "a.yandex-team.ru/zootopia/analytics/drive/models"
)

// Config represents realtime watcher configuration.
type Config struct {
	// DB contains name of events DB connection.
	DB string `json:"db"`
	// StateDB contains name of state DB connection.
	StateDB string `json:"state_db"`
	// CacheDB contains name of cache DB connection.
	CacheDB string `json:"cache_db"`
	// BackendDB contains name of backend DB connection.
	BackendDB string `json:"backend_db"`
}

type Watcher struct {
	// core contains core.
	core *core.Core
	// db contains database.
	db *gosql.DB
	// stateDB contains state database.
	//
	// This database is used for storing watchers state.
	stateDB *gosql.DB
	// backendDB contains backend database.
	backendDB *gosql.DB
	// logger contains logger.
	logger log.Logger
	// userTags contains UserTagStore.
	userTags *bm.UserTagStore
}

func (s *Watcher) Start() {
	s.core.StartClusterDaemon("realtime_watcher", s.daemon)
}

var errTooManyRows = fmt.Errorf("too many rows")

func (s *Watcher) daemon(ctx context.Context) error {
	s.logger.Info("Daemon started")
	defer s.logger.Warn("Daemon stopped")
	consumerName := "realtime_user"
	consumer, err := s.newConsumer(
		consumerName, events.NewStore(
			dm.UserEvent{}, "history_event_id",
			"drive_user_data_history", db.Postgres,
		),
	)
	if err != nil {
		return err
	}
	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-ctx.Done():
			return nil
		case <-ticker.C:
			for repeat := true; repeat; {
				select {
				case <-ctx.Done():
					return nil
				default:
					if err := gosql.WithTxContext(
						ctx, s.backendDB, nil, func(tx *sql.Tx) error {
							if err := s.consumeUserEvents(
								tx, consumer, consumerName,
							); err != nil {
								if err == errTooManyRows {
									return nil
								}
								return err
							}
							// There are no many rows. We should stop consumer.
							repeat = false
							return nil
						},
					); err != nil {
						return err
					}
				}
			}
		}
	}
}

func (s *Watcher) consumeUserEvents(
	tx *sql.Tx, consumer events.Consumer, consumerName string,
) error {
	rowsConsumed := 0
	err := gosql.WithTx(s.db, func(chTx *sql.Tx) error {
		stmt, err := chTx.Prepare(
			`INSERT INTO "realtime_user_status_event" ` +
				`("time", "user_id", "from_status",` +
				` "status", "origin") ` +
				`VALUES (?, ?, ?, ?, ?)`,
		)
		if err != nil {
			return err
		}
		return consumer.ConsumeEvents(
			tx,
			func(e events.Event) error {
				if rowsConsumed++; rowsConsumed > 250 {
					return errTooManyRows
				}
				return s.onUserEvent(stmt, e.(dm.UserEvent))
			},
		)
	})
	if rowsConsumed > 0 {
		if err := s.updateConsumer(
			consumerName, consumer.BeginEventID(),
		); err != nil {
			return err
		}
	}
	return err
}

type UserStatusEvent struct {
	Time       int64
	UserID     uuid.UUID
	FromStatus string
	Status     string
	Origin     string
}

func (s *Watcher) onUserEvent(stmt *sql.Stmt, e dm.UserEvent) error {
	status, err := s.getUserLastStatus(e.ID, string(e.Status))
	if err != nil {
		return err
	}
	if string(e.Status) == status {
		return nil
	}
	row := UserStatusEvent{
		Time:       e.HistoryTimestamp,
		UserID:     e.ID,
		FromStatus: status,
		Status:     string(e.Status),
	}
	tags, err := s.userTags.FindByUser(e.ID)
	if err != nil {
		return err
	}
	for _, tag := range tags {
		if tag.Tag == "user_origin" {
			var data bmt.UserOriginTagData
			if err := tag.ScanData(&data); err != nil {
				s.logger.Error(
					"Unable to parse tag data",
					log.String("tag_id", tag.ID.String()),
				)
				break
			}
			row.Origin = data.Origin
			break
		}
	}
	if _, err := stmt.Exec(
		row.Time, row.UserID, row.FromStatus, row.Status, row.Origin,
	); err != nil {
		return err
	}
	if err := s.setUserLastStatus(e.ID, string(e.Status)); err != nil {
		return err
	}
	return nil
}

func (s *Watcher) getUserLastStatus(
	id uuid.UUID, status string,
) (string, error) {
	row := s.stateDB.QueryRow(
		`SELECT "status" FROM "realtime_user_state" WHERE "user_id" = $1`,
		id,
	)
	var lastStatus string
	if err := row.Scan(&lastStatus); err != nil {
		if err == sql.ErrNoRows {
			_, err := s.stateDB.Exec(
				`INSERT INTO "realtime_user_state"`+
					` ("user_id", "status") VALUES ($1, $2)`,
				id, status,
			)
			if err != nil {
				return "", err
			}
			return status, nil
		}
		return "", err
	}
	return lastStatus, nil
}

func (s *Watcher) setUserLastStatus(
	id uuid.UUID, status string,
) error {
	res, err := s.stateDB.Exec(
		`UPDATE "realtime_user_state" SET "status" = $1 WHERE "user_id" = $2`,
		status, id,
	)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return fmt.Errorf("affected rows: %d", affected)
	}
	return nil
}

func (s *Watcher) newConsumer(
	name string, store events.ROStore,
) (events.Consumer, error) {
	row := s.stateDB.QueryRow(
		`SELECT "begin_id" FROM "consumer_state" WHERE "name" = $1`,
		name,
	)
	var beginID int64
	if err := row.Scan(&beginID); err != nil {
		return nil, err
	}
	return events.NewOrderedConsumer(store, beginID), nil
}

func (s *Watcher) updateConsumer(name string, beginID int64) error {
	res, err := s.stateDB.Exec(
		`UPDATE "consumer_state" SET "begin_id" = $1 WHERE "name" = $2`,
		beginID, name,
	)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return fmt.Errorf("consumer %q not found", name)
	}
	return nil
}

func NewWatcher(c *core.Core, cfg Config) *Watcher {
	db, ok := c.DBs[cfg.DB]
	if !ok {
		panic("DB not found")
	}
	stateDB, ok := c.DBs[cfg.StateDB]
	if !ok {
		panic("state DB does not found")
	}
	backendDB, ok := c.DBs[cfg.BackendDB]
	if !ok {
		panic("backend DB does not found")
	}
	return &Watcher{
		core:      c,
		db:        db,
		stateDB:   stateDB,
		backendDB: backendDB,
		logger:    c.Logger("realtime_watcher"),
		userTags:  bm.NewUserTagStore(backendDB.DB),
	}
}
