package tagshistory

import (
	"context"
	"math"
	"net/http"

	"github.com/gofrs/uuid"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
)

// View represents view for users info.
type View struct {
	core *core.Core
	yt   yt.Client
}

// NewView creates new instance of View.
func NewView(c *core.Core) *View {
	return &View{core: c, yt: c.YT}
}

// GetTagsHistoryResultRow represents result row.
type GetTagsHistoryResultRow struct {
	models.TagEvent
	// Data contains additional data about tag.
	//
	// Content depends on tag but commonly this is a map.
	Data models.TagData `json:"Data"`
	// Snapshot contains snapshot.
	Snapshot models.Snapshot `json:"Snapshot"`
	// Name contains display name of tag.
	Name string `json:"Name"`
	// Comment contains display comment of tag.
	Comment string `json:"Comment"`
	// Type contains type of tag.
	Type string `json:"Type"`
}

// GetTagsHistoryResultUser represents user information from result.
type GetTagsHistoryResultUser struct {
	ID       uuid.UUID `json:"ID"`
	Username string    `json:"Username"`
}

// GetTagsHistoryResult represents result from GetUserTagsHistory.
type GetTagsHistoryResult struct {
	// Rows contains history rows.
	Rows []GetTagsHistoryResultRow `json:"Rows"`
	// Users contains information about users from rows.
	Users []GetTagsHistoryResultUser `json:"Users,omitempty"`
}

// GetTagsHistoryForm represents GET params for GetUserTagsHistory.
type GetTagsHistoryForm struct {
	// Begin declares low limit for timestamps.
	//
	// All timestamps required to be greater or equal to this value.
	// Do not pass this param if you dont want to limit timestamps.
	// Zero value is equivalent to empty value.
	Begin int64 `query:"Begin"`
	// End declares high limit for timestamps.
	//
	// All timestamps required to be strictly less than this value.
	// Do not pass this param if you dont want to limit timestamps.
	// Zero value is equivalent to empty value.
	End int64 `query:"End"`
	// WithUsers specifies that backend should return information
	// about users.
	WithUsers bool `query:"WithUsers"`
}

// GetUserTagsHistory returns tags history for specified user.
func (v *View) GetUserTagsHistory(c echo.Context) error {
	return v.getTagsHistory(
		c, "//home/carsharing/production/cache/user_tags_history",
	)
}

// GetCarTagsHistory returns tags history for specified car.
func (v *View) GetCarTagsHistory(c echo.Context) error {
	return v.getTagsHistory(
		c, "//home/carsharing/production/cache/car_tags_history",
	)
}

func (v *View) getTagsHistory(c echo.Context, path ypath.Path) error {
	objectIDStr := c.Param("ObjectID")
	objectID, err := uuid.FromString(objectIDStr)
	if err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	var form GetTagsHistoryForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	var beginTime int64
	if form.Begin != 0 {
		beginTime = form.Begin
	}
	var endTime int64 = math.MaxInt64
	if form.End != 0 {
		endTime = form.End - 1
	}
	table := path.Rich().
		AddRange(ypath.Range{
			Lower: &ypath.ReadLimit{
				Key: []interface{}{objectID, beginTime},
			},
			Upper: &ypath.ReadLimit{
				Key: []interface{}{objectID, endTime},
			},
		})
	in, err := v.yt.ReadTable(context.TODO(), table, nil)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	defer func() {
		if err := in.Close(); err != nil {
			c.Logger().Warn(err)
		}
	}()
	user, ok := c.Get(core.AuthDriveUser).(core.DriveUser)
	if !ok {
		return c.NoContent(http.StatusUnauthorized)
	}
	observable, err := v.getObservableTags(user.ID.String())
	if err != nil || len(observable) == 0 {
		if err != nil {
			c.Logger().Warn(err)
		}
		return c.NoContent(http.StatusForbidden)
	}
	descriptions, err := v.getTagDescriptions()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	result := GetTagsHistoryResult{}
	uniqueUserIDs := map[string]struct{}{}
	for in.Next() {
		var row models.TagEvent
		if err := in.Scan(&row); err != nil {
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		if _, ok := observable[string(row.Tag.Tag)]; !ok {
			continue
		}
		uniqueUserIDs[row.HistoryUserID] = struct{}{}
		uniqueUserIDs[string(row.Performer)] = struct{}{}
		record := GetTagsHistoryResultRow{TagEvent: row}
		if description, ok := descriptions[string(row.Tag.Tag)]; ok {
			if data, err := row.Tag.ParseData(description.Type); err != nil {
				c.Logger().Warn(err)
			} else {
				record.Data = data
			}
			if snapshot, err := row.Tag.ParseSnapshot(); err != nil {
				c.Logger().Warn(err)
			} else {
				record.Snapshot = snapshot
			}
			record.Name = description.Name
			record.Comment = description.Comment
			record.Type = description.Type
		}
		result.Rows = append(result.Rows, record)
	}
	if err := in.Err(); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if form.WithUsers {
		var err error
		if result.Users, err = v.getUniqueUsers(uniqueUserIDs); err != nil {
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
	}
	return c.JSON(http.StatusOK, result)
}

func (v *View) getUniqueUsers(
	uniqueUserIDs map[string]struct{},
) ([]GetTagsHistoryResultUser, error) {
	if len(uniqueUserIDs) == 0 {
		return nil, nil
	}
	var userIDs []uuid.UUID
	for id := range uniqueUserIDs {
		if userID, err := uuid.FromString(id); err == nil {
			userIDs = append(userIDs, userID)
		}
	}
	users, err := v.core.DriveUsers.FindByIDs(userIDs)
	if err != nil {
		return nil, err
	}
	var result []GetTagsHistoryResultUser
	for _, user := range users {
		result = append(result, GetTagsHistoryResultUser{
			ID:       user.ID,
			Username: user.Username,
		})
	}
	return result, nil
}

func (v *View) getObservableTags(userID string) (map[string]struct{}, error) {
	perms, err := v.core.Drive.GetUserPermissions(userID)
	if err != nil {
		return nil, err
	}
	visibleTags := make(map[string]struct{})
	for _, tag := range perms.ObserveTags {
		visibleTags[tag] = struct{}{}
	}
	return visibleTags, nil
}

func (v *View) getTagDescriptions() (map[string]api.TagDescription, error) {
	tags, err := v.core.Drive.GetTagDescriptions()
	if err != nil {
		return nil, err
	}
	result := make(map[string]api.TagDescription)
	for _, tag := range tags {
		result[tag.Tag] = tag
	}
	return result, nil
}

// Register registers service handlers.
func (v *View) Register(group *echo.Group) {
	group.GET("/cars/:ObjectID", v.GetCarTagsHistory)
	group.GET("/users/:ObjectID", v.GetUserTagsHistory)
}
