#pragma once

#include "localization.h"
#include "notifier.h"
#include "settings.h"

#include <library/cpp/logger/global/global.h>
#include <library/cpp/mediator/messenger.h>
#include <library/cpp/tvmauth/checked_user_ticket.h>

#include <rtline/library/executor/abstract/context.h>
#include <rtline/library/storage/structured.h>

namespace NRTLine {
    class IVersionedStorage;
}

namespace NDrive {
    class TSupportAIClient;
    class TTaxiPromocodesClient;
}

namespace NTvmAuth {
    class TTvmClient;
}

namespace NOpenssl {
    class IAbstractCipher;
};

class IAuthModuleConfig;
class IExternalAccessTokensManager;
class IRequestProcessorConfig;
class IThreadPool;
class IUserDevicesManager;
class TBackgroundProcessesManager;
class TClckClient;
class TCovidPassClient;
class TDatasyncClient;
class TLPMClient;
class TRTBackgroundManager;
class TRTLineAPI;
class TS3Client;
class TStartrekClient;
class TSettingsDB;
class TTaskExecutor;
class TVersionedKey;
struct THttpStatusManagerConfig;
class IReplyContext;

class TServerStartedMessage: public NMessenger::IMessage {
    R_FIELD(TSet<TString>, SequentialTableNames);

public:
    TServerStartedMessage() = default;
};

class IRTLineAPIStorage {
public:
    virtual const TRTLineAPI* GetRTLineAPI(const TString& name) const = 0;
    virtual TVector<TString> ListRTLineAPIs() const = 0;
    virtual ~IRTLineAPIStorage() {}
};

class IServerBase
    : public IDistributedTaskContext
    , public NDrive::INotifiersStorage
    , public IRTLineAPIStorage
{
public:
    IServerBase();
    ~IServerBase();

    virtual const TString& GetCType() const = 0;
    virtual const TString& GetServiceName() const = 0;
    virtual const ISettings& GetSettings() const = 0;
    virtual const TSettingsDB* GetSettingsManager() const = 0;
    virtual bool HasSettings() const = 0;
    virtual const NDrive::INotifiersManager* GetNotifiersManager() const = 0;
    virtual TAtomicSharedPtr<NDrive::INotifier> GetNotifier(const TString& name) const = 0;
    virtual TSet<TString> GetNotifierNames() const = 0;
    virtual const TBackgroundProcessesManager* GetBackgroundProcessesManager() const = 0;
    virtual const TRTBackgroundManager* GetRTBackgroundManager() const = 0;
    virtual TMaybe<TString> GetOrCreateUserId(const TString& username, const TString& operatorUserId) const = 0;
    virtual TMaybe<TString> GetRtBackgroundRobotUserId(const TString& name) const = 0;
    virtual TAtomicSharedPtr<NStorage::IDatabase> GetDatabase(const TString& name) const = 0;
    virtual TVector<TString> GetDatabaseNames() const = 0;
    virtual TAtomicSharedPtr<TDatasyncClient> GetDatasyncClient() const = 0;
    virtual IThreadPool* GetRequestsHandler(const TString& handlerName) const = 0;
    virtual TAtomicSharedPtr<NTvmAuth::TTvmClient> GetTvmClient(ui32 selfClientId, NTvmAuth::EBlackboxEnv env = NTvmAuth::EBlackboxEnv::Prod) const = 0;
    virtual TVector<TString> GetServerProcessors() const = 0;
    virtual const TTaskExecutor& GetGlobalTaskExecutor() const = 0;
    virtual const TS3Client* GetMDSClient() const = 0;
    virtual const TStartrekClient* GetStartrekClient() const = 0;
    virtual const NDrive::TSupportAIClient* GetSupportAIClient() const = 0;
    virtual const TClckClient* GetClckClient() const = 0;
    virtual const THttpStatusManagerConfig& GetHttpStatusManagerConfig() const = 0;
    virtual const IUserDevicesManager* GetUserDevicesManager() const {
        return nullptr;
    }
    virtual TAtomicSharedPtr<NRTLine::IVersionedStorage> GetVersionedStorage(const TString& storageName) const = 0;
    virtual const IAuthModuleConfig* GetAuthModuleInfo(const TString& authModuleName) const = 0;
    virtual TAtomicSharedPtr<IRequestProcessorConfig> GetProcessorInfo(const TVersionedKey& procKey) const = 0;
    virtual const ILocalization* GetLocalization() const = 0;
    virtual const NOpenssl::IAbstractCipher* GetCipher(const TString& cipherName) const = 0;
    virtual const IExternalAccessTokensManager* GetExternalAccessTokensManager() const = 0;
    virtual const TLPMClient* GetLPMClient() const = 0;
    // GetVersionedKey should perform routing for specified request and modify IReplyContext if required.
    virtual TVersionedKey GetVersionedKey(IReplyContext& context) const = 0;
    template <class T>
    const T* GetLocalizationAs() const {
        return dynamic_cast<const T*>(GetLocalization());
    }

    template <class T>
    const T* GetAs() const {
        return dynamic_cast<const T*>(this);
    }

    template <class T>
    const T* GetAsPtrSafe() const {
        const T* result = dynamic_cast<const T*>(this);
        CHECK_WITH_LOG(result);
        return result;
    }

    template <class T>
    const T& GetAsSafe() const {
        return *VerifyDynamicCast<const T*>(this);
    }
};

namespace NDrive {
    bool HasServer();
    const IServerBase& GetServer();
    template <class T>
    const T& GetServerAs() {
        return GetServer().GetAsSafe<T>();
    }

    void RegisterServer(const IServerBase& server);
    void DeregisterServer();
}
