#include "localization.h"

#include <library/cpp/langs/langs.h>

#include <rtline/util/types/cast.h>

#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/printf.h>
#include <util/string/subst.h>

#include <cmath>

ELocalization DefaultLocale = ELocalization::Rus;

namespace {
    class TScanProcessor: public NJson::IScanCallback {
    private:
        const ILocalization* Localization = nullptr;
        ELocalization Locale;

    public:
        bool Do(const TString& /*path*/, NJson::TJsonValue* /*parent*/, NJson::TJsonValue& value) override {
            if (value.IsString()) {
                value = Localization->ApplyResources(value.GetString(), Locale);
            }
            return true;
        }

        TScanProcessor(const ILocalization* localization, ELocalization locale)
            : Localization(localization)
            , Locale(locale)
        {
        }
    };
}

TString ILocalization::ApplyResources(const TString& value, ELocalization locale) const {
    const auto action = [](const TString& valueStr) {
        return valueStr;
    };
    return ApplyResourcesImpl(value, locale, action);
}

void ILocalization::ApplyResourcesForJson(NJson::TJsonValue& value, ELocalization locale) const {
    TScanProcessor scanner(this, locale);
    value.Scan(scanner);
}

TString ILocalization::ApplyResourcesForJson(const TString& value, ELocalization locale) const {
    const auto action = [](const TString& valueStr) {
        return NJson::TJsonValue(valueStr).GetStringRobust();
    };
    return ApplyResourcesImpl(value, locale, action);
}

template <class TAction>
TString ILocalization::ApplyResourcesImpl(const TString& value, ELocalization locale, const TAction& stringProcessor) const {
    static const TString resourceMarker = "(resource:";
    size_t idx = value.find(resourceMarker);
    if (idx == TString::npos) {
        return value;
    }
    TStringBuf strValue = value;
    TString text(strValue.Head(idx));
    while (idx != TString::npos) {
        strValue.Skip(idx + resourceMarker.size());
        TStringBuf resName;
        TStringBuf nextValue;
        strValue.TrySplit(')', resName, nextValue);
        TMaybe<TString> resourceStr = GetLocalString(locale, TString(resName));
        if (resourceStr) {
            text += stringProcessor(*resourceStr);
        }
        strValue = nextValue;
        idx = strValue.find(resourceMarker);
        text += strValue.Head(idx);
    };
    return text;
}

TString ILocalization::GetLocalString(ELocalization locale, const TString& resourceId, TMaybe<TString> defaultValue) const {
    TMaybe<TString> result = GetLocalStringImpl(locale, resourceId);
    if (!result) {
        if (defaultValue && locale == LegacyLocale) {
            return std::move(*defaultValue);
        } else {
            return resourceId;
        }
    } else {
        return *result;
    }
}

TString ILocalization::FormatNumeral(const ui32 value, ELocalization locale, bool withValue, const TString& separator, const TString& localizationKey, TMaybe<TString> default0, TMaybe<TString> default1, TMaybe<TString> default2) const {
    const ui32 idx = value % 10;
    const ui32 idx100 = value % 100;
    TString prefix = withValue ? ToString(value) + separator : "";
    if (idx == 0 || idx >= 5 || (idx100 >= 10 && idx100 <= 20)) {
        return prefix + GetLocalString(locale, localizationKey + "(0)", default0);
    } else if (idx == 1) {
        return prefix + GetLocalString(locale, localizationKey + "(1)", default1);
    }
    return prefix + GetLocalString(locale, localizationKey + "(2)", default2);
}

TString ILocalization::DaysFormat(TDuration d, ELocalization locale) const {
    if (d < TDuration::Days(1)) {
        return HoursFormat(d, locale);
    } else {
        return FormatNumeral(d.Days(), locale, true, " ", "units.full.days", "дней", "день", "дня");
    }
}

TString ILocalization::WeeksFormat(TDuration d, ELocalization locale) const {
    if (d < TDuration::Days(7)) {
        return DaysFormat(d, locale);
    } else {
        return FormatNumeral(d.Days() / 7, locale, true, " ", "units.full.weeks", "недель", "неделя", "недели");
    }
}

TString ILocalization::MonthsFormat(TDuration d, ELocalization locale) const {
    if (d < TDuration::Days(30)) {
        return WeeksFormat(d, locale);
    } else {
        return FormatNumeral(d.Days() / 30, locale, true, " ", "units.full.months", "месяцев", "месяц", "месяца");
    }
}

TString ILocalization::FormatPrice(const ELocalization& localizationId, const ui32 price, const std::initializer_list<TString>& units, TStringBuf separator) const {
    TStringBuilder result;
    result << ::ToString(price / 100);
    if (price % 10) {
        result << ',' << Sprintf("%02d", price % 100);
    } else if (price % 100) {
        result << ',' << Sprintf("%d", (price % 100) / 10);
    }
    if (units.size() > 0) {
        if (separator) {
            result << separator;
        } else {
            result << " ";
        }
    }
    for (auto&& i : units) {
        result << GetLocalString(localizationId, i, i);
    }
    return result;
}

TString ILocalization::DistanceFormatKm(const ELocalization localizationId, const double km, bool round) const {
    if (km < 0) {
        return "0 " + GetLocalString(localizationId, "units.short.km", "км");
    }

    if (km < 1) {
        return Sprintf("%d ", (int)(km * 1000)) + GetLocalString(localizationId, "units.short.meters", "м");
    } else if (round) {
        return Sprintf("%d %s", (int)std::round(km), GetLocalString(localizationId, "units.short.km", "км").c_str());
    } else {
        return Sprintf("%0.1f ", km) + GetLocalString(localizationId, "units.short.km", "км");
    }
}

TString ILocalization::FormatDuration(ELocalization localizationId, TDuration d, bool withSeconds, bool allowEmpty, bool withMilliseconds) const {
    const ui32 days = d.Days();
    const ui32 hours = d.Hours() % 24;
    const ui32 minutes = d.Minutes() % 60;
    const ui32 seconds = d.Seconds() % 60;
    const ui32 milliseconds = d.MilliSeconds() % 1000;
    TStringBuilder result;
    if (days) {
        result << days << " " << GetLocalString(localizationId, "units.short.day", "д");
    }
    if (hours) {
        if (!result.empty()) {
            result << ' ';
        }
        result << hours << " " << GetLocalString(localizationId, "units.short.hour", "ч");
    }
    if (minutes) {
        if (!result.empty()) {
            result << ' ';
        }
        result << minutes << " " << GetLocalString(localizationId, "units.short.minutes", "мин");
    }
    if (withSeconds && seconds) {
        if (!result.empty()) {
            result << ' ';
        }
        result << seconds;
        if (withMilliseconds) {
            result << '.';
            if (milliseconds % 10) {
                result << milliseconds;
            } else if (milliseconds % 100) {
                result << milliseconds / 10;
            } else {
                result << milliseconds / 100;
            }
        }
        result << " " << GetLocalString(localizationId, "units.short.seconds", "с");
    }
    if (result.empty() && !allowEmpty) {
        result << d.Seconds() << " " << GetLocalString(localizationId, "units.short.seconds", "с");
    }
    return result;
}

TString ILocalization::FormatInstant(const ELocalization localizationId, const TInstant t) const {
    TStringBuilder sb;
    sb << ILocalization::FormatTimeOfTheDay(localizationId, t, TDuration::Zero()) << "\u00A0";
    return sb << FormatMonthDay(localizationId, t);
}

TString ILocalization::FormatMonthDay(const ELocalization localizationId, const TInstant t) const {
    tm timeInfo;
    t.GmTime(&timeInfo);
    TStringBuilder sb;
    sb << timeInfo.tm_mday << "\u00A0";
    switch (timeInfo.tm_mon) {
        case 0:
            return sb << GetLocalString(localizationId, "months.full.jan", "января");
        case 1:
            return sb << GetLocalString(localizationId, "months.full.feb", "февраля");
        case 2:
            return sb << GetLocalString(localizationId, "months.full.mar", "марта");
        case 3:
            return sb << GetLocalString(localizationId, "months.full.apr", "апреля");
        case 4:
            return sb << GetLocalString(localizationId, "months.full.may", "мая");
        case 5:
            return sb << GetLocalString(localizationId, "months.full.jun", "июня");
        case 6:
            return sb << GetLocalString(localizationId, "months.full.jul", "июля");
        case 7:
            return sb << GetLocalString(localizationId, "months.full.aug", "августа");
        case 8:
            return sb << GetLocalString(localizationId, "months.full.sep", "сентября");
        case 9:
            return sb << GetLocalString(localizationId, "months.full.oct", "октября");
        case 10:
            return sb << GetLocalString(localizationId, "months.full.nov", "ноября");
        case 11:
            return sb << GetLocalString(localizationId, "months.full.dec", "декабря");
        default:
            return sb;
    }
}

TString ILocalization::FormatInstantWithYear(const ELocalization localizationId, const TInstant t) const {
    tm timeInfo;
    t.GmTime(&timeInfo);
    TStringBuilder sb;
    sb << FormatInstant(localizationId, t) << "\u00A0" << (1900 + timeInfo.tm_year);
    return sb;
}

TString ILocalization::FormatMonthDayWithYear(const ELocalization localizationId, const TInstant t, const TDuration shift) const {
    TInstant shifted = t + shift;
    tm timeInfo;
    shifted.GmTime(&timeInfo);
    TStringBuilder sb;
    sb << FormatMonthDay(localizationId, shifted) << "\u00A0" << (1900 + timeInfo.tm_year);
    return sb;
}

TString ILocalization::FormatTimeOfTheDay(const ELocalization localizationId, const TInstant timestamp, const TDuration shift) const {
    Y_UNUSED(localizationId);
    TInstant shifted = timestamp + shift;
    return shifted.FormatGmTime("%H:%M");
}

TString ILocalization::HoursFormat(TDuration d, ELocalization locale) const {
    if (d < TDuration::Hours(1)) {
        return MinutesFormat(d, locale);
    } else {
        return FormatNumeral(d.Hours(), locale, true, "\u00A0", "units.full.hours", "часов", "час", "часа");
    }
}

TString ILocalization::MinutesFormat(TDuration d, ELocalization locale) const {
    return FormatNumeral(d.Minutes(), locale, true, "\u00A0", "units.full.minutes", "минут", "минута", "минуты");
}

TString ILocalization::FormatBonusRubles(const ELocalization localizationId, const ui32 amount, const bool withAmount) const {
    return FormatNumeral(amount, localizationId, withAmount, " ", "units.full.bonus_rubles", "бонусных рублей", "бонусный рубль", "бонусных рубля");
}

TString ILocalization::FormatRubles(const ELocalization localizationId, const ui32 amount, const bool withAmount) const {
    return FormatNumeral(amount, localizationId, withAmount, " ", "units.full.rubles", "рублей", "рубль", "рубля");
}

TString ILocalization::FormatFreeWaitTime(const ELocalization localizationId, const TDuration freeWaitTime) const {
    if (freeWaitTime == TDuration::Zero()) {
        return GetLocalString(localizationId, "offer_descriptions.elements.no_free_time", "Нет бесплатного ожидания");
    }
    return FormatDuration(localizationId, freeWaitTime) + GetLocalString(localizationId, "offer_descriptions.elements.of_waiting", " ожидания");
}

TString ILocalization::FormatDelegatedStandartOfferTitle(const ELocalization localizationId, const TString& offerName, const ui32 priceRiding) const {
    TString pattern = GetLocalString(localizationId, "offer_descriptions.standart.short.header", "«‎_OfferName_»‎   ∙   _PriceRiding_ ₽/мин");
    SubstGlobal(pattern, "_OfferName_", offerName);
    SubstGlobal(pattern, "_PriceRiding_", FormatPrice(localizationId, priceRiding));
    return pattern;
}

TString ILocalization::FormatDelegatedStandartOfferBody(const ELocalization localizationId, const ui32 priceParking, const TDuration freeWaitTime) const {
    TString pattern = GetLocalString(localizationId, "offer_descriptions.standart.short.body", "Ожидание — _PriceParking_ ₽/мин\n_FreeWaitTime_");
    SubstGlobal(pattern, "_PriceParking_", FormatPrice(localizationId, priceParking));
    SubstGlobal(pattern, "_FreeWaitTime_", FormatFreeWaitTime(localizationId, freeWaitTime));
    return pattern;
}

TString ILocalization::FormatDelegatedPackOfferTitle(const ELocalization localizationId, const TString& offerName) const {
    TString pattern = GetLocalString(localizationId, "offer_descriptions.pack.short.header", "«‎_OfferName_»");
    SubstGlobal(pattern, "_OfferName_", offerName);
    return pattern;
}

TString ILocalization::FormatDelegatedPackOfferBody(const ELocalization localizationId, const ui32 distance, const TDuration duration, const ui32 priceParking, const TDuration freeWaitTime, const ui32 overtimePrice, const ui32 overrunPrice, const TString& insuranceType) const {
    TString pattern;

    if (distance && duration > TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.pack_dist_time.short.body", "_MileageLimit_   ∙   _PackDuration_\n_FreeWaitTime_\n_InsuranceType_");
    } else if (!distance && duration > TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.pack_time.short.body", "_MileageLimit_   ∙   _PackDuration_\n_OverrunPrice_ ₽/км\n_FreeWaitTime_\n_InsuranceType_");
    } else if (distance && duration == TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.pack_dist.short.body", "_MileageLimit_   ∙   _PackDuration_\n_OvertimePrice_ ₽/мин\n_FreeWaitTime_\n_InsuranceType_");
    } else {
        pattern = GetLocalString(localizationId, "offer_descriptions.pack_dist.short.body", "_MileageLimit_   ∙   _PackDuration_\n_OvertimePrice_ ₽/мин\n_OverrunPrice_ ₽/км\n_FreeWaitTime_\n_InsuranceType_");
    }

    SubstGlobal(pattern, "_MileageLimit_", ToString(distance) + " км");
    SubstGlobal(pattern, "_PackDuration_", FormatDuration(localizationId, duration));
    SubstGlobal(pattern, "_PriceParking_", FormatPrice(localizationId, priceParking));
    SubstGlobal(pattern, "_FreeWaitTime_", FormatFreeWaitTime(localizationId, freeWaitTime));
    SubstGlobal(pattern, "_OvertimePrice_", FormatPrice(localizationId, overtimePrice));
    SubstGlobal(pattern, "_OverrunPrice_", FormatPrice(localizationId, overrunPrice));
    SubstGlobal(pattern, "_InsuranceType_", GetLocalString(localizationId, "offer_descriptions.insurance_type." + insuranceType, TString{}));
    return pattern;
}

TString ILocalization::FormatDelegatedIntercityOfferBody(const ELocalization localizationId, const ui32 distance, const TDuration duration, const ui32 priceParking, const TDuration freeWaitTime, const ui32 overtimePrice, const ui32 overrunPrice, const TString& insuranceType) const {
    TString pattern;

    if (distance && duration > TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.intercity_dist_time.short.body", "_MileageLimit_   ∙   _PackDuration_\nОжидание — _PriceParking_ ₽/мин\n_FreeWaitTime_\n_InsuranceType_");
    } else if (!distance && duration > TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.intercity_time.short.body", "_MileageLimit_   ∙   _PackDuration_\nОжидание — _PriceParking_ ₽/мин\n_OverrunPrice_ ₽/км\n_FreeWaitTime_\n_InsuranceType_");
    } else if (distance && duration == TDuration::Zero()) {
        pattern = GetLocalString(localizationId, "offer_descriptions.intercity_dist.short.body", "_MileageLimit_   ∙   _PackDuration_\n_OvertimePrice_ ₽/мин\n_FreeWaitTime_\n_InsuranceType_");
    } else {
        pattern = GetLocalString(localizationId, "offer_descriptions.intercity_dist.short.body", "_MileageLimit_   ∙   _PackDuration_\n_OvertimePrice_ ₽/мин\n_OverrunPrice_ ₽/км\n_FreeWaitTime_\n_InsuranceType_");
    }

    SubstGlobal(pattern, "_MileageLimit_", ToString(distance) + " км");
    SubstGlobal(pattern, "_PackDuration_", FormatDuration(localizationId, duration));
    SubstGlobal(pattern, "_PriceParking_", FormatPrice(localizationId, priceParking));
    SubstGlobal(pattern, "_FreeWaitTime_", FormatFreeWaitTime(localizationId, freeWaitTime));
    SubstGlobal(pattern, "_OvertimePrice_", FormatPrice(localizationId, overtimePrice));
    SubstGlobal(pattern, "_OverrunPrice_", FormatPrice(localizationId, overrunPrice));
    SubstGlobal(pattern, "_InsuranceType_", GetLocalString(localizationId, "offer_descriptions.insurance_type." + insuranceType, TString{}));
    return pattern;
}

TString ILocalization::FormatServiceFuelingCost(const ELocalization localizationId, const ui32 cost) const {
    TString pattern = GetLocalString(localizationId, "service_delegation.fueling.cost", "_PriceFueling_ ₽/л.");
    SubstGlobal(pattern, "_PriceFueling_", FormatPrice(localizationId, cost));
    return pattern;
}

TString ILocalization::FormatDelegationBodyWithPackOffer(const ELocalization localizationId, const TString& offerName, const ui32 remainingDistance, const TDuration remainingTime) const {
    TString pattern = GetLocalString(localizationId, "offer_descriptions.pack.delegation.with_offer", "Тариф «‎_OfferName_»\n_MileageLimit_   ∙   _PackDuration_");
    SubstGlobal(pattern, "_OfferName_", offerName);
    SubstGlobal(pattern, "_MileageLimit_", ToString(remainingDistance) + " км");
    SubstGlobal(pattern, "_PackDuration_", FormatDuration(localizationId, remainingTime));
    return pattern;
}

TString ILocalization::FormatDelegationBodyWithoutPackOffer(const ELocalization localizationId, const ui32 remainingDistance, const TDuration remainingTime) const {
    TString pattern = GetLocalString(localizationId, "offer_descriptions.pack.delegation.without_offer", "Тогда _MileageLimit_ и _PackDuration_ сгорят");
    SubstGlobal(pattern, "_MileageLimit_", ToString(remainingDistance) + " км");
    SubstGlobal(pattern, "_PackDuration_", FormatDuration(localizationId, remainingTime));
    return pattern;
}

TString ILocalization::FormatIncomingDelegationMessage(const ELocalization localizationId, const TString& delegatorName, const TString& modelName) const {
    TString pattern = GetLocalString(localizationId, "delegation.p2p.incoming_push_message", "_DelegatorName_ желает передать вам управление _ModelName_");
    SubstGlobal(pattern, "_DelegatorName_", delegatorName);
    SubstGlobal(pattern, "_ModelName_", modelName);
    return pattern;
}

template <>
ELanguage enum_cast<ELanguage, ELocalization>(ELocalization value) {
    switch (value) {
    case ELocalization::Cze:    return LANG_CZE;
    case ELocalization::Eng:    return LANG_ENG;
    case ELocalization::Ger:    return LANG_GER;
    case ELocalization::Rus:    return LANG_RUS;
    }
}
