#pragma once

#include <library/cpp/json/json_value.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>

enum class ELocalization {
    Cze /* "cze", "cs" */,
    Eng /* "eng", "en" */,
    Ger /* "ger", "de" */,
    Rus /* "rus", "ru" */,
};

extern ELocalization DefaultLocale;
constexpr auto LegacyLocale = ELocalization::Rus;

class ILocalization {
public:
    virtual ~ILocalization() = default;

    TString GetLocalString(ELocalization locale, const TString& resourceId, TMaybe<TString> defaultValue = {}) const;

    TString ApplyResources(const TString& value, ELocalization locale) const;
    TString ApplyResourcesForJson(const TString& value, ELocalization locale) const;
    void ApplyResourcesForJson(NJson::TJsonValue& value, ELocalization locale) const;

    TString FormatNumeral(const ui32 value, ELocalization locale, bool withValue, const TString& separator, const TString& localizationKey, TMaybe<TString> default0 = {}, TMaybe<TString> default1 = {}, TMaybe<TString> default2 = {}) const;

    TString FormatPrice(const ELocalization& localizationId, const ui32 price, const std::initializer_list<TString>& units = {}, TStringBuf separator = {}) const;
    TString DistanceFormatKm(const ELocalization localizationId, const double km, bool round = false) const;
    TString FormatDuration(ELocalization locale, TDuration d, bool withSeconds = false, bool allowEmpty = false, bool withMilliseconds = false) const;
    TString FormatInstant(const ELocalization localizationId, const TInstant t) const;
    TString FormatInstantWithYear(const ELocalization localizationId, const TInstant t) const;
    TString FormatMonthDayWithYear(const ELocalization localizationId, const TInstant t, const TDuration shift = TDuration::Hours(3)) const;
    TString FormatMonthDay(const ELocalization localizationId, const TInstant t) const;
    TString FormatTimeOfTheDay(const ELocalization localizationId, const TInstant timestamp, const TDuration shift = TDuration::Hours(3)) const;
    TString FormatBonusRubles(const ELocalization localizationId, const ui32 amount, const bool withAmount = false) const;
    TString FormatRubles(const ELocalization localizationId, const ui32 amount, const bool withAmount = false) const;
    TString FormatFreeWaitTime(const ELocalization localizationId, const TDuration freeWaitTime) const;
    TString FormatIncomingDelegationMessage(const ELocalization localizationId, const TString& delegatorName, const TString& modelName) const;

    TString FormatDelegatedStandartOfferTitle(const ELocalization localizationId, const TString& offerName, const ui32 priceRiding) const;
    TString FormatDelegatedStandartOfferBody(const ELocalization localizationId, const ui32 priceParking, const TDuration freeWaitTime) const;
    TString FormatDelegatedPackOfferTitle(const ELocalization localizationId, const TString& offerName) const;
    TString FormatDelegatedPackOfferBody(const ELocalization localizationId, const ui32 distance, const TDuration duration, const ui32 priceParking, const TDuration freeWaitTime, const ui32 overtimePrice, const ui32 overrunPrice, const TString& insuranceType) const;
    TString FormatDelegatedIntercityOfferBody(const ELocalization localizationId, const ui32 distance, const TDuration duration, const ui32 priceParking, const TDuration freeWaitTime, const ui32 overtimePrice, const ui32 overrunPrice, const TString& insuranceType) const;

    TString FormatDelegationBodyWithPackOffer(const ELocalization localizationId, const TString& offerName, const ui32 remainingDistance, const TDuration remainingTime) const;
    TString FormatDelegationBodyWithoutPackOffer(const ELocalization localizationId, const ui32 remainingDistance, const TDuration remainingTime) const;

    TString FormatServiceFuelingCost(const ELocalization localizationId, const ui32 cost) const;

    TString HoursFormat(TDuration d, ELocalization locale) const;
    TString MinutesFormat(TDuration d, ELocalization locale) const;
    TString DaysFormat(TDuration d, ELocalization locale) const;
    TString WeeksFormat(TDuration d, ELocalization locale) const;
    TString MonthsFormat(TDuration d, ELocalization locale) const;

protected:
    virtual TMaybe<TString> GetLocalStringImpl(ELocalization locale, const TString& resourceId) const = 0;

private:
    template <class TAction>
    TString ApplyResourcesImpl(const TString& value, ELocalization locale, const TAction& stringProcessor) const;
};

class TFakeLocalization: public ILocalization {
public:
    virtual TMaybe<TString> GetLocalStringImpl(ELocalization /*locale*/, const TString& /*resourceId*/) const override {
        return {};
    }
};
