#include "notifier.h"

#include <drive/backend/logging/events.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/logger/global/global.h>

#include <rtline/util/types/messages_collector.h>

#include <util/digest/multi.h>

const ui32 NDrive::INotifierConfig::DefaultMessageLengthLimit = 4096;

bool NDrive::INotifierConfig::DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& errors) {
    return
        NJson::ParseField(info["use_event_log"], UseEventLog, /* required = */ false, errors) &&
        NJson::ParseField(info["message_length_limit"], MessageLengthLimit, /* required = */ false, errors);
}

NJson::TJsonValue NDrive::INotifierConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "use_event_log", UseEventLog);
    NJson::InsertField(result, "message_length_limit", MessageLengthLimit);
    return result;
}

NJson::TJsonValue NDrive::INotifierConfig::GetUserReport(const IServerBase& /*server*/) const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "message_length_limit", MessageLengthLimit);
    return result;
}

NDrive::TScheme NDrive::INotifierConfig::GetScheme(const IServerBase& /*server*/) const {
    NDrive::TScheme result;
    result.Add<TFSBoolean>("use_event_log", "Использовать event log").SetDefault(false);
    result.Add<TFSNumeric>("message_length_limit", "Максимальная длина сообщения (не ограничена если ноль)").SetDefault(DefaultMessageLengthLimit);
    return result;
}

void NDrive::INotifierConfig::Init(const TYandexConfig::Section* section) {
    UseEventLog = section->GetDirectives().Value("UseEventLog", UseEventLog);
    MessageLengthLimit = section->GetDirectives().Value("MessageLengthLimit", MessageLengthLimit);
    DoInit(section);
}

void NDrive::INotifierConfig::InitFromString(const TString& configStr)  {
    TAnyYandexConfig config;
    Y_ENSURE_BT(config.ParseMemory(configStr.c_str()), "cannot parse YandexConfig from " << configStr);
    Init(config.GetRootSection());
}

void NDrive::INotifierConfig::ToString(IOutputStream& os) const {
    os << "NotificationType: " << TypeName << Endl;
    os << "UseEventLog: " << UseEventLog << Endl;
    os << "MessageLengthLimit: " << UseEventLog << Endl;
    DoToString(os);
}

NJson::TJsonValue NDrive::TNotifierResult::SerializeToJson() const {
    NJson::TJsonValue res;
    NJson::InsertNonNull(res, "transit_id", TransitId);
    NJson::InsertNonNull(res, "error_message", ErrorMessage);
    NJson::InsertField(res, "error_details", NonDefault(ErrorDetails, NJson::TJsonValue{}));
    return res;
}

TString NDrive::TNotifierMessage::GetMessageHash() const {
    if (MessageHash) {
        return *MessageHash;
    } else {
        return ToString(MultiHash(
            Name,
            Title,
            Header,
            Body,
            AdditionalInfo.GetStringRobust()
        ));
    }
}

NJson::TJsonValue NDrive::TNotifierMessage::SerializeToJson() const {
    NJson::TJsonValue json;
    json.InsertValue("TransitId", TransitId);
    DoSerializeToJson(json);
    return json;
}

bool NDrive::TNotifierMessage::DeserializeFromJson(const NJson::TJsonValue& data) {
    return (NJson::ParseField(data["TransitId"], TransitId) &&
            NJson::ParseField(data["MessageHash"], MessageHash) &&
            DoDeserializeFromJson(data));
}

void NDrive::TNotifierMessage::DoSerializeToJson(NJson::TJsonValue& result) const {
    result.InsertValue("Name", Name);
    result.InsertValue("Title", Title);
    result.InsertValue("Header", Header);
    result.InsertValue("Body", Body);
    result.InsertValue("AdditionalInfo", AdditionalInfo);
}

bool NDrive::TNotifierMessage::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    return
        NJson::ParseField(data["Name"], Name) &&
        NJson::ParseField(data["Title"], Title) &&
        NJson::ParseField(data["Header"], Header) &&
        NJson::ParseField(data["Body"], Body) &&
        NJson::ParseField(data["AdditionalInfo"], AdditionalInfo);
}

const double NDrive::INotifier::LongMessageStripMultiplier = 0.95;

void NDrive::INotifier::Start(const IServerBase* server) {
    Started = true;
    try {
        DoStart(server);
    } catch (const std::exception& e) {
        Started = false;
        ERROR_LOG << FormatExc(e) << Endl;
    }
    Stopped = false;
}

void NDrive::INotifier::Stop() {
    if (Stopped) {
        return;
    }
    try {
        DoStop();
        Stopped = true;
    } catch (const std::exception& e) {
        ERROR_LOG << FormatExc(e) << Endl;
    }
    Started = false;
}

TMaybe<ui32> NDrive::INotifier::GetSelfTvmId() const {
    return {};
}

void NDrive::INotifier::SetTvmClient(TAtomicSharedPtr<NTvmAuth::TTvmClient> /* tvmClient */) {
}

NDrive::INotifier::TResultPtr NDrive::INotifier::Notify(const NDrive::INotifier::TMessage& message, const TContext& context) const {
    if (!Started) {
        ERROR_LOG << "Not started notifier: " << NotifierName << Endl;
        return nullptr;
    }

    if (UseEventLog) {
        const auto& userId = context.GetRecipients().size() == 1 ? context.GetRecipients()[0].GetUserId() : Default<TString>();
        if (userId) {
            NDrive::TEventLog::TUserIdGuard ug(userId);
            NDrive::TEventLog::Log(NotifierName, message.SerializeToJson());
        }
        NDrive::TEventLog::Log(NotifierName + "_notify", NJson::TMapBuilder
            ("message", message.SerializeToJson())
            ("recepients", NJson::ToJson(context.GetRecipients()))
            ("external_recepients", NJson::ToJson(context.GetExternalRecipients()))
        );
    }

    TResultPtr res;
    if (IsMessageLengthLimitExceeded(message)) {
        res = DoNotify(GetMessageShortCopy(message), context);
    } else {
        res = DoNotify(message, context);
    }

    if (res && res->GetTransitId().empty() && message.GetTransitId()) {
        res->SetTransitId(message.GetTransitId());
    }

    if (!!res && res->HasErrors()) {
        ERROR_LOG << NotifierName << ": Error while sending notification: " << res->SerializeToJson().GetStringRobust() << Endl;
        TString eventName = NotifierName + "Error";
        NDrive::TEventLog::Log(eventName, res->SerializeToJson());
    }

    return res;
}

NDrive::INotifier::TResultPtr NDrive::INotifier::Notify(const NJson::TJsonValue& messageData, const TContext& context) const {
    TMessage::TPtr messagePtr = ConstructMessage(messageData);
    if (!messagePtr) {
        ERROR_LOG << NotifierName << ": Cannot parse message data: " << messageData.GetStringRobust() << Endl;
        return nullptr;
    }
    return Notify(*messagePtr, context);
}

NDrive::INotifier::TResults NDrive::INotifier::MultiLinesNotify(const TString& commonHeader, const TMessages& reportMessages, const TContext& context) const {
    TResults results;
    for (ui32 i = 0; i < reportMessages.size(); ) {
        size_t next = reportMessages.size();

        TStringStream ss;
        ss << commonHeader << Endl;

        for (ui32 rLine = i; rLine < next; ++rLine) {
            if (!reportMessages[rLine]) {
                continue;
            }
            const TString nextLine = ToString(rLine + 1) + "." + reportMessages[rLine]->GetBody();
            if (rLine > i && IsMessageLengthLimitExceeded(ss.Size() + nextLine.size())) {
                next = rLine;
                break;
            } else {
                ss << nextLine << Endl;
            }
        }
        i = next;

        DEBUG_LOG << ss.Str() << Endl;

        results.push_back(Notify(TMessage(ss.Str()), context));
    }
    return results;
}

bool NDrive::INotifier::SendDocument(const NDrive::INotifier::TMessage& /*message*/, const TString& /*mimeType*/, const TContext& /* context */) const {
    ERROR_LOG << "Document pusher is undefined" << Endl;
    return false;
}

bool NDrive::INotifier::SendPhoto(const NDrive::INotifier::TMessage& /*message*/, const TContext& /* context */) const {
    ERROR_LOG << "Photo pusher is undefined" << Endl;
    return false;
}

NDrive::INotifier::TResultPtr NDrive::INotifier::Notify(NDrive::INotifier::TPtr notifier, const TString& message, const TContext& context) {
    TResultPtr result;
    if (notifier) {
        result = notifier->Notify(NDrive::INotifier::TMessage(message), context);
    } else {
        WARNING_LOG << message << Endl;
    }
    return result;
}

NDrive::INotifier::TResults NDrive::INotifier::MultiLinesNotify(NDrive::INotifier::TPtr notifier, const TString& commonHeader, const TVector<TString>& reportLines, const TContext& context) {
    NDrive::INotifier::TResults results;
    if (notifier) {
        TMessages reportMessages;
        for (auto line : reportLines) {
            reportMessages.emplace_back(MakeAtomicShared<TMessage>(line));
        }
        results = notifier->MultiLinesNotify(commonHeader, reportMessages, context);
    } else {
        TStringStream ss;
        ss << commonHeader << Endl;
        for (ui32 rLine = 0; rLine < reportLines.size(); rLine++) {
            ss << ToString(rLine + 1) << "." << reportLines[rLine] << Endl;
        }
        WARNING_LOG << ss.Str() << Endl;
    }
    return results;
}

bool NDrive::INotifier::SendDocument(NDrive::INotifier::TPtr notifier, const NDrive::INotifier::TMessage& message, const TString& mimeType, const TContext& context) {
    if (notifier) {
        return notifier->SendDocument(message, mimeType, context);
    } else {
        return false;
    }
}

bool NDrive::INotifier::SendPhoto(NDrive::INotifier::TPtr notifier, const NDrive::INotifier::TMessage& message, const TContext& context) {
    if (notifier) {
        return notifier->SendPhoto(message, context);
    } else {
        return false;
    }
}

NDrive::INotifier::TMessage::TPtr NDrive::INotifier::ConstructMessage(const NJson::TJsonValue& messageData) const {
    if (messageData.IsString()) {
        return MakeAtomicShared<TMessage>(messageData.GetString());
    }
    auto message = MakeAtomicShared<TMessage>("");
    if (!message->DeserializeFromJson(messageData)) {
        return nullptr;
    }
    return message;
}

bool NDrive::INotifier::IsMessageLengthLimitExceeded(size_t totalSize) const {
    return !!MessageLengthLimit && LongMessageStripMultiplier * totalSize > MessageLengthLimit;
}

bool NDrive::INotifier::IsMessageLengthLimitExceeded(const TMessage& message) const {
    return IsMessageLengthLimitExceeded(message.GetBody().size() + message.GetHeader().size());
}

NDrive::INotifier::TMessage NDrive::INotifier::GetMessageShortCopy(const TMessage& message) const {
    if (!IsMessageLengthLimitExceeded(message)) {
        return message;
    }

    const size_t messageLength = message.GetBody().size() + message.GetHeader().size();

    WARNING_LOG << "Message total length of " << messageLength << " exceeds the limit of " << MessageLengthLimit << " and will be trimmed; "
                << "message header: " << message.GetHeader() << Endl;

    const double correction = MessageLengthLimit / (LongMessageStripMultiplier * messageLength);

    const size_t shortHeaderSize = Max<size_t>(0, message.GetHeader().size() * correction - 3);  // strlen("...") = 3
    const size_t shortBodySize = Max<size_t>(0, message.GetBody().size() * correction - 3);  // strlen("...") = 3

    TMessage messageCopy = TMessage(TString(SubstrUTF8(message.GetHeader(), 0, shortHeaderSize)) + "...",
                                    TString(SubstrUTF8(message.GetBody(), 0, shortBodySize)) + "...");

    messageCopy.SetName(message.GetName()).SetTransitId(message.GetTransitId());
    return messageCopy;
}
