#pragma once

#include "users_contacts.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/mediator/messenger.h>
#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/ptr.h>
#include <util/stream/output.h>

class IServerBase;

namespace NTvmAuth {
    class TTvmClient;
}

template <class TObjectContainer>
class IDBEntitiesManager;

template <class TObjectContainer>
class IDBEntitiesWithPropositionsManager;

class TNotifierContainer;

class TInternalNotificationMessage: public NMessenger::IMessage {
public:
    TInternalNotificationMessage() = default;
    TInternalNotificationMessage(const TString& uid, const TString& message)
        : UID(uid)
        , Message(message)
    {
    }

private:
    R_READONLY(TString, UID);
    R_READONLY(TString, Message);
};

namespace NDrive {

class INotifier;

class INotifierConfig {
public:
    using TFactory = NObjectFactory::TParametrizedObjectFactory<INotifierConfig, TString>;
    using TPtr = TAtomicSharedPtr<INotifierConfig>;

    INotifierConfig() = default;
    virtual ~INotifierConfig() = default;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info, TMessagesCollector& /*errors*/);
    virtual NJson::TJsonValue SerializeToJson() const;
    virtual NJson::TJsonValue GetUserReport(const IServerBase& server) const;

    virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const;

    virtual void Init(const TYandexConfig::Section* section) final;
    virtual void InitFromString(const TString& configStr) final;
    virtual void ToString(IOutputStream& os) const final;

    template<typename TNotifierConfig>
    static TNotifierConfig BuildFromString(const TString& configStr) {
        TNotifierConfig configInstance;
        configInstance.InitFromString(configStr);
        return configInstance;
    }

    virtual bool UserPatch(const NJson::TJsonValue& /*data*/, TMessagesCollector& /*errors*/) {
        return true;
    };

    virtual TAtomicSharedPtr<INotifier> Construct() const = 0;

protected:
    virtual void DoToString(IOutputStream& os) const = 0;
    virtual void DoInit(const TYandexConfig::Section* section) = 0;

private:
    static const ui32 DefaultMessageLengthLimit;

    R_FIELD(TString, TypeName);
    R_FIELD(TString, Name);
    R_READONLY(bool, UseEventLog, true);
    R_READONLY(ui32, MessageLengthLimit, DefaultMessageLengthLimit);
};

class TNotifierResult {
protected:
    TNotifierResult() = default;

public:
    using TPtr = TAtomicSharedPtr<TNotifierResult>;

    explicit TNotifierResult(const TString& error)
        : ErrorMessage(error)
    {
    }

    TNotifierResult(const TString& error, const NJson::TJsonValue& errorDetails)
        : ErrorMessage(error)
        , ErrorDetails(errorDetails)
    {
    }

    virtual ~TNotifierResult() = default;

    bool HasErrors() const noexcept {
        return !ErrorMessage.empty();
    }

    virtual NJson::TJsonValue SerializeToJson() const;

private:
    R_FIELD(TString, TransitId);
    R_FIELD(TString, ErrorMessage);
    R_FIELD(NJson::TJsonValue, ErrorDetails);
};

class TNotifierMessage {
protected:
    TNotifierMessage() = default;

public:
    using TPtr = TAtomicSharedPtr<TNotifierMessage>;

    explicit TNotifierMessage(const TString& body)
        : Body(body)
    {
    }

    TNotifierMessage(const TString& header, const TString& body)
        : Header(header)
        , Body(body)
    {
    }

    virtual ~TNotifierMessage() = default;

    TString GetMessageHash() const;

    bool HasAdditionalInfo() const noexcept {
        return !AdditionalInfo.IsNull();
    }

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& data);

    template <class T>
    const T* GetAs() const {
        return dynamic_cast<const T*>(this);
    }

protected:
    virtual void DoSerializeToJson(NJson::TJsonValue& result) const;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data);

private:
    R_FIELD(TString, Name);
    R_FIELD(TString, Title);
    R_FIELD(TString, Header);
    R_FIELD(TString, Body);

    R_FIELD(NJson::TJsonValue, AdditionalInfo, NJson::JSON_NULL);

    R_FIELD(TString, TransitId);
    R_OPTIONAL(TString, MessageHash);
};

class TNotifierContext {
public:
    using TExternalRecipients = TVector<TString>;
    using TRecipients = TVector<TUserContacts>;

public:
    TNotifierContext() = default;
    virtual ~TNotifierContext() = default;

    const IServerBase* GetServer() const noexcept {
        return Server;
    }

    auto& SetServer(const IServerBase* server) {
        Server = server;
        return *this;
    }

private:
    R_FIELD(TExternalRecipients, ExternalRecipients);
    R_FIELD(TRecipients, Recipients);

    const IServerBase* Server = nullptr;
};

class INotifier {
public:
    using TPtr = TAtomicSharedPtr<INotifier>;

    using TResult = TNotifierResult;
    using TResultPtr = TResult::TPtr;
    using TResults = TVector<TResultPtr>;

    using TContext = TNotifierContext;
    using TRecipients = TContext::TRecipients;

    using TMessage = TNotifierMessage;
    using TMessages = TVector<TMessage::TPtr>;

    INotifier(const INotifierConfig& config)
        : NotifierName(config.GetName())
        , UseEventLog(config.GetUseEventLog())
        , MessageLengthLimit(config.GetMessageLengthLimit())
    {
    }

    virtual ~INotifier() = default;

    bool IsActive() const noexcept {
        return Started;
    }

    const TString& GetNotifierName() const noexcept {
        return NotifierName;
    }

    TResultPtr Notify(const TMessage& message, const TContext& context = Default<TContext>()) const;
    TResultPtr Notify(const NJson::TJsonValue& messageData, const TContext& context = Default<TContext>()) const;

    void Start(const IServerBase* server);
    void Stop();

    virtual TMaybe<ui32> GetSelfTvmId() const;
    virtual void SetTvmClient(TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient);

    virtual TResults MultiLinesNotify(const TString& commonHeader, const TMessages& reportMessages, const TContext& context = Default<TContext>()) const;
    virtual bool SendDocument(const TMessage& message, const TString& mimeType, const TContext& context = Default<TContext>()) const;
    virtual bool SendPhoto(const TMessage& message, const TContext& context = Default<TContext>()) const;

    static TResultPtr Notify(INotifier::TPtr notifier, const TString& message, const TContext& context = Default<TContext>());

    static TResults MultiLinesNotify(INotifier::TPtr notifier, const TString& commonHeader, const TVector<TString>& reportLines, const TContext& context = Default<TContext>());
    static bool SendDocument(INotifier::TPtr notifier, const INotifier::TMessage& message, const TString& mimeType, const TContext& context = Default<TContext>());
    static bool SendPhoto(INotifier::TPtr notifier, const INotifier::TMessage& message, const TContext& context = Default<TContext>());

protected:
    virtual void DoStart(const IServerBase* server) = 0;
    virtual void DoStop() = 0;

    virtual TMessage::TPtr ConstructMessage(const NJson::TJsonValue& messageData) const;
    virtual TResultPtr DoNotify(const TMessage& message, const TContext& context) const = 0;

    TString NotifierName;
    bool UseEventLog;
    ui32 MessageLengthLimit;

private:
    bool IsMessageLengthLimitExceeded(size_t totalSize) const;
    bool IsMessageLengthLimitExceeded(const TMessage& message) const;

    TMessage GetMessageShortCopy(const TMessage& message) const;

private:
    static const double LongMessageStripMultiplier;

    bool Started = false;
    bool Stopped = true;
};

class INotifiersStorage {
public:
    virtual INotifier::TPtr GetNotifier(const TString& name) const = 0;
    virtual ~INotifiersStorage() {}
};

class INotifyResultProvider {
public:
    using TNotifyResultPtr = typename INotifier::TResultPtr;

    virtual ~INotifyResultProvider() = default;
    virtual TNotifyResultPtr GetHandlingResult(const TString& transitId) const = 0;
};

using INotifiersManager = IDBEntitiesWithPropositionsManager<TNotifierContainer>;

} // namespace NDrive
