#include "setting.h"

#include <rtline/library/json/parse.h>

TSetting::TSettingDecoder::TSettingDecoder(const TMap<TString, ui32>& decoderBase) {
    Key = GetFieldDecodeIndexOrThrow("setting_key", decoderBase);
    Value = GetFieldDecodeIndexOrThrow("setting_value", decoderBase);
    Revision = GetFieldDecodeIndexOrThrow("revision", decoderBase);
}

bool TSetting::DeserializeWithDecoder(const TSettingDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Key);
    READ_DECODER_VALUE(decoder, values, Value);
    if (!Key) {
        return false;
    }
    READ_DECODER_VALUE_DEF(decoder, values, Revision, Max<ui64>());
    return true;
}

bool TSetting::Parse(const NStorage::TTableRecord& record) {
    return DeserializeFromTableRecord(record, nullptr);
}

bool TSetting::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
    Key = record.Get("setting_key");
    if (!Key || !TryFromString(record.Get("revision"), MutableRevision())) {
        return false;
    }
    Value = record.Get("setting_value");
    return true;
}

NStorage::TTableRecord TSetting::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("setting_key", Key).Set("setting_value", Value);
    if (HasRevision()) {
        result.Set("revision", Revision);
    }
    return result;
}

NStorage::TTableRecord TSetting::SerializeUniqueToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("setting_key", Key);
    return result;
}

bool TSetting::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!NJson::ParseField(jsonInfo, "setting_key", Key, true) ||
        !NJson::ParseField(jsonInfo, "setting_value", Value, false) ||
        !NJson::ParseField(jsonInfo, "revision", Revision, false)) {
        return false;
    }
    return true;
}

NJson::TJsonValue TSetting::SerializeToJson() const {
    return SerializeToTableRecord().SerializeToJson();
}

NJson::TJsonValue TSetting::BuildJsonReport() const {
    return SerializeToJson();
}
