#pragma once

#include "setting.h"

#include <library/cpp/mediator/messenger.h>

#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>

template <class TObject>
class TObjectEvent;

namespace NDrive {
    class ISettingGetter: public TAtomicRefCount<ISettingGetter> {
    public:
        virtual ~ISettingGetter() = default;

        virtual bool GetValueStr(TStringBuf key, TString& result, TInstant reqActuality = TInstant::Zero()) const = 0;

        NJson::TJsonValue GetJsonValue(TStringBuf key, bool throwOnParseError = false) const;
        NJson::TJsonValue GetJsonValue(TConstArrayRef<TString> pathes, const TString& key, TString* resultPath = nullptr, bool throwOnParseError = false) const;

        template <class T>
        bool GetValue(TStringBuf key, T& result, TInstant reqActuality = TInstant::Zero()) const {
            TString resultStr;
            if (!GetValueStr(key, resultStr, reqActuality)) {
                return false;
            }
            return TryFromString(resultStr, result);
        }

        template <class T>
        TMaybe<T> GetValue(TStringBuf key, TInstant reqActuality = TInstant::Zero()) const {
            T result;
            if (!GetValue(key, result, reqActuality)) {
                return {};
            } else {
                return result;
            }
        }
    };
    using TSettingGetterConstPtr = TIntrusiveConstPtr<ISettingGetter>;

    class ISettings
        : public ISettingGetter
        , public IMessageProcessor
    {
    public:
        ISettings()  {
            RegisterGlobalMessageProcessor(this);
        }
        virtual ~ISettings() {
            UnregisterGlobalMessageProcessor(this);
        }
        virtual bool Process(IMessage* message) override {
            Y_UNUSED(message);
            return false;
        }
        virtual TString Name() const override {
            return ToString((ui64)this);
        }

        bool GetIntervalParamValueJsonDef(const TString& prefix, const int value, const NJson::TJsonValue& defaultValue, NJson::TJsonValue& result, const TInstant reqActuality = TInstant::Zero()) const;

        virtual bool HasValues(const TSet<TString>& keys, TSet<TString>& existKeys, const TInstant reqActuality) const = 0;
        virtual bool RemoveKeys(const TVector<TString>& keys, const TString& userId) const = 0;
        virtual bool SetValues(const TVector<TSetting>& values, const TString& userId) const = 0;
        virtual bool GetHistory(const TInstant since, TVector<TAtomicSharedPtr<TObjectEvent<TSetting>>>& result, const TInstant reqActuality = TInstant::Zero()) const = 0;
        virtual bool GetAllSettings(TVector<TSetting>& result, const TInstant reqActuality) const = 0;

        bool SetValue(const TString& key, const TString& value, const TString& userId) const;

        template <class T>
        T GetHandlerValueDef(TStringBuf handlerName, TStringBuf key, const T defValue, const TInstant reqActuality = TInstant::Zero()) const {
            return GetValueDef(TString::Join("handlers.", handlerName, ".", key), defValue, reqActuality);
        }

        template <class T>
        T GetBackgroundValueDef(const TString& processId, const TString& key, const T defValue, const TInstant reqActuality = TInstant::Zero()) const {
            return GetValueDef("backgrounds." + processId + "." + key, defValue, reqActuality);
        }

        template <class T>
        T GetValueDef(TStringBuf key, T defValue, TInstant reqActuality = TInstant::Zero()) const {
            TString resultStr;
            if (!GetValueStr(key, resultStr, reqActuality)) {
                return defValue;
            }
            T result;
            if (!TryFromString(resultStr, result)) {
                WARNING_LOG << "Cannot parse value from '" << resultStr << "' string (" << key << ")" << Endl;
                return defValue;
            }
            return result;
        }

        template <class T>
        T GetValueDef(TConstArrayRef<TString> pathes, const TString& key, T defValue, TInstant reqActuality = TInstant::Zero()) const {
            for (auto&& i : pathes) {
                TString resultStr;
                if (!GetValueStr(i + "." + key, resultStr, reqActuality)) {
                    continue;
                }
                T result;
                if (TryFromString(resultStr, result)) {
                    return result;
                } else {
                    WARNING_LOG << "Cannot parse value from '" << resultStr << "' string (" << key << ")" << Endl;
                    return defValue;
                }
            }
            return defValue;
        }
    };
}

using ISettings = NDrive::ISettings;

class TFakeSettings: public ISettings {
public:
    virtual bool HasValues(const TSet<TString>& /*keys*/, TSet<TString>& existKeys, const TInstant /*reqActuality*/) const override {
        existKeys.clear();
        return true;
    }
    virtual bool GetValueStr(TStringBuf /*key*/, TString& /*result*/, TInstant /*reqActuality*/) const override {
        return false;
    }
    virtual bool RemoveKeys(const TVector<TString>& /*keys*/, const TString& /*userId*/) const override {
        return true;
    }
    virtual bool SetValues(const TVector<TSetting>& /*values*/, const TString& /*userId*/) const override {
        return false;
    }
    virtual bool GetHistory(const TInstant /*since*/, TVector<TAtomicSharedPtr<TObjectEvent<TSetting>>>& /*result*/, const TInstant /*reqActuality*/) const override {
        return true;
    }
    virtual bool GetAllSettings(TVector<TSetting>& /*result*/, const TInstant /*reqActuality*/) const override {
        return true;
    }
};
