#include "action.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/roles/manager.h>

#include <library/cpp/json/json_reader.h>

#include <util/digest/fnv.h>

bool TUserAction::CheckRequest(IReplyContext::TPtr context) const {
    if (!context || ClientVersions.empty()) {
        return true;
    }

    i32 v;
    for (auto&& [headerName, minimalVersion] : ClientVersions) {
        auto headerData = context->GetRequestData().HeaderIn(headerName);
        if (!headerData) {
            continue;
        }
        if (!minimalVersion) {
            return true;
        }
        if (TryFromString(*headerData, v)) {
            return v >= minimalVersion;
        }
    }
    return ClientVersionCheckPolicy == EClientVersionCheckPolicy::AllowWithoutAppVersion;
}

bool TUserAction::CheckUser(const TTaggedObject& taggedUser) const {
    if (!UserTagsFilter) {
        return true;
    }
    return UserTagsFilter.IsMatching(taggedUser);
}

bool TUserAction::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE_DEF(decoder, values, SequentialId, Max<ui32>());
    READ_DECODER_VALUE(decoder, values, Name);
    READ_DECODER_VALUE_DEF(decoder, values, Revision, Max<ui64>());
    if (!Name) {
        return false;
    }
    READ_DECODER_VALUE(decoder, values, Enabled);
    READ_DECODER_VALUE_DEF(decoder, values, Deprecated, false);
    READ_DECODER_VALUE(decoder, values, Description);
    READ_DECODER_VALUE_DEF(decoder, values, ActionParent, "");
    if (ActionParent == "get_null()") {
        ActionParent = "";
    }

    NJson::TJsonValue jsonSpecials;
    READ_DECODER_VALUE_JSON(decoder, values, jsonSpecials, Meta);

    return DeserializeCommonFromJson(jsonSpecials) && DeserializeSpecialsFromJson(jsonSpecials);
}

bool TUserAction::DeserializeCommonFromJson(const NJson::TJsonValue& jsonSpecials) {
    if (!UserAreaTagsFilter.DeserializeFromJson(jsonSpecials)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonSpecials, "landing", Landing)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonSpecials, "contr_landing", ContrLanding)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(jsonSpecials, "groupping_tags", GrouppingTags)) {
        return false;
    }
    JREAD_STRING_OPT(jsonSpecials, "admin_icon", AdminIcon);

    if (jsonSpecials.Has("client_version")) {
        if (!jsonSpecials["client_version"].IsArray()) {
            return false;
        }
        for (auto&& i : jsonSpecials["client_version"].GetArraySafe()) {
            TString header;
            long long version;
            if (i["header"].GetString(&header) && i["version"].GetInteger(&version)) {
                if (!!header) {
                    ClientVersions.emplace(header, version);
                }
            } else {
                return false;
            }
        }
    }
    JREAD_FROM_STRING_OPT(jsonSpecials, "client_version_check_policy", ClientVersionCheckPolicy);

    if (jsonSpecials.Has("experiment")) {
        if (!Experiment.DeserializeFromJson(jsonSpecials["experiment"])) {
            return false;
        }
    }

    if (jsonSpecials.Has("limited_policy")) {
        if (!SourceContext.DeserializeFromJson(jsonSpecials["limited_policy"])) {
            return false;
        }
    }

    if (!NJson::ParseField(jsonSpecials["user_tags_filter"], UserTagsFilter)) {
        return false;
    }

    return true;
}

void TUserAction::SerializeCommonToJson(NJson::TJsonValue& meta) const {
    if (AdminIcon) {
        meta["admin_icon"] = AdminIcon;
    }
    if (Landing.size()) {
        TJsonProcessor::WriteContainerArray(meta, "landing", Landing);
    }
    if (ContrLanding.size()) {
        TJsonProcessor::WriteContainerArray(meta, "contr_landing", ContrLanding);
    }
    if (UserTagsFilter) {
        meta["user_tags_filter"] = UserTagsFilter.ToString();
    }
    UserAreaTagsFilter.SerializeToJson(meta);
    if (ClientVersions.size()) {
        NJson::TJsonValue& versionsJson = meta.InsertValue("client_version", NJson::JSON_ARRAY);
        for (auto&& i : ClientVersions) {
            NJson::TJsonValue& versionInfo = versionsJson.AppendValue(NJson::JSON_MAP);
            versionInfo.InsertValue("header", i.first);
            versionInfo.InsertValue("version", i.second);
        }
    }
    JWRITE_ENUM(meta, "client_version_check_policy", ClientVersionCheckPolicy);

    TJsonProcessor::WriteContainerString(meta, "groupping_tags", GrouppingTags);
    meta.InsertValue("experiment", Experiment.SerializeToJson());

    if (SourceContext.IsLimitedPolicy()) {
        meta.InsertValue("limited_policy", SourceContext.SerializeToJson());
    }
}

NJson::TJsonValue TUserAction::SerializeCompactToJson() const {
    NJson::TJsonValue meta;
    if (!GrouppingTags.empty()) {
        meta["groupping_tags"] = JoinStrings(GrouppingTags.begin(), GrouppingTags.end(), ",");
    }
    return meta;
}

NDrive::TScheme TUserAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme;
    {
        auto gLandings = server->GetDriveAPI()->GetLandingsDB()->GetCachedObjectsMap();
        scheme.Add<TFSVariants>("landing", "Landing для данного действия", 100000).SetMultiSelect(true).SetVariants(gLandings);
        scheme.Add<TFSVariants>("contr_landing", "Альтернативный Landing для данного действия", 100000).SetMultiSelect(true).SetVariants(gLandings);
    }

    scheme.Add<TFSString>("admin_icon", "icon for admin", 200000);
    {
        NDrive::TScheme& schemeClientVersion = scheme.Add<TFSArray>("client_version", "Минимальная версия клиентов для использования", 100000).SetElement<NDrive::TScheme>();
        schemeClientVersion.Add<TFSString>("header", "Заголовок версии клиента");
        schemeClientVersion.Add<TFSNumeric>("version", "Минимальная версия для использования").SetMin(0);
    }
    scheme.Add<TFSVariants>("client_version_check_policy", "Настройки проверки по версии клиента", 100000)
            .InitVariants<EClientVersionCheckPolicy>()
            .SetDefault(::ToString(EClientVersionCheckPolicy::Default));
    scheme.Add<TFSStructure>("experiment", "Условия для активации action").SetStructure<NDrive::TScheme>(Experiment.GetScheme());
    scheme.Add<TFSString>("groupping_tags", "Теги для группировки и поиска");
    scheme.Add<TFSString>("user_tags_filter", "Teги пользователя для активации action");
    TUserAreaTagsFilter::AddScheme(scheme, *server, 100000);
    return scheme;
}

NJson::TJsonValue TUserAction::BuildJsonReport(bool serializeMeta) const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    result.InsertValue("action_description", Description);
    result.InsertValue("action_id", Name);
    result.InsertValue("action_type", GetType());
    result.InsertValue("enabled", Enabled);
    result.InsertValue("deprecated", Deprecated);
    if (HasRevision()) {
        result.InsertValue("action_revision", Revision);
    }
    if (ActionParent) {
        result.InsertValue("parent", ActionParent);
    }
    if (serializeMeta) {
        NJson::TJsonValue& specials = result.InsertValue("action_meta", SerializeSpecialsToJson());
        SerializeCommonToJson(specials);
    } else {
        NJson::TJsonValue meta = SerializeCompactToJson();
        if (meta.IsDefined()) {
            result.InsertValue("action_meta", std::move(meta));
        }
    }
    return result;
}

TUserAction::TUniquePtr TUserAction::BuildFromTableRow(const NStorage::TTableRecord& row, bool strict) {
    auto result = Hold(TFactory::Construct(row.Get("action_type")));
    if (!result) {
        WARNING_LOG << "Cannot construct action with type: " << row.Get("action_type") << Endl;
        return nullptr;
    }

    if (!TBaseDecoder::DeserializeFromTableRecordStrictable(*result, row, strict)) {
        WARNING_LOG << "action ignored: " << row.SerializeToJson() << Endl;
        return nullptr;
    }
    return result;
}

TUserAction::TUniquePtr TUserAction::BuildFromJson(const NJson::TJsonValue& jsonInfo) {
    const NJson::TJsonValue* typeName;
    if (!jsonInfo.GetValuePointer("action_type", &typeName) || !typeName->IsString()) {
        WARNING_LOG << "Cannot construct action withno type info: " << jsonInfo.GetStringRobust() << Endl;
        return nullptr;
    }
    auto result = Hold(TFactory::Construct(typeName->GetString()));
    if (!result) {
        WARNING_LOG << "Cannot construct action with type: " << typeName->GetString() << Endl;
        return nullptr;
    }

    if (!TBaseDecoder::DeserializeFromJson(*result, jsonInfo)) {
        WARNING_LOG << "Cannot parse action type " << typeName->GetString() << ": " << jsonInfo << Endl;
        return nullptr;
    }

    return result;
}

TUserAction::TUniquePtr TUserAction::Clone() const {
    auto serialized = SerializeToTableRow();
    return BuildFromTableRow(serialized, false);
}

NStorage::TTableRecord TUserAction::SerializeToTableRow() const {
    NStorage::TTableRecord row;
    row.Set("action_description", Description);
    row.Set("action_id", Name);
    row.Set("action_type", GetType());
    NJson::TJsonValue meta = SerializeSpecialsToJson();
    SerializeCommonToJson(meta);
    row.Set("action_meta", meta.GetStringRobust());
    if (ActionParent) {
        row.Set("parent", ActionParent);
    } else {
        row.Set("parent", "get_null()");
    }
    row.Set("enabled", Enabled);
    row.Set("deprecated", Deprecated);
    if (HasRevision()) {
        row.Set("action_revision", Revision);
    }
    if (SequentialId != Max<ui32>()) {
        row.Set("id", SequentialId);
    }
    return row;
}

NDrive::TScheme TUserAction::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("action_id", "Идентификатор");
    scheme.Add<TFSBoolean>("enabled", "Активность").SetDefault(false);
    scheme.Add<TFSBoolean>("deprecated", "Устарел").SetDefault(false);
    {
        TSet<TString> sameClassActions;
        TVector<TDBAction> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector();
        for (auto&& i : actions) {
            if (i->GetType() == GetType()) {
                sameClassActions.emplace(i->GetName());
            }
        }
        scheme.Add<TFSVariants>("parent", "Родительский объект").SetVariants(sameClassActions);
    }

    scheme.Add<TFSNumeric>("action_revision", "Версия конфигурации").SetReadOnly(true);
    scheme.Add<TFSText>("action_description", "Описание");
    scheme.Add<TFSStructure>("action_meta", "Метаданные").SetStructure(DoGetScheme(server));
    return scheme;
}

NJson::TJsonValue TUserAction::GetPublicReport(ELocalization locale, const ILocalization& localization) const {
    Y_UNUSED(locale);
    NJson::TJsonValue result;
    NJson::InsertField(result, "title", localization.ApplyResources(Description, locale));
    NJson::InsertField(result, "id", Name);
    return result;
}
