#pragma once

#include <drive/backend/cars_filter/cars_filter.h>

#include <drive/library/cpp/common/object_sharding.h>
#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/searchserver/context/replier.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class IHistoryContext;

namespace NDrive {
    class IServer;
}

class TUserAreaTagsFilterNamesConstructor {
public:
    static TString GetFieldNameTags() {
        return "user_tags_in_point";
    }

    static TString GetFieldNameUndefinedPolicy() {
        return "user_undefined_position_policy";
    }

    static TString GetObjectType() {
        return "пользователь";
    }
};

class TUserAction {
private:
    using TClientVersions = TMap<TString, i32>;
    using TUserAreaTagsFilter = TAreaTagsFilterImpl<TUserAreaTagsFilterNamesConstructor>;

    class TSourceContext {
        R_FIELD(bool, PromoCode, false);
        R_FIELD(bool, LimitedPolicy, false);
        R_FIELD(TInstant, Since, TInstant::Zero());
        R_FIELD(TInstant, Until, TInstant::Max());
        R_FIELD(TString, TagId);

    public:
        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue json;
            NJson::InsertField(json, "limited_policy", LimitedPolicy);
            NJson::InsertField(json, "promocode", PromoCode);
            NJson::InsertField(json, "since", Since);
            NJson::InsertField(json, "until", Until);
            NJson::InsertField(json, "tag_id", TagId);
            return json;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "limited_policy", LimitedPolicy, true)
                && NJson::ParseField(json, "promocode", PromoCode, false)
                && NJson::ParseField(json, "since", Since)
                && NJson::ParseField(json, "until", Until)
                && NJson::ParseField(json, "tag_id", TagId);
        }
    };

public:
    enum class EClientVersionCheckPolicy {
        Default = 0,
        AllowWithoutAppVersion,
    };

    using TRevision = ui64;
    using TSequentialId = ui32;

private:
    R_FIELD(TString, Name);
    R_FIELD(TRevision, Revision, Max<TRevision>());
    R_FIELD(TSequentialId, SequentialId, Max<TSequentialId>());
    R_FIELD(TString, Description);
    R_FIELD(TVector<TString>, Landing);
    R_FIELD(TVector<TString>, ContrLanding);
    R_FIELD(TTagsFilter, UserTagsFilter);
    R_FIELD(TUserAreaTagsFilter, UserAreaTagsFilter);
    R_FIELD(TClientVersions, ClientVersions);
    R_FIELD(EClientVersionCheckPolicy, ClientVersionCheckPolicy, EClientVersionCheckPolicy::Default);
    R_FIELD(bool, Enabled, true);
    R_FIELD(bool, Deprecated, false);
    R_FIELD(TString, AdminIcon);
    R_FIELD(TString, ActionParent);
    R_FIELD(TObjectSharding, Experiment);
    R_FIELD(TSet<TString>, GrouppingTags);
    R_FIELD(TSourceContext, SourceContext);

protected:
    bool DeserializeCommonFromJson(const NJson::TJsonValue& jsonSpecials);
    void SerializeCommonToJson(NJson::TJsonValue& meta) const;

    virtual NJson::TJsonValue SerializeCompactToJson() const;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& /*jsonValue*/) {
        return true;
    }

    virtual NJson::TJsonValue SerializeSpecialsToJson() const {
        return NJson::JSON_MAP;
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const;

public:
    bool CheckRequest(IReplyContext::TPtr context) const;
    bool CheckUser(const TTaggedObject& taggedUser) const;

public:
    using TFactory = NObjectFactory::TParametrizedObjectFactory<TUserAction, TString>;
    using TPtr = TAtomicSharedPtr<TUserAction>;
    using TConstPtr = TAtomicSharedPtr<const TUserAction>;
    using TUniquePtr = THolder<TUserAction>;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, SequentialId, -1);
        R_FIELD(i32, Description, -1);
        R_FIELD(i32, Enabled, -1);
        R_FIELD(i32, Deprecated, -1);
        R_FIELD(i32, Revision, -1);
        R_FIELD(i32, Meta, -1);
        R_FIELD(i32, ActionParent, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            SequentialId = GetFieldDecodeIndex("id", decoderBase);
            Name = GetFieldDecodeIndex("action_id", decoderBase);
            Description = GetFieldDecodeIndex("action_description", decoderBase);
            Enabled = GetFieldDecodeIndex("enabled", decoderBase);
            Deprecated = GetFieldDecodeIndex("deprecated", decoderBase);
            Meta = GetFieldDecodeIndex("action_meta", decoderBase);
            Revision = GetFieldDecodeIndex("action_revision", decoderBase);
            ActionParent = GetFieldDecodeIndex("parent", decoderBase);
        }
    };

public:
    bool HasRevision() const {
        return Revision != Max<ui64>();
    }

    TMaybe<ui64> OptionalRevision() const {
        return HasRevision() ? Revision : TMaybe<ui64>();
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);

public:
    static TUniquePtr BuildFromTableRow(const NStorage::TTableRecord& row, bool strict = true);
    static TUniquePtr BuildFromJson(const NJson::TJsonValue& jsonInfo);

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const final;
    virtual NJson::TJsonValue GetPublicReport(ELocalization locale, const ILocalization& localization) const;

public:
    TUserAction() = default;
    TUserAction(const TString& name)
        : Name(name)
    {
    }

    virtual ~TUserAction() = default;

    TUniquePtr Clone() const;

    NJson::TJsonValue BuildJsonReport(bool serializeMeta = true) const;
    virtual TString GetType() const = 0;
    virtual NStorage::TTableRecord SerializeToTableRow() const final;
};

using TUserActions = TVector<TUserAction::TConstPtr>;
