#include "administrative.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/promo_codes/common/action.h>
#include <drive/backend/roles/manager.h>

#include <drive/telematics/protocol/vega.h>

#include <util/generic/serialized_enum.h>

TUserAction::TFactory::TRegistrator<TAdministrativeAction> TAdministrativeAction::Registrator("adm");

template <>
NJson::TJsonValue NJson::ToJson(const TAdministrativeAction::EAction& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TAdministrativeAction::EAction& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

namespace {
    bool Append(TSet<TString>& container, const NJson::TJsonValue& value) {
        NJson::TJsonValue::TArray arr;
        if (value.GetArray(&arr)) {
            for (auto&& i : arr) {
                if (i.IsString()) {
                    container.insert(i.GetString());
                } else {
                    return false;
                }
            }
        }
        return true;
    }
}

TMaybe<double> TAdministrativeAction::GetLimit(const TString& instance) const {
    auto p = Limits.find(instance);
    if (p != Limits.end()) {
        return p->second;
    } else {
        return {};
    }
}

bool TAdministrativeAction::CheckAction(const EAction action, const EEntity entity, const TString& handler, const TString& instance, const TMaybe<TSet<TString>>& instanceTags) const {
    if (!Actions.contains(action) || !Entities.contains(entity)) {
        return false;
    }
    if (Handlers.size() && !Handlers.contains(handler)) {
        return false;
    }
    if (entity == EEntity::RTBackground) {
        if (RTBackgroundTagsFilter.ToString() == "__ALL__") {
            return true;
        }
        if (!RTBackgroundTagsFilter.IsEmpty()) {
            return instanceTags && RTBackgroundTagsFilter.IsMatching(*instanceTags);
        }
        return false;
    }

    if (entity == EEntity::Action) {
//        if (action == EAction::Modify || action == EAction::Observe) {
            if (ActionTypes.empty() && ActionTagsFilter.IsEmpty()) {
                return true;
            }
//        }
        if (ActionTypes.contains(instance) || ActionTypes.contains("*")) {
            return true;
        }
        if (instanceTags && !ActionTagsFilter.IsEmpty() && ActionTagsFilter.IsMatching(*instanceTags)) {
            return true;
        }
        return false;
    }
    if (entity == EEntity::User && UserIds.size()) {
        return UserIds.contains(instance);
    }
    if (entity == EEntity::Settings) {
        if (SettingPrefixes.empty() || !instance) {
            return false;
        }

        if (SettingPrefixes.contains("*")) {
            return true;
        }

        for (auto&& i : SettingPrefixes) {
            if (instance.StartsWith(i)) {
                return true;
            }
        }
        return false;
    }
    if (entity == EEntity::Role && Roles.size()) {
        return Roles.contains(instance);
    }
    if (entity == EEntity::Tag && TagTypes.size()) {
        return TagTypes.contains(instance);
    }
    if (entity == EEntity::Wallet && Wallets.size()) {
        return Wallets.contains(instance);
    }
    if (entity == EEntity::PromoCodes && PromoCodes.size()) {
        return PromoCodes.contains(instance);
    }
    if (entity == EEntity::MDS && MdsBuckets.size()) {
        return MdsBuckets.contains(instance);
    }
    if (entity == EEntity::B2BOrganization && B2BOrganizations.size()) {
        return B2BOrganizations.contains(instance);
    }
    if (entity == EEntity::ChatRobot && ChatRobots.size()) {
        return ChatRobots.contains(instance);
    }
    if (Commands.size() && !Commands.contains(instance)) {
        return false;
    }
    if (Instances.size() && !Instances.contains(instance)) {
        return false;
    }
    return true;
}

TMaybe<TSet<TString>> TAdministrativeAction::GetInstances(const EAction action, const EEntity entity, const TString& handler) const {
    if (!Actions.contains(action) || !Entities.contains(entity)) {
        return TSet<TString>();
    }

    if (Handlers.size() && !Handlers.contains(handler)) {
        return TSet<TString>();
    }

    switch(entity) {
    case EEntity::User:
        if (UserIds.empty()) {
            return Nothing();
        } else {
            return UserIds;
        }
    case EEntity::Action:
        if (ActionTypes.empty()) {
            return Nothing();
        } else {
            return ActionTypes;
        }
    case EEntity::Role:
        if (Roles.empty()) {
            return Nothing();
        } else {
            return Roles;
        }
    case EEntity::Wallet:
        if (Wallets.empty()) {
            return Nothing();
        } else {
            return Wallets;
        }
    case EEntity::PromoCodes:
        if (PromoCodes.empty()) {
            return Nothing();
        } else {
            return PromoCodes;
        }
    case EEntity::MDS:
        if (MdsBuckets.empty()) {
            return Nothing();
        } else {
            return MdsBuckets;
        }
    case EEntity::Settings:
        if (SettingPrefixes.empty()) {
            return Nothing();
        } else {
            return SettingPrefixes;
        }
    case EEntity::Tag:
        if (TagTypes.empty()) {
            return Nothing();
        } else {
            return TagTypes;
        }
    case EEntity::B2BOrganization:
        if (B2BOrganizations.empty()) {
            return Nothing();
        } else {
            return B2BOrganizations;
        }
    case EEntity::ChatRobot:
        if (ChatRobots.empty()) {
            return Nothing();
        } else {
            return ChatRobots;
        }
    default:
        return Instances;
    }
}

bool TAdministrativeAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    NJson::TJsonValue::TArray jsonActions;
    NJson::TJsonValue::TArray jsonEntities;
    if (!jsonValue["adm_actions"].GetArray(&jsonActions) || !jsonValue["adm_entities"].GetArray(&jsonEntities)) {
        return false;
    }
    for (auto&& i : jsonActions) {
        EAction action;
        if (TryFromString(i.GetString(), action)) {
            Actions.emplace(action);
        }
    }

    for (auto&& i : jsonEntities) {
        EEntity entity;
        if (TryFromString(i.GetString(), entity)) {
            Entities.emplace(entity);
        }
    }

    if (!Append(Handlers, jsonValue["handlers"])) {
        return false;
    }
    if (!Append(Instances, jsonValue["instances"])) {
        return false;
    }
    if (!Append(Commands, jsonValue["commands"])) {
        return false;
    }
    if (!Append(Roles, jsonValue["roles"])) {
        return false;
    }
    if (jsonValue.Has("rt_background_tags")) {
        if (!RTBackgroundTagsFilter.DeserializeFromString(jsonValue["rt_background_tags"].GetString())) {
            return false;
        }
    }
    if (jsonValue.Has("action_tags") && !ActionTagsFilter.DeserializeFromString(jsonValue["action_tags"].GetString())) {
        return false;
    }
    if (!Append(ActionTypes, jsonValue["action_types"])) {
        return false;
    }
    if (!Append(TagTypes, jsonValue["tag_types"])) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonValue, "user_ids", UserIds)) {
        return false;
    }
    if (!Append(SettingPrefixes, jsonValue["setting_prefixes"])) {
        return false;
    }
    if (!Append(Wallets, jsonValue["wallets"])) {
        return false;
    }
    if (!Append(PromoCodes, jsonValue["promo_codes"])) {
        return false;
    }
    if (!Append(MdsBuckets, jsonValue["mds_buckets"])) {
        return false;
    }
    if (!Append(ChatRobots, jsonValue["chat_robots"])) {
        return false;
    }
    const auto& limits = jsonValue["limits"];
    if (limits.IsDefined() && !limits.IsArray()) {
        return false;
    }
    for (auto&& i : limits.GetArray()) {
        if (!i.IsMap()) {
            return false;
        }
        const auto& instance = i["instance"];
        const auto& value = i["value"];
        if (!instance.IsString()) {
            return false;
        }
        if (!value.IsDouble()) {
            return false;
        }
        Limits.emplace(instance.GetString(), value.GetDouble());
    }

    return !Actions.empty() && !Entities.empty();
}

NJson::TJsonValue TAdministrativeAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::TJsonValue actionsJson(NJson::JSON_ARRAY);
    for (auto&& i : Actions) {
        actionsJson.AppendValue(ToString(i));
    }
    NJson::TJsonValue entitiesJson(NJson::JSON_ARRAY);
    for (auto&& i : Entities) {
        entitiesJson.AppendValue(ToString(i));
    }
    NJson::TJsonValue handlersJson(NJson::JSON_ARRAY);
    for (auto&& i : Handlers) {
        handlersJson.AppendValue(i);
    }
    {
        NJson::TJsonValue& objectsJson = result.InsertValue("instances", NJson::JSON_ARRAY);
        for (auto&& i : Instances) {
            objectsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& objectsJson = result.InsertValue("commands", NJson::JSON_ARRAY);
        for (auto&& i : Commands) {
            objectsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& objectsJson = result.InsertValue("roles", NJson::JSON_ARRAY);
        for (auto&& i : Roles) {
            objectsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& objectsJson = result.InsertValue("tag_types", NJson::JSON_ARRAY);
        for (auto&& i : TagTypes) {
            objectsJson.AppendValue(i);
        }
    }
    {
        if (!RTBackgroundTagsFilter.IsEmpty()) {
            result.InsertValue("rt_background_tags", RTBackgroundTagsFilter.ToString());
        }
        if (!ActionTagsFilter.IsEmpty()) {
            result.InsertValue("action_tags", ActionTagsFilter.ToString());
        }
        NJson::TJsonValue& objectsJson = result.InsertValue("action_types", NJson::JSON_ARRAY);
        for (auto&& i : ActionTypes) {
            objectsJson.AppendValue(i);
        }
    }
    result.InsertValue("user_ids", JoinSeq(", ", UserIds));
    {
        NJson::TJsonValue& objectsJson = result.InsertValue("setting_prefixes", NJson::JSON_ARRAY);
        for (auto&& i : SettingPrefixes) {
            objectsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& walletsJson = result.InsertValue("wallets", NJson::JSON_ARRAY);
        for (auto&& i : Wallets) {
            walletsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& promoCodesJson = result.InsertValue("promo_codes", NJson::JSON_ARRAY);
        for (auto&& i : PromoCodes) {
            promoCodesJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& mdsBucketsJson = result.InsertValue("mds_buckets", NJson::JSON_ARRAY);
        for (auto&& i : MdsBuckets) {
            mdsBucketsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& chatRobotsJson = result.InsertValue("chat_robots", NJson::JSON_ARRAY);
        for (auto&& i : ChatRobots) {
            chatRobotsJson.AppendValue(i);
        }
    }
    {
        NJson::TJsonValue& limits = result.InsertValue("limits", NJson::JSON_ARRAY);
        for (auto&&[instance, value] : Limits) {
            NJson::TJsonValue element;
            element["instance"] = instance;
            element["value"] = value;
            limits.AppendValue(std::move(element));
        }
    }

    result["handlers"] = std::move(handlersJson);
    result["adm_actions"] = std::move(actionsJson);
    result["adm_entities"] = std::move(entitiesJson);
    return result;
}

NDrive::TScheme TAdministrativeAction::DoGetScheme(const NDrive::IServer* server) const {
    auto roles = server->GetDriveAPI()->GetRolesManager()->GetRoles(TInstant::Zero());
    TSet<TString> rolesSet;
    for (auto&& role : roles) {
        rolesSet.emplace(role.GetName());
    }
    TSet<TString> actionTypes;
    TUserAction::TFactory::GetRegisteredKeys(actionTypes);
    actionTypes.emplace("*");

    TSet<TString> wallets;
    auto registeredAccounts = server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetRegisteredAccounts();
    for (auto&& description : registeredAccounts) {
        if (!description.GetIsPersonal()) {
            wallets.insert(description.GetName());
        }
    }

    TVector<TSetting> settings;
    server->GetSettings().GetAllSettings(settings, TInstant::Zero());
    TSet<TString> settingKeys;
    for (auto&& i : settings) {
        ui32 deep = 0;
        TString prefix;
        for (auto&& c : i.GetKey()) {
            if (c == '.') {
                settingKeys.emplace(prefix);
                if (++deep > 1) {
                    break;
                }
            }
            prefix += c;
        }
    }
    settingKeys.insert("*");

    auto promoGenerators = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IPromoCodeGenerator>(/*reportDeprecated=*/false);

    TSet<TString> chatRobots = MakeSet(NContainer::Keys(server->GetChatRobots()));

    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("setting_prefixes").SetVariants(settingKeys).SetEditable(true).SetMultiSelect(true);
    scheme.Add<TFSVariants>("adm_actions").InitVariants<EAction>().SetMultiSelect(true);
    scheme.Add<TFSVariants>("adm_entities").InitVariants<EEntity>().SetMultiSelect(true);
    scheme.Add<TFSVariants>("handlers").SetVariants(server->GetServerProcessors()).SetMultiSelect(true);
    scheme.Add<TFSVariants>("commands").InitVariants<NDrive::NVega::ECommandCode>().SetMultiSelect(true);
    scheme.Add<TFSVariants>("roles").SetVariants(rolesSet).SetMultiSelect(true);
    scheme.Add<TFSVariants>("action_types").SetVariants(actionTypes).SetMultiSelect(true);
    scheme.Add<TFSString>("action_tags", "фильтр для тегов action");
    scheme.Add<TFSString>("rt_background_tags", "фильтр для тегов rt_background");
    scheme.Add<TFSVariants>("tag_types").InitVariantsClass<TTagDescription>().SetMultiSelect(true);
    scheme.Add<TFSString>("user_ids");
    scheme.Add<TFSVariants>("wallets").SetVariants(wallets).SetMultiSelect(true);
    scheme.Add<TFSVariants>("promo_codes").SetVariants(promoGenerators).SetMultiSelect(true);
    scheme.Add<TFSVariants>("chat_robots").SetVariants(chatRobots).SetMultiSelect(true).SetEditable(true);
    scheme.Add<TFSArray>("mds_buckets").SetElement<TFSString>();
    scheme.Add<TFSArray>("instances").SetElement<TFSString>();

    NDrive::TScheme& limits = scheme.Add<TFSArray>("limits").SetElement<NDrive::TScheme>();
    limits.Add<TFSString>("instance");
    limits.Add<TFSNumeric>("value");
    return scheme;
}
