#include "evolution.h"

#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/json_processing.h>

TUserAction::TFactory::TRegistrator<TTagEvolutionAction> TTagEvolutionAction::Registrator("tag_evolution");

bool TTagEvolutionAction::CheckEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, NDrive::TEntitySession& session) const {
    if (GetTagsFilterDenied().IsEmpty() && GetTagsFilterAvailable().IsEmpty() && !GetPhotosNecessaryCount()) {
        return true;
    }
    auto taggedObject = tagsManager.RestoreObject(tag.GetObjectId(), session);
    if (!taggedObject) {
        return false;
    }

    TString errorId;
    if (!CheckEvolution(tagsManager, tag, *taggedObject, session, &errorId)) {
        session.SetErrorInfo("evolution", errorId, EDriveSessionResult::InconsistencyUser);
        return false;
    }
    return true;
}

bool TTagEvolutionAction::CheckEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, const TTaggedObject& object, NDrive::TEntitySession& session, TString* error /*= nullptr*/) const {
    if (!GetTagsFilterDenied().IsEmpty()) {
        if (GetTagsFilterDenied().IsMatching(object.GetTags(), error)) {
            if (!!error) {
                *error = "object_is_not_matched_by_evolution_filters_denied";
            }
            return false;
        }
    }
    if (!GetTagsFilterAvailable().IsEmpty()) {
        if (!GetTagsFilterAvailable().IsMatching(object.GetTags(), error)) {
            if (!!error) {
                *error = "object_is_not_matched_by_evolution_filters_available";
            }
            return false;
        }
    }
    if (GetPhotosNecessaryCount()) {
        auto optionalEvents = tagsManager.GetEventsByTag(tag.GetTagId(), session);
        if (!optionalEvents) {
            if (error) {
                *error = "cannot_get_events_by_tag";
            }
            return false;
        }
        ui32 photosCount = 0;
        for (auto&& i : *optionalEvents) {
            if (i.GetHistoryAction() == EObjectHistoryAction::AddSnapshot && i && tag->GetName() == i->GetName()) {
                ++photosCount;
            }
        }
        if (photosCount < GetPhotosNecessaryCount()) {
            if (error) {
                *error = "not_enough_photos_for_evolution";
            }
            return false;
        }
    }
    return true;
}

bool TTagEvolutionAction::PostEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!PostAction) {
        return true;
    }
    switch (*PostAction) {
        case ETagAction::DropPerform:
            return tagsManager.DropTagPerformer(tag, permissions.GetUserId(), session);
        case ETagAction::RemovePerform:
            return tagsManager.RemoveTag(tag, permissions.GetUserId(), server, session, /*force=*/true);
        case ETagAction::Remove:
            return tagsManager.RemoveTag(tag, permissions.GetUserId(), server, session, /*force=*/false);
        default:
            session.SetErrorInfo("TagEvolutionAction::PostEvolution", TStringBuilder() << "unsupported PostAction: " << PostAction);
            return false;
    }
}

bool TTagEvolutionAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    JREAD_BOOL_OPT(jsonValue, "only_performed", OnlyPerformed);
    if (!TJsonProcessor::ReadFromString("priority_policy", jsonValue, PriorityPolicy)) {
        return false;
    }
    if (!TJsonProcessor::ReadFromString("comment_policy", jsonValue, CommentPolicy)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "external_priority", ExternalPriority)) {
        return false;
    }
    if (jsonValue.Has("tags_filter_available")) {
        TString filterStr;
        if (!jsonValue["tags_filter_available"].GetString(&filterStr)) {
            return false;
        }
        if (!TagsFilterAvailable.DeserializeFromString(filterStr)) {
            return false;
        }
    }
    if (jsonValue.Has("tags_filter_denied")) {
        TString filterStr;
        if (!jsonValue["tags_filter_denied"].GetString(&filterStr)) {
            return false;
        }
        if (!TagsFilterDenied.DeserializeFromString(filterStr)) {
            return false;
        }
    }
    if (jsonValue.Has("photos_count")) {
        if (!jsonValue["photos_count"].IsUInteger()) {
            return false;
        }
        PhotosNecessaryCount = jsonValue["photos_count"].GetUInteger();
    }
    TagNameFrom = jsonValue["tag_name_from"].GetString();
    if (TagNameFrom.StartsWith("$")) {
        MatcherFrom.Compile(TagNameFrom.substr(1));
        if (!MatcherFrom.IsCompiled()) {
            return false;
        }
    }
    TagNameTo = jsonValue["tag_name_to"].GetString();
    if (TagNameTo.StartsWith("$")) {
        MatcherTo.Compile(TagNameTo.substr(1));
        if (!MatcherTo.IsCompiled()) {
            return false;
        }
    }

    if (jsonValue.Has("two_side") && !jsonValue["two_side"].GetBoolean(&TwoSideEvolution)) {
        return false;
    }

    JREAD_FROM_STRING_OPT(jsonValue, "device_operation", CommandCode);
    JREAD_FROM_STRING_OPT(jsonValue, "force_device_operation", ForceCommandCode);

    if (ForceCommandCode == NDrive::NVega::ECommandCode::UNKNOWN) {
        ForceCommandCode = CommandCode;
    }
    return true
        && NJson::ParseField(jsonValue["client_post_actions"], ClientPostActions)
        && NJson::ParseField(jsonValue["display_name"], DisplayName)
        && NJson::ParseField(jsonValue["post_action"], PostAction)
    ;
}

NJson::TJsonValue TTagEvolutionAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    TJsonProcessor::WriteAsString(result, "priority_policy", PriorityPolicy);
    TJsonProcessor::WriteAsString(result, "comment_policy", CommentPolicy);
    TJsonProcessor::Write(result, "external_priority", ExternalPriority);
    JWRITE(result, "only_performed", OnlyPerformed);
    if (!TagsFilterAvailable.IsEmpty()) {
        JWRITE(result, "tags_filter_available", TagsFilterAvailable.ToString());
    }
    if (!TagsFilterDenied.IsEmpty()) {
        JWRITE(result, "tags_filter_denied", TagsFilterDenied.ToString());
    }
    if (PhotosNecessaryCount) {
        JWRITE(result, "photos_count", PhotosNecessaryCount);
    }
    if (!ClientPostActions.empty()) {
        result["client_post_actions"] = NJson::ToJson(ClientPostActions);
    }
    if (DisplayName) {
        result["display_name"] = DisplayName;
    }
    if (PostAction) {
        result["post_action"] = NJson::ToJson(PostAction);
    }
    result["device_operation"] = ToString(CommandCode);
    result["force_device_operation"] = ToString(ForceCommandCode);
    result["tag_name_to"] = TagNameTo;
    result["tag_name_from"] = TagNameFrom;
    result["two_side"] = TwoSideEvolution;
    return result;
}

NDrive::TScheme TTagEvolutionAction::DoGetScheme(const NDrive::IServer* server) const {
    TSet<TString> clientPostActions;
    if (server) {
        auto clientPostActionsString = server->GetSettings().GetValue<TString>("administration.client_post_actions").GetOrElse("gvar:administration.client_post_actions");
        clientPostActions = StringSplitter(clientPostActionsString).SplitBySet(" ,").SkipEmpty();
    }
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("display_name");
    result.Add<TFSString>("tag_name_from");
    result.Add<TFSString>("tag_name_to");
    result.Add<TFSString>("tags_filter_available", "Фильтр объектов, для которых допустима эволюция");
    result.Add<TFSString>("tags_filter_denied", "Фильтр объектов, для которых НЕ допустима эволюция");
    result.Add<TFSNumeric>("photos_count");
    result.Add<TFSVariants>("device_operation", "Действия с машиной в нормальной ситуации").InitVariants<NDrive::NVega::ECommandCode>();
    result.Add<TFSVariants>("force_device_operation", "Действия с машиной в ситуации FORCE").InitVariants<NDrive::NVega::ECommandCode>();
    result.Add<TFSBoolean>("two_side", "Двустороннее действие").SetDefault(false);
    result.Add<TFSBoolean>("only_performed", "Только исполняемые").SetDefault(true);
    result.Add<TFSVariants>("priority_policy", "Политика трансляции приоритета").InitVariants<EPriorityPolicy>();
    result.Add<TFSNumeric>("external_priority", "Внешний приоритет");
    result.Add<TFSVariants>("client_post_actions", "Client post-evolution actions").SetMultiSelect(true).SetVariants(std::move(clientPostActions));
    result.Add<TFSVariants>("comment_policy", "Политика трансляции комментариев").InitVariants<ECommentPolicy>();
    result.Add<TFSVariants>("post_action", "Post-evolution action").SetVariants({
        ETagAction::DropPerform,
        ETagAction::RemovePerform,
        ETagAction::Remove,
    });
    return result;
}
