#pragma once

#include <drive/backend/actions/abstract/action.h>

#include <drive/backend/tags/tags.h>

#include <drive/telematics/protocol/vega.h>

#include <library/cpp/regex/pcre/regexp.h>

#include <util/generic/serialized_enum.h>

class IEntityTagsManager;

class TTagEvolutionAction: public TUserAction {
private:
    using TBase = TUserAction;

private:
    TRegExMatch MatcherFrom;
    TRegExMatch MatcherTo;

private:
    static TFactory::TRegistrator<TTagEvolutionAction> Registrator;

public:
    enum class EPriorityPolicy {
        New = 1 << 0,
        Default = 1 << 1,
        Provide = 1 << 2,
        NPD = 1 << 3,
        External = 1 << 4,
    };

    enum class ECommentPolicy {
        New = 1 << 0,
        Empty = 1 << 1,
        Provide = 1 << 2,
        NP = 1 << 3,
    };

    using ETagAction = NTagActions::ETagAction;

public:
    R_FIELD(NDrive::NVega::ECommandCode, CommandCode, NDrive::NVega::ECommandCode::UNKNOWN);
    R_FIELD(NDrive::NVega::ECommandCode, ForceCommandCode, NDrive::NVega::ECommandCode::UNKNOWN);
    R_FIELD(TString, DisplayName);
    R_FIELD(TString, TagNameFrom);
    R_FIELD(TString, TagNameTo);
    R_FIELD(bool, TwoSideEvolution, false);
    R_FIELD(bool, OnlyPerformed, true);
    R_FIELD(TTagsFilter, TagsFilterAvailable);
    R_FIELD(TTagsFilter, TagsFilterDenied);
    R_FIELD(ui32, PhotosNecessaryCount, 0);

    R_FIELD(TVector<TString>, ClientPostActions);
    R_FIELD(ECommentPolicy, CommentPolicy, ECommentPolicy::NP);
    R_FIELD(EPriorityPolicy, PriorityPolicy, EPriorityPolicy::Provide);
    R_OPTIONAL(i32, ExternalPriority);
    R_OPTIONAL(ETagAction, PostAction);

public:
    bool CheckEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, NDrive::TEntitySession& session) const;
    bool CheckEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, const TTaggedObject& object, NDrive::TEntitySession& session, TString* error = nullptr) const;
    bool PostEvolution(const IEntityTagsManager& tagsManager, const TDBTag& tag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

    NJson::TJsonValue GetPublicJsonReport() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        result.InsertValue("from", TagNameFrom);
        result.InsertValue("to", TagNameTo);
        if (!TagsFilterAvailable.IsEmpty()) {
            result.InsertValue("available_filter", TagsFilterAvailable.ToString());
        }
        if (!TagsFilterDenied.IsEmpty()) {
            result.InsertValue("denied_filter", TagsFilterDenied.ToString());
        }
        if (PhotosNecessaryCount) {
            result.InsertValue("photos_count", PhotosNecessaryCount);
        }
        return result;
    }

    TTagEvolutionAction BuildReversed() const {
        TTagEvolutionAction result = *this;
        std::swap(result.MatcherFrom, result.MatcherTo);
        std::swap(result.TagNameFrom, result.TagNameTo);
        return result;
    }

    using TBase::TBase;

    NDrive::NVega::ECommandCode GetDeviceOperation(const EEvolutionMode eMode) const {
        switch (eMode) {
            case EEvolutionMode::Default:
                return CommandCode;
            case EEvolutionMode::Force:
            case EEvolutionMode::IgnoreTelematic:
                return ForceCommandCode;
        }
    }

    bool HasDeviceOperation(const EEvolutionMode eMode) const {
        switch (eMode) {
            case EEvolutionMode::Default:
                return CommandCode != NDrive::NVega::ECommandCode::UNKNOWN;
            case EEvolutionMode::Force:
                return ForceCommandCode != NDrive::NVega::ECommandCode::UNKNOWN;
            case EEvolutionMode::IgnoreTelematic:
                return false;
        }
    }

    bool MatchFrom(const TTagDescription& tagDescription) const {
        if (TagNameFrom.StartsWith("$")) {
            return MatcherFrom.Match(tagDescription.GetName().data());
        } else {
            return tagDescription.GetName() == TagNameFrom;
        }
    }

    bool MatchTo(const TTagDescription& tagDescription) const {
        if (TagNameTo.StartsWith("$")) {
            return MatcherTo.Match(tagDescription.GetName().data());
        } else {
            return tagDescription.GetName() == TagNameTo;
        }
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "tag_evolution";
    }
    virtual TString GetType() const override {
        return GetTypeName();
    }
};
